/*
 * ROX-Filer, filer for the ROX desktop project
 * Copyright (C) 2006, Thomas Leonard and others (see changelog for details).
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA  02111-1307  USA
 */

/* diritem.c - get details about files */

/* Don't load icons larger than 400K (this is rather excessive, basically
 * we just want to stop people crashing the filer with huge icons).
 */
#define MAX_ICON_SIZE (400 * 1024)

#include "config.h"

#include <gtk/gtk.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include "global.h"

#include "diritem.h"
#include "support.h"
#include "gui_support.h"
#include "mount.h"
#include "type.h"
#include "usericons.h"
#include "options.h"
#include "fscache.h"
#include "pixmaps.h"
#include "xtypes.h"

#define RECENT_DELAY (5 * 60)	/* Time in seconds to consider a file recent */
#define ABOUT_NOW(time) (diritem_recent_time - time < RECENT_DELAY)
/* If you want to make use of the RECENT flag, make sure this is set to
 * the current time before calling diritem_restat().
 */
time_t diritem_recent_time;
static GMutex m_diritems;
static GSList *mfree = NULL; //free on main loop
static GSList *munref = NULL; //unref on main loop
static guint onmainidle = 0;

static gboolean onmaincb(void *notused)
{
	g_mutex_lock(&m_diritems);

	if (mfree)
		g_slist_free_full(mfree, g_free);
	mfree = NULL;

	if (munref)
		g_slist_free_full(munref, g_object_unref);
	munref = NULL;

	onmainidle = 0;
	g_mutex_unlock(&m_diritems);
	return FALSE;
}

/****************************************************************
 *			EXTERNAL INTERFACE			*
 ****************************************************************/

void diritem_init(void)
{
	read_globicons();
}

/* Bring this item's structure uptodate.
 * 'parent' is optional; it saves one stat() for directories.
 */
void diritem_restat(
		const guchar *path,
		DirItem *retitem,
		struct stat *parent,
		gboolean examine_now)
{
	struct stat	info;

	g_mutex_lock(&m_diritems);
	DirItem newitem = *retitem;
	g_mutex_unlock(&m_diritems);

	DirItem *item = &newitem;

	item->_image = NULL;
	item->flags &=
		(ITEM_FLAG_CAPS | ITEM_FLAG_IN_RESCAN_QUEUE | ITEM_FLAG_IN_EXAMINE);
	item->mime_type = NULL;

	if (mc_lstat(path, &info) == -1)
	{
		item->lstat_errno = errno;
		item->base_type = TYPE_ERROR;
		item->size = 0;
		item->mode = 0;
		item->mtime = item->ctime = item->atime = 0;
		item->uid = (uid_t) -1;
		item->gid = (gid_t) -1;
	}
	else
	{
		guchar *target_path;

		item->lstat_errno = 0;
		item->size = info.st_size;
		item->mode = info.st_mode;
		item->atime = info.st_atime;
		item->ctime = info.st_ctime;
		item->mtime = info.st_mtime;
		item->uid = info.st_uid;
		item->gid = info.st_gid;
		if (ABOUT_NOW(item->mtime) || ABOUT_NOW(item->ctime))
			item->flags |= ITEM_FLAG_RECENT;

		if (xattr_have(path))
			item->flags |= ITEM_FLAG_HAS_XATTR;

		if (item->label)
		{
			g_mutex_lock(&m_diritems);
			if (retitem->label)
				mfree = g_slist_prepend(mfree, item->label);
			retitem->label = NULL;
			g_mutex_unlock(&m_diritems);
		}
		item->label = xlabel_get(path);

		if (S_ISLNK(info.st_mode))
		{
			if (mc_stat(path, &info))
				item->base_type = TYPE_ERROR;
			else
				item->base_type =
					mode_to_base_type(info.st_mode);

			item->flags |= ITEM_FLAG_SYMLINK;

			target_path = pathdup(path);
		}
		else
		{
			item->base_type = mode_to_base_type(info.st_mode);
			target_path = (guchar *) path;
		}

		if (item->base_type == TYPE_DIRECTORY)
		{
			item->size = 0;
			if (mount_is_mounted(target_path, &info,
					target_path == path ? parent : NULL))
				item->flags |= ITEM_FLAG_MOUNT_POINT
						| ITEM_FLAG_MOUNTED;
			else if (g_hash_table_lookup(fstab_mounts,
							target_path))
				item->flags |= ITEM_FLAG_MOUNT_POINT;
		}

		if (path != target_path)
			g_free(target_path);
	}

	if (item->base_type == TYPE_DIRECTORY)
	{
		/* KRJW: info.st_uid will be the uid of the dir, regardless
		 * of whether `path' is a dir or a symlink to one.  Note that
		 * if path is a symlink to a dir, item->uid will be the uid
		 * of the *symlink*, but we really want the uid of the dir
		 * to which the symlink points.
		 */
		check_globicon(path, item);

		if (item->flags & ITEM_FLAG_MOUNT_POINT)
		{
			item->mime_type = inode_mountpoint;
			/* Try to avoid automounter problems */

			if (item->flags & ITEM_FLAG_MOUNTED)
				item->flags |= ITEM_FLAG_NEED_EXAMINE;
		}
		else if (info.st_mode & S_IWOTH)
		{/* Don't trust world-writable dirs */}
		else
			item->flags |= ITEM_FLAG_NEED_EXAMINE;

	}
	else if (item->base_type == TYPE_FILE)
	{
		if (item->flags & ITEM_FLAG_SYMLINK)
		{
			guchar *link_path;
			link_path = pathdup(path);
			item->mime_type = type_from_path(link_path
					? link_path
					: path);
			g_free(link_path);
		}
		else
			item->mime_type = type_from_path(path);

		/* Note: for symlinks we need the mode of the target */
		if (info.st_mode & (S_IXUSR | S_IXGRP | S_IXOTH))
		{
			/* Note that the flag is set for ALL executable
			 * files, but the mime_type must also be executable
			 * for clicking on the file to run it.
			 */
			item->flags |= ITEM_FLAG_EXEC_FILE;

			if (item->mime_type == NULL ||
			    item->mime_type == application_octet_stream)
			{
				item->mime_type = application_executable;
			}
			else if (item->mime_type == text_plain &&
			         !strchr(item->leafname, '.'))
			{
				item->mime_type = application_x_shellscript;
			}
		}
		else if (item->mime_type == application_x_desktop)
		{
			item->flags |= ITEM_FLAG_EXEC_FILE;
		}

		if (!item->mime_type)
			item->mime_type = text_plain;

		check_globicon(path, item);

		if (item->mime_type == application_x_desktop && item->_image == NULL)
		{
			g_mutex_lock(&m_diritems);
			item->_image = pixmap_from_desktop_file(path);
			g_mutex_unlock(&m_diritems);
		}
	}
	else
		check_globicon(path, item);

	if (!item->mime_type)
		item->mime_type = mime_type_from_base_type(item->base_type);

	g_mutex_lock(&m_diritems);
	if (retitem->_image)
		munref = g_slist_prepend(munref, retitem->_image);
	*retitem = newitem;
	g_mutex_unlock(&m_diritems);

	if (examine_now && item->flags & ITEM_FLAG_NEED_EXAMINE)
		diritem_examine_dir(path, retitem);
}

DirItem *diritem_new(const guchar *leafname)
{
	DirItem		*item;

	item = g_new0(DirItem, 1);
	item->leafname = g_strdup(leafname);
	item->base_type = TYPE_UNKNOWN;

	//collate key
	gchar *to_free = NULL;
	if (!g_utf8_validate(leafname, -1, NULL))
		leafname = to_free = to_utf8(leafname);

	if (g_unichar_isupper(g_utf8_get_char(leafname)))
		item->flags |= ITEM_FLAG_CAPS;

	gchar *tmp = g_utf8_strdown(leafname, -1);
	item->collatekey = g_utf8_collate_key_for_filename(tmp, -1);
	g_free(tmp);

	if (to_free)
		g_free(to_free);	/* Only taken for invalid UTF-8 */

	return item;
}

void diritem_free(DirItem *item)
{
	g_return_if_fail(item != NULL);

	if (item->_image)
		g_object_unref(item->_image);
	if (item->label)
		g_free(item->label);

	g_free(item->collatekey);
	g_free(item->leafname);
	g_free(item);
}

/* For use by di_image() only. Sets item->_image. */
MaskedPixmap *_diritem_get_image(DirItem *item, gboolean mainthread)
{
	MaskedPixmap *ret;

	g_mutex_lock(&m_diritems);
	if (!item->_image && item->base_type != TYPE_UNKNOWN)
	{
		if (item->base_type == TYPE_ERROR)
			item->_image = g_object_ref(im_error);
		else
			item->_image = type_to_icon(item->mime_type);
	}

	ret = item->_image;

	if (mainthread && (mfree || munref) && !onmainidle)
		onmainidle = g_idle_add(onmaincb, NULL);

	g_mutex_unlock(&m_diritems);
	return ret;
}

/* Fill in more details of the DirItem for a directory item.
 * - Looks for an image (but maybe still NULL on error)
 * - Updates ITEM_FLAG_APPDIR
 */
gboolean diritem_examine_dir(const guchar *path, DirItem *item)
{
	guchar *rpath = pathdup(path); //realpath

	int oldsize = item->size;
	DIR *d = mc_opendir(rpath);
	if (d)
	{
		int cnt = 0;
		while ((mc_readdir(d))) cnt++;
		item->size = cnt - 2; //. and ..
		mc_closedir(d);
	}
	g_free(rpath);

	g_mutex_lock(&m_diritems);
	item->flags &= ~ITEM_FLAG_NEED_EXAMINE;
	g_mutex_unlock(&m_diritems);

	return item->size != oldsize;
}
