// SPDX-License-Identifier: GPL-2.0-or-later
/*
 * File operations (independent of GUI)
 *
 * Copyright (C) 2018, 2019 Tavmjong Bah
 *
 * The contents of this file may be used under the GNU General Public License Version 2 or later.
 *
 */

#include <iostream>
#include <gtkmm.h>

#include "file.h"

#include "document.h"
#include "document-undo.h"

#include "extension/system.h"     // Extension::open()
#include "extension/extension.h"
#include "extension/db.h"
#include "extension/output.h"
#include "extension/input.h"

#include "object/sp-root.h"

/**
 * Create a blank document, remove any template data.
 * Input: Empty string or template file name.
 */
SPDocument*
ink_file_new(const std::string &Template)
{
    SPDocument *doc = SPDocument::createNewDoc ((Template.empty() ? nullptr : Template.c_str()), true, true );

    if (doc) {
        // Remove all the template info from xml tree
        Inkscape::XML::Node *myRoot = doc->getReprRoot();
        Inkscape::XML::Node *nodeToRemove = sp_repr_lookup_name(myRoot, "inkscape:_templateinfo");
        if (nodeToRemove != nullptr) {
            Inkscape::DocumentUndo::ScopedInsensitive no_undo(doc);
            sp_repr_unparent(nodeToRemove);
            delete nodeToRemove;
        }
    } else {
        std::cout << "ink_file_new: Did not create new document!" << std::endl;
    }

    return doc;
}


SPDocument*
ink_file_open(const Glib::RefPtr<Gio::File>& file, bool &cancelled)
{
    cancelled = false;

    SPDocument *doc = nullptr;

    std::string path = file->get_path();

    try {
        doc = Inkscape::Extension::open(nullptr, path.c_str());
    } catch (Inkscape::Extension::Input::no_extension_found &e) {
        doc = nullptr;
    } catch (Inkscape::Extension::Input::open_failed &e) {
        doc = nullptr;
    } catch (Inkscape::Extension::Input::open_cancelled &e) {
        cancelled = true;
        doc = nullptr;
    }

    // Try to open explicitly as SVG.
    if (doc == nullptr && !cancelled) {
        try {
            doc = Inkscape::Extension::open(Inkscape::Extension::db.get(SP_MODULE_KEY_INPUT_SVG), path.c_str());
        } catch (Inkscape::Extension::Input::no_extension_found &e) {
            doc = nullptr;
        } catch (Inkscape::Extension::Input::open_failed &e) {
            doc = nullptr;
        } catch (Inkscape::Extension::Input::open_cancelled &e) {
            cancelled = true;
            doc = nullptr;
        }
    }

    if (doc == nullptr) {
        std::cerr << "ink_file_open: '" << path << "' cannot be opened!" << std::endl;
    } else {

        // This is the only place original values should be set.
        SPRoot *root = doc->getRoot();
        root->original.inkscape = root->version.inkscape;
        root->original.svg      = root->version.svg;
    }

    return doc;
}

/*
  Local Variables:
  mode:c++
  c-file-style:"stroustrup"
  c-file-offsets:((innamespace . 0)(inline-open . 0)(case-label . +))
  indent-tabs-mode:nil
  fill-column:99
  End:
*/
// vim: filetype=cpp:expandtab:shiftwidth=4:tabstop=8:softtabstop=4 :
