#!/bin/bash
# $Id: massconvert32.sh,v 1.38 2013/02/22 11:40:24 eha Exp eha $
#
# Written 2009, 2010, 2011, 2012, 2013  Eric Hameleers, Eindhoven, NL
#
# Convert 32bit slackware packages 'en masse' into compatibility packages
# to be installed on slackware64 - providing the 32bit part of multilib.
# The string "-compat32" is added at the end of package name when a
# compatibility package gets created.  This allows it to be installed
# on slackware64 alongside the native 64bit versions.
# For example: the original 32bit package "bzip2" will be converted to a new
#              package with the name "bzip2-compat32"
#
# You also have to install multilib versions of glibc and gcc !

# Before we start
[ -x /bin/id ] && CMD_ID="/bin/id" || CMD_ID="/usr/bin/id"
if [ "$($CMD_ID -u)" != "0" ]; then
  echo "You need to be root to run $(basename $0)."
  exit 1
fi

# Should we be verbose?
VERBOSE=${VERBOSE:-1}

# In case we need temporary storage:
TMP=${TMP:-/tmp}
mkdir -p $TMP
if [ ! -w "$TMP" ]; then
  echo "Can not write to temporary directory '$TMP'!"
  exit 3
fi

# Zero some other variables:
SLACK32ROOT=""
SLACK32URL=""
TARGET64ROOT=""

# Helpful instructions in case the user asks for it:
function show_help () {
  # Write the help text to output:
  cat <<EOF

Usage: $0 <-i 32bit_package_tree|-u 32bit_package_url>  [-d output_directory]

$(basename $0) converts an essential subset of 32-bit Slackware
packages into 'compatibility' packages for 64-bit Slackware.

Required parameter - one of these two::
  -i 32bit_package_tree        A 32bit Slackware package-tree. It should have
                               the a,ap,d,..,y directories immediately below.
  -u 32bit_package_url         The URL of a http or ftp server containing 32bit
                               Slackware packages. It should have the
                               a,ap,d,..,y directories immediately below.
Optional parameter::
  -d destination_directory     create packages in this directory.
                               By default, the new packages will be created
                               in your current directory.

Example of a useable Slackware URL:
  http://slackware.mirrors.tds.net/pub/slackware/slackware-13.0/slackware

EOF
}

# Parse the commandline parameters:
while [ ! -z "$1" ]; do
  case $1 in
    -d|--destdir)
      TARGET64ROOT="$(cd $(dirname "${2}"); pwd)/$(basename "${2}")"
      shift 2
      ;;
    -h|--help)
      show_help
      exit 0
      ;;
    -i|--inputdir)
      SLACK32ROOT="$(cd $(dirname "${2}"); pwd)/$(basename "${2}")"
      shift 2
      ;;
    -u|--inputurl)
      SLACK32URL="${2}"
      shift 2
      ;;
    -*)
      echo "Unsupported parameter '$1'!"
      exit 1
      ;;
    *)
      # Do nothing
      shift
      ;;
  esac
done

# A function to determine if there are spaces in the pathname:
function contains_spaces () {
  local CHOPPED=$(echo "$1" | tr -d ' ')
  [ "x$CHOPPED" = "x$1" ] && return 1 || return 0
}

# A function to download a package:
function download_pkg () {
  local IP="$1"
  local BP="$(basename $1)"
  local DP="$(dirname $1)"

  local PS="$(basename $DP)"

  local DURL="$2"

  mkdir -p "$DP"
  cd "$DP"
  echo "--- Downloading '$PS/$BP'..."
  lftp -c 'open '${DURL}' ; mget '${PS}/${BP}'-*.t?z'
}

# A function to retrieve the fullname of a package:
function get_pkgfullpath () {
  local IP="$1"
  local BP="$(basename $1)"
  local FP=""

  for FP in $(ls ${IP}-*.t?z) ; do
    if [ "$(echo "$FP"|rev|cut -d- -f4-|cut -d/ -f1| rev)" = "$BP" ];
    then
      echo "$FP"
      continue
    fi
  done
}

# A function to cconvert a package,
# downloading it first if needed, and taking patches into account:
function conv_pkg () {
  local BP="$1"
  local SERIES="$2"
  local PATCHLIST="$3"
 
  # Does the package we want have a patch available?
  echo "$PATCHLIST" | tr - _ | grep -wq $(echo $BP |tr - _)
  if [ $? -eq 0 ]; then
    echo "--- Using Slackware's patch package for $BP"
    PKGPATH="../patches/packages/$BP"
    SLACK32URLFIX="$(dirname $SLACK32URL)/patches"
  else
    PKGPATH="$SERIES/$BP"
    SLACK32URLFIX="$SLACK32URL"
  fi

  [ -n "$SLACK32URL" ] && download_pkg $SLACK32ROOT/$PKGPATH $SLACK32URLFIX
  FULLPKG=$(get_pkgfullpath $SLACK32ROOT/$PKGPATH)
  if [ -z "$FULLPKG" ]; then
    echo "*** FAIL: package '$BP' was not found!"
    continue
  fi
  [ "$VERBOSE" = "1" ] && echo "--- $BP"
  sh $CONV32 -i $FULLPKG -d $TARGET64ROOT/${SERIES}-compat32
}

# Safety checks in case a URL was provided: 
if [ -n "$SLACK32URL" ]; then
  if [ -n "$SLACK32ROOT" ]; then
    echo "*** Options '-i' and '-u' can not be used together!"
    exit 1
  else
    # Define a 'temporary' root directory where we will download packages:
    SLACK32ROOT="${TMP}/alienBOB/slackware"
    if ! which lftp 1>/dev/null 2>&1 ; then
      echo  "No lftp binary detected! Need lftp for package downloading!"
      exit 1
    fi
  fi
fi

# The root directory of 32bit Slackware packages
# (should have the a,ap,d,..,y directories immediately below):
# Let's use a fallback directory in case none was specified:
SLACK32ROOT="${SLACK32ROOT:-"/home/ftp/pub/Linux/Slackware/slackware-current/slackware"}"

# The output directory for our converted packages; defaults to the current dir.
# Note that {a,ap,d,l,n,x}-compat32 directories will be created below this
# directory if they do not yet exist:
TARGET64ROOT="${TARGET64ROOT:-"$(pwd)"}"

# Abort if we got directories with spaces in them:
if contains_spaces "$SLACK32ROOT" ; then
  echo "Directories with spaces are unsupported: '$SLACK32ROOT'!"
  exit 1
fi
if contains_spaces "$TARGET64ROOT" ; then
  echo "Directories with spaces are unsupported: '$TARGET64ROOT'!"
  exit 1
fi

# Where the scripts are:
SRCDIR=$(cd $(dirname $0); pwd)

# The script that does the package conversion:
CONV32=$SRCDIR/convertpkg-compat32

# Bail out if the conversion script is not available/working:
if [ ! -f $CONV32 ]; then
  echo "Required script '$CONV32' is not present or not executable! Aborting..."
  exit 1
fi

# We can not proceed if there are no packages and we did not get an URL:
if [ -z "$SLACK32URL" ]; then
  if [ ! -d $SLACK32ROOT/a -o ! -d $SLACK32ROOT/ap -o ! -d $SLACK32ROOT/d -o ! -d $SLACK32ROOT/l -o ! -d $SLACK32ROOT/n -o ! -d $SLACK32ROOT/x -o ! -d $SLACK32ROOT/xap ]; then
    echo "Required package directories a,ap,d,l,n,x,xap below '$SLACK32ROOT' are not found! Aborting..."
    exit 1
  fi
fi

# If a destination_directory was specified, abort now if we can not create it:
if [ -n "$TARGET64ROOT" -a ! -d "$TARGET64ROOT" ]; then
  echo "Creating output directory '$TARGET64ROOT'..."
  mkdir -p $TARGET64ROOT
  if [ ! -w "$TARGET64ROOT" ]; then
    echo "Creation of output directory '$TARGET64ROOT' failed!"
    exit 3
  fi
fi

# Get a list of available patches
if [ -n "$SLACK32URL" ]; then
  PATCH_LIST=$(echo $(lftp -c "open $(dirname $SLACK32URL)/patches/packages ; cls *.t?z" | rev | cut -f4- -d- |rev))
else
  PATCH_LIST=$(echo $(cd $(dirname $SLACK32ROOT)/patches/packages ; ls -1 *.t?z | rev | cut -f4- -d- |rev))
fi

# This is the script's internal list of what I consider as the essential
# 32bit multilib package set for your Slackware64:

# The A/ series:
A_COMPAT32="
aaa_elflibs
bzip2
cups
cxxlibs
dbus
e2fsprogs
openssl-solibs
util-linux
"

# The AP/ series:
AP_COMPAT32="
flac
mpg123
mysql
sqlite
"

# The D/ series:
D_COMPAT32="
libtool
"

# The L/ series:
L_COMPAT32="
alsa-lib
alsa-oss
atk
audiofile
cairo
dbus-glib
esound
expat
freetype
gamin
gdk-pixbuf2
glib2
gmp
gtk+2
gst-plugins-base
gst-plugins-good
gstreamer
hal
jasper
lcms
lesstif
libart_lgpl
libelf
libexif
libffi
libglade
libgphoto2
libidn
libieee1284
libjpeg
libmng
libmpc
libogg
libpcap
libpng
libsamplerate
libsndfile
libtermcap
libtiff
libusb
libvorbis
libxml2
libxslt
ncurses
pango
popt
qt
readline
sdl
seamonkey-solibs
startup-notification
svgalib
v4l-utils
zlib
"

# The N/ series:
N_COMPAT32="
curl
cyrus-sasl
gnutls
libgcrypt
libgpg-error
nettle
openldap-client
openssl
p11-kit
"

# The X/ series:
X_COMPAT32="
fontconfig
glew
glu
libFS
libICE
libSM
libX11
libXScrnSaver
libXTrap
libXau
libXaw
libXcomposite
libXcursor
libXdamage
libXdmcp
libXevie
libXext
libXfixes
libXfont
libXfontcache
libXft
libXi
libXinerama
libXmu
libXp
libXpm
libXprintUtil
libXrandr
libXrender
libXres
libXt
libXtst
libXv
libXvMC
libXxf86dga
libXxf86misc
libXxf86vm
libdmx
libdrm
libfontenc
libpciaccess
libxcb
mesa
pixman
xcb-util
"

# The XAP/ series:
XAP_COMPAT32="
sane
"

# Create target directories if they do not exist:
for TDIR in a-compat32 ap-compat32 d-compat32 l-compat32 n-compat32 x-compat32 xap-compat32 ; do
  mkdir -p $TARGET64ROOT/$TDIR
  if [ ! -w $TARGET64ROOT/$TDIR ]; then
    echo "Directory '$TARGET64ROOT/$TDIR' is not writable! Aborting..."
    exit 1
  fi
done

# Convert the 32bit packages from A AP D L N and X series, checking for patches:
echo "***"
echo "*** Starting the conversion process:"

echo "*** 'A' series:"
for INPKG in $A_COMPAT32 ; do
  conv_pkg $INPKG a "$PATCH_LIST"
done

echo "*** 'AP' series:"
for INPKG in $AP_COMPAT32 ; do
  conv_pkg $INPKG ap "$PATCH_LIST"
done

echo "*** 'D' series:"
for INPKG in $D_COMPAT32 ; do
  conv_pkg $INPKG d "$PATCH_LIST"
done

echo "*** 'L' series:"
for INPKG in $L_COMPAT32 ; do
  conv_pkg $INPKG l "$PATCH_LIST"
done

echo "*** 'N' series:"
for INPKG in $N_COMPAT32 ; do
  conv_pkg $INPKG n "$PATCH_LIST"
done

echo "*** 'X' series:"
for INPKG in $X_COMPAT32 ; do
  conv_pkg $INPKG x "$PATCH_LIST"
done

echo "*** 'XAP' series:"
for INPKG in $XAP_COMPAT32 ; do
  conv_pkg $INPKG xap "$PATCH_LIST"
done


# Mention downloaded packages if we used a URL as source:
if [ -n "$SLACK32URL" ]; then
  echo "WARNING: packages which were downloaded from '$SLACK32URL'"
  echo "have been left in directory '$SLACK32ROOT'."
  echo "It is safe to remove these now."
  echo ""
fi

echo "*** Conversion done!"
echo "***"
