/*
 * SPDX-License-Identifier: GPL-3.0-only
 * MuseScore-CLA-applies
 *
 * MuseScore
 * Music Composition & Notation
 *
 * Copyright (C) 2021 MuseScore BVBA and others
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

#ifndef MU_ENGRAVING_CHORDLIST_H
#define MU_ENGRAVING_CHORDLIST_H

#include <map>

#include "global/allocator.h"
#include "types/string.h"
#include "containers.h"
#include "io/iodevice.h"

#include "modularity/ioc.h"
#include "iengravingconfiguration.h"

namespace mu::engraving::compat {
class ReadChordListHook;
}

namespace mu::engraving {
class ChordList;
class MStyle;
class XmlWriter;
class XmlReader;

//---------------------------------------------------------
//   class HDegree
//---------------------------------------------------------

enum class HDegreeType : char {
    UNDEF, ADD, ALTER, SUBTRACT
};

class HDegree
{
public:
    HDegree() = default;
    HDegree(int v, int a, HDegreeType t) { m_value = v; m_alter = a; m_type = t; }
    int value() const { return m_value; }
    int alter() const { return m_alter; }
    HDegreeType type() const { return m_type; }
    String text() const;

private:
    int m_value = 0;
    int m_alter = 0;         // -1, 0, 1  (b - - #)
    HDegreeType m_type = HDegreeType::UNDEF;
};

//---------------------------------------------------------
//   HChord
//---------------------------------------------------------

class HChord
{
public:
    HChord() = default;
    HChord(int k) { m_keys = k; }
    HChord(int a, int b, int c=-1, int d=-1, int e=-1, int f=-1, int g=-1, int h=-1, int i=-1, int k=-1, int l=-1);
    HChord(const String&);

    void rotate(int semiTones);

    bool contains(int key) const           // key in chord?
    {
        return (1 << (key % 12)) & m_keys;
    }

    HChord& operator+=(int key)
    {
        m_keys |= (1 << (key % 12));
        return *this;
    }

    HChord& operator-=(int key)
    {
        m_keys &= ~(1 << (key % 12));
        return *this;
    }

    bool operator==(const HChord& o) const { return m_keys == o.m_keys; }
    bool operator!=(const HChord& o) const { return m_keys != o.m_keys; }

    int getKeys() const { return m_keys; }
    void print() const;

    String name(int tpc) const;
    String voicing() const;
    void add(const std::vector<HDegree>& degreeList);

protected:
    int m_keys = 0;

private:
    String m_str;
};

//---------------------------------------------------------
//   RenderAction
//---------------------------------------------------------

struct RenderAction {
    enum class RenderActionType : char {
        SET, MOVE, PUSH, POP,
        NOTE, ACCIDENTAL
    };

    RenderActionType type = RenderActionType::SET;
    double movex = 0.0, movey = 0.0; // MOVE
    String text;                    // SET

    RenderAction() {}
    RenderAction(RenderActionType t)
        : type(t) {}
    void print() const;
};

//---------------------------------------------------------
//   ChordToken
//---------------------------------------------------------

enum class ChordTokenClass : char {
    ALL, QUALITY, EXTENSION, MODIFIER, ALTERATION, ADJUST, MODE, SUSPENSION, ADDITION, SUBTRACTION
};

class ChordToken
{
public:
    ChordTokenClass tokenClass;
    StringList names;
    std::list<RenderAction> renderList;
    void read(XmlReader&);
    void write(XmlWriter&) const;
};

//---------------------------------------------------------
//   ParsedChord
//---------------------------------------------------------

class ParsedChord
{
public:
    ParsedChord() = default;

    bool parse(const String&, const ChordList*, bool syntaxOnly = false, bool preferMinor = false);
    String fromXml(const String&, const String&, const String&, const String&, const std::list<HDegree>&, const ChordList*);
    const std::list<RenderAction>& renderList(const ChordList*);
    bool parseable() const { return m_parseable; }
    bool understandable() const { return m_understandable; }
    const String& name() const { return m_name; }
    const String& quality() const { return m_quality; }
    const String& extension() const { return m_extension; }
    const String& modifiers() const { return m_modifiers; }
    const StringList& modifierList() const { return m_modifierList; }
    const String& xmlKind() const { return m_xmlKind; }
    const String& xmlText() const { return m_xmlText; }
    const String& xmlSymbols() const { return m_xmlSymbols; }
    const String& xmlParens() const { return m_xmlParens; }
    const StringList& xmlDegrees() const { return m_xmlDegrees; }
    int keys() const { return m_chord.getKeys(); }
    const String& handle() const { return m_handle; }
    operator String() const {
        return m_handle;
    }
    bool operator==(const ParsedChord& c) const { return this->m_handle == c.m_handle; }
    bool operator!=(const ParsedChord& c) const { return !(*this == c); }

private:
    void configure(const ChordList*);
    void correctXmlText(const String& s = String());
    void addToken(String, ChordTokenClass);

    String m_name;
    String m_handle;
    String m_quality;
    String m_extension;
    String m_modifiers;
    StringList m_modifierList;
    std::list<ChordToken> m_tokenList;
    std::list<RenderAction> m_renderList;
    String m_xmlKind;
    String m_xmlText;
    String m_xmlSymbols;
    String m_xmlParens;
    StringList m_xmlDegrees;
    StringList m_major, m_minor, m_diminished, m_augmented, m_lower, m_raise, m_mod1, m_mod2, m_symbols;
    HChord m_chord;
    bool m_parseable = false;
    bool m_understandable = false;
};

//---------------------------------------------------------
//   ChordDescription
//---------------------------------------------------------

struct ChordDescription {
    int id = 0;               // Chord id number (Band In A Box Chord Number)
    StringList names;        // list of alternative chord names
                             // that will by recognized from keyboard entry (without root/base)
    std::list<ParsedChord> parsedChords;
    // parsed forms of primary name (optionally also include parsed forms of other names)
    String xmlKind;          // MusicXml: kind
    String xmlText;          // MusicXml: kind text=
    String xmlSymbols;       // MusicXml: kind use-symbols=
    String xmlParens;        // MusicXml: kind parentheses-degrees=
    StringList xmlDegrees;   // MusicXml: list of degrees (if any)
    HChord chord;             // C based chord
    std::list<RenderAction> renderList;
    bool generated = false;
    bool renderListGenerated = false;
    bool exportOk = false;

    ChordDescription() {}
    ChordDescription(int);
    ChordDescription(const String&);
    String quality() const { return m_quality; }
    void complete(ParsedChord* pc, const ChordList*);
    void read(XmlReader&);
    void write(XmlWriter&) const;

private:
    String m_quality;
};

//---------------------------------------------------------
//   ChordSymbol
//---------------------------------------------------------

struct ChordSymbol {
    int fontIdx = -1;
    String name;
    String value;
    Char code;

    bool isValid() const { return fontIdx != -1; }
};

//---------------------------------------------------------
//   ChordFont
//---------------------------------------------------------

struct ChordFont {
    String family;
    String fontClass;
    double mag = 1.0;
};

//---------------------------------------------------------
//   ChordList
//---------------------------------------------------------

class ChordList : public std::map<int, ChordDescription>
{
    OBJECT_ALLOCATOR(engraving, ChordList)

    INJECT(IEngravingConfiguration, configuration)

public:
    std::list<ChordFont> fonts;
    std::list<RenderAction> renderListRoot;
    std::list<RenderAction> renderListFunction;
    std::list<RenderAction> renderListBase;
    std::list<ChordToken> chordTokenList;
    static int privateID;

    bool autoAdjust() const { return m_autoAdjust; }
    double nominalMag() const { return m_nmag; }
    double nominalAdjust() const { return m_nadjust; }
    void configureAutoAdjust(double emag = 1.0, double eadjust = 0.0, double mmag = 1.0, double madjust = 0.0);
    double position(const StringList& names, ChordTokenClass ctc) const;

    bool read(const String&);
    bool read(io::IODevice* device);
    bool write(const String&) const;
    bool write(io::IODevice* device) const;
    bool loaded() const;
    void unload();

    const ChordDescription* description(int id) const;
    ChordSymbol symbol(const String& s) const { return mu::value(m_symbols, s); }

    void setCustomChordList(bool t) { m_customChordList = t; }
    bool customChordList() const { return m_customChordList; }

    void checkChordList(const MStyle& style);

private:

    friend class compat::ReadChordListHook;

    void read(XmlReader&);
    void write(XmlWriter& xml) const;

    std::map<String, ChordSymbol> m_symbols;
    bool m_autoAdjust = false;
    double m_nmag = 1.0, m_nadjust = 0.0;
    double m_emag = 1.0, m_eadjust = 0.0;
    double m_mmag = 1.0, m_madjust = 0.0;

    bool m_customChordList = false; // if true, chordlist will be saved as part of score
};
} // namespace mu::engraving
#endif
