/*
 * SPDX-License-Identifier: GPL-3.0-only
 * MuseScore-CLA-applies
 *
 * MuseScore
 * Music Composition & Notation
 *
 * Copyright (C) 2021 MuseScore BVBA and others
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

#include <gtest/gtest.h>

#include "engraving/engravingerrors.h"
#include "engraving/dom/masterscore.h"

#include "engraving/tests/utils/scorecomp.h"
#include "engraving/tests/utils/scorerw.h"

using namespace mu;
using namespace mu::engraving;

namespace mu::iex::capella {
extern engraving::Err importCapella(MasterScore*, const QString& name);
extern engraving::Err importCapXml(MasterScore*, const QString& name);
}

static const String CAPELLA_DIR("data/");

class Capella_Tests : public ::testing::Test
{
public:
    void capReadTest(const char* file);
    void capxReadTest(const char* file);
};

//---------------------------------------------------------
//   capReadTest
//   read a Capella file, write to a MuseScore file and verify against reference
//---------------------------------------------------------

void Capella_Tests::capReadTest(const char* file)
{
    auto importFunc = [](MasterScore* score, const io::path_t& path) -> engraving::Err {
        return mu::iex::capella::importCapella(score, path.toQString());
    };

    String fileName = String::fromUtf8(file);
    MasterScore* score = ScoreRW::readScore(CAPELLA_DIR + fileName + ".cap", false, importFunc);
    EXPECT_TRUE(score);
    score->setMetaTag(u"originalFormat", u"cap");

    EXPECT_TRUE(ScoreComp::saveCompareScore(score, fileName + u".mscx", CAPELLA_DIR + fileName + u".cap-ref.mscx"));
    delete score;
}

//---------------------------------------------------------
//   capxReadTest
//   read a CapellaXML file, write to a MuseScore file and verify against reference
//---------------------------------------------------------

void Capella_Tests::capxReadTest(const char* file)
{
    auto importFunc = [](MasterScore* score, const io::path_t& path) -> engraving::Err {
        return mu::iex::capella::importCapXml(score, path.toQString());
    };

    String fileName = String::fromUtf8(file);
    MasterScore* score = ScoreRW::readScore(CAPELLA_DIR + fileName + ".capx", false, importFunc);
    EXPECT_TRUE(score);
    score->setMetaTag(u"originalFormat", u"capx");

    EXPECT_TRUE(ScoreComp::saveCompareScore(score, fileName + u".mscx", CAPELLA_DIR + fileName + u".capx-ref.mscx"));
    delete score;
}

TEST_F(Capella_Tests, capTest1) {
    capReadTest("test1");
}
TEST_F(Capella_Tests, capTest2) {
    capReadTest("test2");
}
TEST_F(Capella_Tests, capTest3) {
    capReadTest("test3");
}
TEST_F(Capella_Tests, capTest4) {
    capReadTest("test4");
}                                                               // wrong enharmonic spelling
TEST_F(Capella_Tests, capTest5) {
    capReadTest("test5");
}
TEST_F(Capella_Tests, capTest6) {
    capReadTest("test6");
}
TEST_F(Capella_Tests, capTest7) {
    capReadTest("test7");
}                                                               // double bar missing (auto-generated by Capella programs)
TEST_F(Capella_Tests, capTest8) {
    capReadTest("test8");
}
TEST_F(Capella_Tests, capTestTuplet2) {
    capReadTest("testTuplet2");
}                                                                           // generates different beaming with respect to the original
TEST_F(Capella_Tests, capxTest1) {
    capxReadTest("test1");
}
TEST_F(Capella_Tests, capxTest2) {
    capxReadTest("test2");
}
TEST_F(Capella_Tests, capxTest3) {
    capxReadTest("test3");
}
TEST_F(Capella_Tests, capxTest4) {
    capxReadTest("test4");
}                                                                 // wrong enharmonic spelling
TEST_F(Capella_Tests, capxTest5) {
    capxReadTest("test5");
}
TEST_F(Capella_Tests, capxTest6) {
    capxReadTest("test6");
}
TEST_F(Capella_Tests, DISABLED_capxTest7) { // wrong cap xml ( element double close (<head pitch="E5"/> </head>) )  // double bar missing (auto-generated by Capella programs)
    capxReadTest("test7");
}
TEST_F(Capella_Tests, capxTestEmptyStaff1) {
    capxReadTest("testEmptyStaff1");
}
TEST_F(Capella_Tests, capxTestEmptyStaff2) {
    capxReadTest("testEmptyStaff2");
}
TEST_F(Capella_Tests, capxTestPianoG4G5) {
    capxReadTest("testPianoG4G5");
}
TEST_F(Capella_Tests, capxTestScaleC4C5) {
    capxReadTest("testScaleC4C5");
}
TEST_F(Capella_Tests, capxTestSlurTie) {
    capxReadTest("testSlurTie");
}
TEST_F(Capella_Tests, capxTestText1) {
    capxReadTest("testText1");
}
TEST_F(Capella_Tests, capxTestTuplet1) {
    capxReadTest("testTuplet1");
}                                                                             // generates different (incorrect ?) l1 and l2 values in beams
TEST_F(Capella_Tests, capxTestTuplet2) {
    capxReadTest("testTuplet2");
}                                                                             // generates different beaming with respect to the original
TEST_F(Capella_Tests, capxTestVolta1) {
    capxReadTest("testVolta1");
}

//!Note Temporarily disabled, in some cases Capella Import produces a wrong amount of measures
TEST_F(Capella_Tests, DISABLED_capxTestBarline) {
    capxReadTest("testBarline");
}
