/***************************************************************************
 *   Copyright (C) 2005 by Vincent Wagelaar                                *
 *   vincent@ricardis.tudelft.nl                                           *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <kglobal.h>
#include <klocale.h>
#include <kconfig.h>
#include <kapplication.h>
#include <kmessagebox.h>
#include <qstringlist.h>
#include <qlayout.h>
#include <kaboutdata.h>
#include <kaboutapplication.h>
#include <qsizepolicy.h>
#include <qnamespace.h>
#include <qframe.h>
#include <qtimer.h>
#include <kconfigdialog.h>
#include <kfilemetainfo.h>
#include <qimage.h>
#include <qpainter.h>
#include <kprotocolinfo.h>
#include <qpalette.h>
#include <qsimplerichtext.h>
#include <qdrawutil.h>
#include <klistview.h>
#include "diskfree.h"
#include "diskfreesettings.h"
#include "prefs.h"

static const char *version = VERSION;

class DiskFreeItem
{
public:
    DiskFreeItem() : mName(0) {}
    DiskFreeItem(const QString&name, const QString&info, const QColor& color, const double usage)
    : mName(name), mInfo(info), mColor(color), mUsage(usage){}
    QString name() const { return mName; }
    QString info() const { return mInfo; }
    QColor color() const { return mColor; }
    double usage() const { return mUsage; }

private:
    QString mName;
    QString mInfo;
    QColor mColor;
    double mUsage;
};

typedef QValueList<DiskFreeItem> DiskFreeItemList;

class DiskFreeLabel : public QLabel
{
public:
    DiskFreeLabel(QWidget *parent =0, const char *name =0 );
    void setItems(DiskFreeItemList items);
    virtual QSize minimumSizeHint () const;
    virtual QSize sizeHint () const;
    virtual void setFont ( const QFont &font);

protected:
    virtual void drawContents(QPainter *p);
    void recalculateSize();
    void drawUsageRect(QPainter *p, const int x, const int y, const double usage, const QColor& col);
private:
    QStringList mItems;
    int mMinimumSize;
    DiskFreeItemList mDiskFreeItems;
};

DiskFreeLabel::DiskFreeLabel(QWidget *parent, const char *name) :
    QLabel(parent, name)
{
    setSizePolicy(QSizePolicy(QSizePolicy::Minimum, QSizePolicy::MinimumExpanding));
    setMargin(0);
    setMinimumSize(40,10);
    mMinimumSize = 10;
}

void DiskFreeLabel::setItems(DiskFreeItemList items)
{
    mDiskFreeItems = items;
    recalculateSize();
    repaint();
}

void DiskFreeLabel::setFont(const QFont &font)
{
    QFrame::setFont(font);
    recalculateSize();
}

void DiskFreeLabel::recalculateSize()
{
    int heightInc = fontMetrics().height();
    int numrows = height() / heightInc;
    numrows = numrows > 0 ? numrows : 1;
    int x=0, w=0, w2=0;

    DiskFreeItemList::Iterator it = mDiskFreeItems.begin();
    DiskFreeItemList::Iterator dfiIt = mDiskFreeItems.begin();

    QFont f(font());
    f.setBold(true);
    QFontMetrics fm(f);

    for (uint iRow=0; iRow < mDiskFreeItems.count(); iRow++)
    {
        mMinimumSize = 0;
        if (iRow % numrows == 0)
        {
            w=0;
            int end = iRow + numrows < mDiskFreeItems.count() ? iRow + numrows : mDiskFreeItems.count();
            for (mDiskFreeItems.at(iRow); it != mDiskFreeItems.at(end); ++it)
            {
                w = QMAX(w, fontMetrics().width((*it).name()));

            }
            //qDebug("%dbladiebla", iRow);
        }
        w2 = QMAX(w2, fm.width((*dfiIt).info()));
        if ((iRow+1) % numrows == 0)
        {
            x+=w+w2+fontMetrics().width("l")*10;
            mMinimumSize = x;
            w2=0;
        }
        ++dfiIt;
    }
    if (!mMinimumSize)
        mMinimumSize = x+w+w2+fontMetrics().width("l")*10;
    adjustSize();
}

void DiskFreeLabel::drawContents(QPainter *p)
{
    if (!isVisible())
        return;
    if (!mDiskFreeItems.count())
        return;
    recalculateSize();
    QRect cr = contentsRect();
    // Put the top of the character to the top op the contentsRect
    int h = fontMetrics().ascent()-fontMetrics().lineSpacing();
    int w = 0, x=0, w2=0;

    //Heightincrement not using linespacing to make it more compact
    int heightInc = fontMetrics().height();
    /*
      For centering horizontally we first calculate the number of rows possisble
      than the difference between the height available and
      plus the negative offset between ascent and linespacing.
    */
    int starth = h + cr.height()/2 - (cr.height() / (heightInc-1)) * (heightInc-1)/2;

    h = starth;
    uint iRow=0;
    uint numrows = cr.height() / heightInc;
    numrows = numrows > 0 ? numrows : 1;

    QFont f(font());
    f.setBold(true);
    QFontMetrics fm(f);


    DiskFreeItemList::Iterator it = mDiskFreeItems.begin();
    DiskFreeItemList::Iterator dfiIt = mDiskFreeItems.begin();

    for (iRow=0; iRow < mDiskFreeItems.count(); iRow++)
    {

        //drawUsageRect(p, x+w, h, 20);

        if (iRow % numrows == 0)
        {
            w=0;
            int end = iRow + numrows < mDiskFreeItems.count() ? iRow + numrows : mDiskFreeItems.count();
            for (mDiskFreeItems.at(iRow); it != mDiskFreeItems.at(end); ++it)
            {
                w = QMAX(w, fontMetrics().width((*it).name()));

            }
        }
        if ((*dfiIt).info().length() > 0)
        {
            drawUsageRect(p, x+w, h, (*dfiIt).usage(), colorGroup().highlight());
        }
        else
        {
            drawUsageRect(p, x+w, h, (*dfiIt).usage(), (*dfiIt).color());
        }

        p->drawText(x, h+heightInc, (*dfiIt).name());
        QFont f(font());
        f.setBold(true);
        QPen oldPen = p->pen();
        p->setFont(f);
        p->setPen(QPen((*dfiIt).color()));
        p->drawText(x+w+9*fontMetrics().width("l"), h+heightInc, (*dfiIt).info());

        h += heightInc-1;
        p->setFont(font());
        p->setPen(oldPen);
        w2 = QMAX(w2, fm.width((*dfiIt).info()));
        if ((iRow+1) % numrows == 0)
        {
            x+=w+w2+10*fontMetrics().width("l");
            h=starth;
            w2=0;
        }
        ++dfiIt;
    }
}

void DiskFreeLabel::drawUsageRect(QPainter *p, const int x, const int y, const double usage, const QColor& col)
{
    //qDebug("%f", usage);
    QPen old = p->pen();
    QBrush oldb = p->brush();
    p->setPen(QPen(gray, 0, SolidLine, RoundCap, RoundJoin));
    //p->setBrush(QBrush(QColor(144, 173,81)));
    p->drawRect(x, y+fontMetrics().lineSpacing()-1, 8*fontMetrics().width("l"), -fontMetrics().ascent()+2);
    p->setBrush(QBrush(col));
    p->setPen(QPen(black, 0, NoPen));

    p->drawRect(x+2, y+fontMetrics().lineSpacing()-3, (int)((8*fontMetrics().width("l")-4)*usage/100.0), -fontMetrics().ascent()+6);
    //QBrush *brush = new QBrush(QColor(126, 190, 246));
    //qDrawWinButton(p, x+2, y+fontMetrics().lineSpacing()-3, 8*fontMetrics().width("l"), -fontMetrics().ascent()+2, colorGroup(), false, brush);
    p->setPen(old);
    p->setBrush(oldb);
}
QSize DiskFreeLabel::minimumSizeHint () const
{
    return QSize(mMinimumSize, height());
}

QSize DiskFreeLabel::sizeHint () const
{
    return QSize(mMinimumSize, height());
}


DiskFree::DiskFree(const QString& configFile, Type type, int actions, QWidget *parent, const char *name)
    : KPanelApplet(configFile, type, actions, parent, name),
    mConfigFile(configFile.latin1()),
    mSettings(new DiskFreeSettings(sharedConfig()))

{
    mSettings->readConfig();
    readSettings();

    //Check for media protocol for KDE 3.4
    mURL = KURL("media:/");
    if ( KProtocolInfo::isKnownProtocol(KURL("media:/")) )
    {
        mIncludeFilter << "media/hdd_mounted";
    }
    else
    {
        //Else use devices protocol from KDE < 3.4
        mURL = KURL("devices:/");
        mIncludeFilter << "kdedevice/hdd_mounted";
    }



    /*QPixmap image(100, 100);
    image.fill(colorGroup().background());
    QPainter p;

    p.begin(&image);
    p.setPen(QPen(black, 6));
    p.setBrush(QBrush(yellow));
    p.drawEllipse(10,10,90, 90);
    p.setBrush(QBrush(green));
    p.drawPie(10, 10, 90, 90, 0, 45*16);
    p.end();

    QMimeSourceFactory::defaultFactory()->setPixmap( "myimage", image );*/

    //QHBoxLayout *layout = new QHBoxLayout(this);
    //layout->setResizeMode(QLayout::Minimum);
    //mLabel = new QLabel(this);
    //mLabel->setFont(mFont);
    //mLabel->setSizePolicy(QSizePolicy(QSizePolicy::Minimum, QSizePolicy::Minimum));
    //mLabel->setAlignment( AlignVCenter );
    //mLabel->hide();
    //QStringList loadMsg;
    //loadMsg << "<b></b>";
    //loadMsg << "<b>Loading...</b>";
    //loadMsg << "<b></b>";

    mLabel2 = new DiskFreeLabel(this);
    mLabel2->setFont(mFont);

    mDirLister = new KDirLister();

    connect (mDirLister, SIGNAL(completed()),
             this, SLOT(slotDirListComplete()));

    mTimer = new QTimer(this);
    connect (mTimer, SIGNAL(timeout()),
             this, SLOT(slotDirListComplete()));


    slotDoNewDirList();
    mTimer->start(mRefreshInterval);
    //mLabel2->setItems(loadMsg);
}


DiskFree::~DiskFree()
{
    KGlobal::locale()->removeCatalogue("diskfree");
    delete mDirLister;
    delete mSettings; mSettings =0;
    config()->sync();
}

void DiskFree::slotDoNewDirList()
{
    if (!mDirLister->isFinished())
        return;

    mDirLister->stop();
    mDirLister->setDirOnlyMode(true);
    mDirLister->clearMimeFilter();
    mDirLister->setMimeFilter(mIncludeFilter);
    mDirLister->openURL( mURL, false, true );
}

void DiskFree::slotDirListComplete()
{

    mMedia = mDirLister->items();
    QString drive;
    QString driveInfo;
    double usage;
    DiskFreeItemList listje;
    for (uint i=0; i< mMedia.count(); i++)
    {
        if (mExcludedDrives.contains(mMedia.at(i)->url().url()))
            continue;
        driveInfo = "";
        drive = "";
        QColor itemColor;
        KFileMetaInfo info(mMedia.at(i)->url(), mMedia.at(i)->mimetype());

        //KDE 3.3 is/was crashing here...
        //qDebug("%s", info.supportedKeys().join("\n").latin1());
        if (info.item("free").value().toInt() > mSufficientSpace * 1024)
        {
            itemColor = mSufficientColor;
        }
        else if (info.item("free").value().toInt() < mCriticalSpace * 1024)
        {
            itemColor = mCriticalColor;
        }
        else
        {
            itemColor = mWarnColor;
        }
        if (mShowPartition)
        {
            drive.append(mMedia.at(i)->url().path().replace("/", "") + " ");
        }
        if (mShowMountDir)
        {
            drive.append(info.item("baseURL").string().replace("file://", "") + " ");
        }
        if (mShowFreeSpace)
        {
            driveInfo.append(" "+info.item("free").string());
        }
        if (mShowUsedSpace)
        {
            driveInfo.append(" "+info.item("used").string());
        }
        if (mShowTotalSpace)
        {
            driveInfo.append(" " + info.item("total").string());
        }
        double freeprc = info.item("free").value().toDouble() / info.item("total").value().toDouble() * 100.0;
        usage = 100 - freeprc;
        if (mShowFreePercentage)
        {
            QString tmp1;
            tmp1= tmp1.sprintf(" %.0f%%", freeprc);
            driveInfo.append(tmp1);
        }
        if (mShowUsedPercentage)
        {
            QString tmp1;
            tmp1= tmp1.sprintf(" %.0f%%", 100.0 - freeprc);
            driveInfo.append(tmp1);
        }

        listje << DiskFreeItem(drive, driveInfo.stripWhiteSpace(), itemColor, usage);
    }

    mLabel2->setItems(listje);
    emit updateLayout();
}

void DiskFree::about()
{

    KAboutData data("diskfreeapplet",
                    I18N_NOOP("Free Disk Space Applet"),
                    version,
                    I18N_NOOP("Free Disk Space Applet"),
                    KAboutData::License_GPL_V2,
                    "(c) 2005, Vincent Wagelaar");
    data.addAuthor("Vincent Wagelaar",
                   I18N_NOOP("Maintainer"),
                   "vincent@ricardis.tudelft.nl",
                   "http://hannibal.lr-s.tudelft.nl/diskfree/");
    KAboutApplication dialog(&data);
    dialog.exec();
}


void DiskFree::help()
{
    KMessageBox::information(0, i18n("This is a help box"));
}


void DiskFree::preferences()
{

    Prefs *dialog = dynamic_cast<Prefs *>(KConfigDialog::exists(mConfigFile));

    if (!dialog)
    {
    //KConfigDialog didn't find an instance of this dialog, so lets create it :
        dialog = new Prefs(this, mMedia, mSettings, mConfigFile);


    //User edited the configuration - update your local copies of the
    //configuration data
        connect( dialog, SIGNAL(settingsChanged()),
                 this, SLOT(slotUpdateSettings()) );
    }

    dialog->show();

}

int DiskFree::widthForHeight(int /*height*/) const
{
    if (mLabel2)
        return mLabel2->minimumSizeHint().width();
    else
        return 40;
}

int DiskFree::heightForWidth(int /*width*/) const
{
    return height();
}

void DiskFree::slotUpdateSettings()
{
    mSettings->writeConfig();
    readSettings();
    slotDirListComplete();
    mLabel2->setFont(mFont);
    mTimer->changeInterval(mRefreshInterval);
}

void DiskFree::resizeEvent(QResizeEvent */*e*/)
{
    mLabel2->resize(mLabel2->minimumSizeHint().width(), height());
    resize(mLabel2->minimumSizeHint().width(), height());
}
void DiskFree::readSettings()
{
    mRefreshInterval    = mSettings->refreshInterval();
    mSufficientColor    = mSettings->sufficientColor();
    mWarnColor          = mSettings->warnColor();
    mCriticalColor      = mSettings->criticalColor();
    mFont               = mSettings->font();
    mSufficientSpace    = mSettings->sufficientSpace();
    mCriticalSpace      = mSettings->criticalSpace();
    mShowTotalSpace     = mSettings->showTotalSpace();
    mShowFreeSpace      = mSettings->showFreeSpace();
    mShowUsedSpace      = mSettings->showUsedSpace();
    mShowFreePercentage = mSettings->showFreePercentage();
    mShowUsedPercentage = mSettings->showUsedPercentage();
    mExcludedDrives     = mSettings->excludedDrives();
    mShowMountDir       = mSettings->showMountDir();
    mShowPartition      = mSettings->showPartition();
}

extern "C"
{
    KPanelApplet* init( QWidget *parent, const QString& configFile)
    {
        KGlobal::locale()->insertCatalogue("diskfree");
        return new DiskFree(configFile, KPanelApplet::Normal,
                            KPanelApplet::About /*|  KPanelApplet::Help*/ | KPanelApplet::Preferences,
                             parent, "diskfree");
    }
}
