/***************************************************************************
 *   Copyright (C) 2005 by Vincent Wagelaar                                *
 *   vincent@ricardis.tudelft.nl                                           *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "prefs.h"
#include "diskfreesettings.h"
#include "settingswidget.h"
#include <kmessagebox.h>

MediumItem::MediumItem(QListView *parent, const QString name, const KFileItem medium)
    : QCheckListItem(parent, name, CheckBox),
        mMedium(medium)
{
    KFileMetaInfo info(mMedium.url(), mMedium.mimetype());
    setText(1, info.item("baseURL").string().replace("file://", " "));
}
const QString MediumItem::itemURL() const
{
    return mMedium.url().url();
}
void MediumItem::stateChange(bool state)
{
    static_cast<MediumListView *>(listView())->_myStateChange(this, state);
}

MediumListView::MediumListView(QWidget *parent, const char *name)
    : KListView(parent, name)
{
    addColumn( i18n("Partition to Display") );
    addColumn( i18n("Mountpoint") );
    setResizeMode( AllColumns );
}

void MediumListView::_myStateChange(QCheckListItem *item, bool state)
{
    emit stateChange(item, state);
}


MediumWidget::MediumWidget(KFileItemList items, QWidget *parent, const char *name)
    : QWidget(parent, name),
    mMedia(items)
{
    QVBoxLayout *l = new QVBoxLayout(this);
    mMediaListView = new MediumListView(this);
    l->addWidget(mMediaListView);
}

void MediumWidget::initFiles()
{
    disconnect (mMediaListView, SIGNAL(stateChange(QCheckListItem *, bool)), 0, 0);
    mMediaListView->clear();
    KFileItemListIterator it( mMedia );
    KFileItem *file;

    while ( (file = it.current()) != 0 )
    {
        ++it;
        bool ok = mExcludedList.contains(file->url().url())==0;
        MediumItem *item = new MediumItem(mMediaListView, file->text(), *file);
        item->setOn(ok);
    }
    connect (mMediaListView, SIGNAL(stateChange(QCheckListItem *, bool)),
             this, SLOT(slotStateChange(QCheckListItem *, bool) ) );

}

void MediumWidget::slotStateChange(QCheckListItem *item, bool state)
{
    if (state)
        mExcludedList.remove( static_cast<MediumItem *>(item)->itemURL() );
    else
        mExcludedList << static_cast<MediumItem *>(item)->itemURL();
    emit changed();
}

QStringList MediumWidget::excludedDrives()
{
    return mExcludedList;
}

void MediumWidget::setExcludedDrives(QStringList drives)
{
    mExcludedList = drives;
}

Prefs::Prefs(QWidget *parent, KFileItemList files, DiskFreeSettings *settings, const char *name):
       KConfigDialog( parent, name, settings, KDialogBase::IconList),
       mSettings(settings)

{
    addPage(new SettingsWidget(0, "settings_widget" ), i18n("General"), "misc");
    mMediumWidget = new MediumWidget(files, 0, "media_widget");
    addPage(mMediumWidget, i18n("Partitions"), "folder");
    connect( mMediumWidget, SIGNAL (changed()), this, SLOT(updateButtons()));
}

void Prefs::updateWidgets()
{
    mMediumWidget->setExcludedDrives(mSettings->excludedDrives());
    mMediumWidget->initFiles();
}

void Prefs::updateWidgetsDefault()
{
    mSettings->useDefaults(true);
    mMediumWidget->setExcludedDrives(mSettings->excludedDrives());
    mSettings->useDefaults(false);
    mMediumWidget->initFiles();
}

void Prefs::updateSettings()
{
    mSettings->setExcludedDrives(mMediumWidget->excludedDrives());
    settingsChangedSlot();
}
bool Prefs::hasChanged()
{
    QStringList drives1 = mMediumWidget->excludedDrives();
    QStringList drives2 = mSettings->excludedDrives();

    if (drives1.count() != drives2.count())
        return true;
    for ( QStringList::Iterator it = drives1.begin(); it != drives1.end(); ++it )
    {
        if (!drives2.contains((*it)))
            return true;
    }
    return false;
}
bool Prefs::isDefault()
{
    return (mMediumWidget->excludedDrives().count() == 0);
}
