/*
 * Copyright (c) 2003 Patrick van Staveren <pvanstav@cs.wmich.edu>
 * Copyright (c) 2005 Kemal Ilgar Eroglu <kieroglu@math.washington.edu>
 * Copyright (c) 2005 Mario Streiber <mario.streiber@gmx.de>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <sys/wait.h>

#include <gtk/gtk.h>

/*#include <panel/plugins.h>
#include <panel/xfce.h>*/
#include <libxfce4util/libxfce4util.h>
#include <libxfcegui4/libxfcegui4.h>
#include <libxfce4panel/xfce-panel-plugin.h>

#include "xmms_plugin.h"
#include "popupmenu.h"
#include "preferences.h"
#include "playerctrl.h"

gboolean xmms_plugin_control_new(XfcePanelPlugin *plugin);
XFCE_PANEL_PLUGIN_REGISTER_EXTERNAL(xmms_plugin_control_new);

/******************/
/* xmms_control.c */
/******************/


/**************************************/
/* displays or hides the xmms windows */
/**************************************/
static void display_xmms(plugin_data *pd, gboolean show) {
	/* do nothing if xmms is not running */
	if (!pd->pctrl->player_is_running(pd->xmms_session)) return;

	if (show) {
		/* display xmms windows based on their prior status */
		if (pd->show_mw) pd->pctrl->player_main_win_toggle(pd->xmms_session, TRUE);
		if (pd->show_pl) pd->pctrl->player_pl_win_toggle  (pd->xmms_session, TRUE);
		if (pd->show_eq) pd->pctrl->player_eq_win_toggle  (pd->xmms_session, TRUE);
		pd->xmmsvisible = TRUE;
	}
	else {
		/* save status of each xmms window */
		pd->show_mw = pd->pctrl->player_is_main_win(pd->xmms_session);
		pd->show_pl = pd->pctrl->player_is_pl_win  (pd->xmms_session);
		pd->show_eq = pd->pctrl->player_is_eq_win  (pd->xmms_session);

		/* make sure at least the main window is shown on redisplay */
		if (!(pd->show_pl || pd->show_eq)) pd->show_mw = TRUE;

		/* hide all */
		pd->pctrl->player_pl_win_toggle  (pd->xmms_session, FALSE);
		pd->pctrl->player_eq_win_toggle  (pd->xmms_session, FALSE);
		// should hide main window at the end of process
		// if not, audacious cannot store the window position
		pd->pctrl->player_main_win_toggle(pd->xmms_session, FALSE);
		pd->xmmsvisible = FALSE;
	}
}


/*********************************************************/
/* callback to jump to the previous song in the playlist */
/* left click - simply jump                              */
/* middle click - jump and remove current song           */
/*********************************************************/
static void prev(GtkWidget *widget, GdkEventButton* event, gpointer data) {
	gint pos;
	plugin_data *pd = (plugin_data *) data;
	switch (event->button) {
		case 1: // left click
			pd->pctrl->player_playlist_prev(pd->xmms_session);
			break;
		case 2: // middle click
			pos = pd->pctrl->player_get_playlist_pos(pd->xmms_session);
			pd->pctrl->player_playlist_prev(pd->xmms_session);
			pd->pctrl->player_playlist_delete(pd->xmms_session, pos);
			break;
	}
}


/************************************************************************/
/* callback to start playing; if xmms is not running it will be started */
/************************************************************************/
static void play(GtkWidget *widget, GdkEventButton* event, gpointer data) {
	plugin_data *pd = (plugin_data*) data;
	int i=0;
	pid_t pid;
	int status;
	gchar *cmd;

	if (event->button != 1) return;

	/* start xmms and hide its windows if not already running */
	if (!pd->pctrl->player_is_running(pd->xmms_session)) {
		cmd = g_strconcat(player_control_get_command(pd->player), " -p", NULL);
		if (exec_command(cmd)) {
			while (!pd->pctrl->player_is_running(pd->xmms_session) && (i <= 5)) {
				sleep(1);
				i++;
			}
		}
		g_free(cmd);
		if ((i < 5) && (!pd->xmmsvisible))
			display_xmms(pd, FALSE);
	}
		else pd->pctrl->player_play(pd->xmms_session);
}


/*****************************/
/* callback for pausing xmms */
/*****************************/
static void paus(GtkWidget *widget, GdkEventButton* event, gpointer data) {
	if (event->button != 1) return;
	plugin_data *pd = (plugin_data *) data;
	pd->pctrl->player_pause(pd->xmms_session);
}


/******************************/
/* callback for stopping xmms */
/******************************/
static void stop(GtkWidget *widget, GdkEventButton* event, gpointer data) {
	if (event->button != 1) return;
	plugin_data *pd = (plugin_data *) data;
	pd->pctrl->player_stop(pd->xmms_session);
}

/*****************************************************/
/* callback to jump to the next song in the playlist */
/* left click - simply jump                          */
/* middle click - jump and remove current song       */
/*****************************************************/
static void next(GtkWidget *widget, GdkEventButton* event, gpointer data) {
	gint pos;
	plugin_data *pd = (plugin_data *) data;
	switch (event->button) {
		case 1: // left click
			pd->pctrl->player_playlist_next(pd->xmms_session);
			break;
		case 2: // middle click
			pos = pd->pctrl->player_get_playlist_pos(pd->xmms_session);
			pd->pctrl->player_playlist_delete(pd->xmms_session, pos);
			set_song_title(pd);
			break;
	}
}

/**
 * callback for pressing menu button
 * left click - popup menu
 * middle click - toggle xmms window visibility
 */
static void menu_btn_pressed(GtkWidget *widget, GdkEventButton *event, gpointer data) {
	gint pos;
	plugin_data *pd = (plugin_data *) data;
	switch (event->button) {
		case 1: // left click
			gtk_menu_popup(GTK_MENU(create_popupmenu(pd)),
					NULL, NULL, NULL, NULL,
					event->button, event->time);
			break;
		case 2: // middle click
			display_xmms(pd, !pd->xmmsvisible);
			break;
	}
}

/****************************************************************/
/* callback for changing the volume when using the scroll wheel */
/****************************************************************/
static void box_scroll(GtkWidget* widget, GdkEventScroll* event, gpointer data) {
	gint vl, vr;
	plugin_data *pd = (plugin_data*) data;

	if (! pd->pctrl->player_is_running(pd->xmms_session))
		return;

	pd->pctrl->player_get_volume(pd->xmms_session, &vl, &vr);
	if(event->direction == GDK_SCROLL_UP)
		pd->pctrl->player_set_volume(pd->xmms_session, vl+8, vr+8);
	else
		pd->pctrl->player_set_volume(pd->xmms_session, vl-8, vr-8);

	pd->pctrl->player_get_volume(pd->xmms_session, &vl, &vr);
	/* XMMS may return -1 (Alsa/OSS output issue) */
	if (vl >= 0)
		gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(pd->vol_pbar), ((double)(MAX(vl,vr)))/100);
}

/**********************************************/
/* callback for clicking on the progress bar  */
/* button 1: jump to time in the current song */
/* button 2: toggle xmms window visibility    */
/* button 3: popup menu                       */
/**********************************************/
static void pbar_click(GtkWidget* widget, GdkEventButton* event, gpointer data) {
	gint width, time, total;
	plugin_data *pd = (plugin_data *) data;

	switch (event->button) {
		case 1:
			// jump to time within this song
			width = pd->pbar->allocation.width;
			total = pd->pctrl->player_get_playlist_time(pd->xmms_session,
					pd->pctrl->player_get_playlist_pos(pd->xmms_session));
			time  = (int) (event->x * total / width);
			pd->pctrl->player_jump_to_time(pd->xmms_session, time);
			break;
		case 2:
			// toggle xmms window visibility
			display_xmms(pd, !pd->xmmsvisible);
			break;
		case 3:
			// popup menu
			gtk_menu_popup(GTK_MENU(create_popupmenu(pd)),
					NULL, NULL, NULL, NULL,
					event->button, event->time);
			break;
	}
}

/**
 * callback for changing the volume when pressing the progress bar
 */
static void vol_pbar_pressed(GtkWidget *widget, GdkEventButton *event, gpointer data) {
	gint vol;
	plugin_data *pd = (plugin_data *) data;

	if (event->button != 1) return;

	if (gtk_progress_bar_get_orientation(GTK_PROGRESS_BAR(widget))
				== GTK_PROGRESS_LEFT_TO_RIGHT)
		vol = (gint) (event->x * 100 / widget->allocation.width);
	else
		vol = (gint) ((widget->allocation.height - event->y) * 100
				/ widget->allocation.height);

	pd->pctrl->player_set_main_volume(pd->xmms_session, vol);
}

/*******************/
/* panel_display.c */
/*******************/


/************************************************/
/* Set song title in tooltip and scrolled label */
/************************************************/
void set_song_title(plugin_data *pd) {
	gchar    *title, *tooltip, *label;
	gint     pos, time;
	gboolean running = pd->pctrl->player_is_running(pd->xmms_session);

	if (running) {
		pos        = pd->pctrl->player_get_playlist_pos(pd->xmms_session);
		time       = pd->pctrl->player_get_playlist_time(pd->xmms_session, pos) / 1000;
		title      = pd->pctrl->player_get_playlist_title(pd->xmms_session, pos);
		if (pd->simple_title)
			tooltip = g_strdup_printf("%s", title);
		else
			tooltip = g_strdup_printf("%d: %s (%d:%02d)", pos+1, title, time/60, time%60);
	}
	else
		tooltip = g_strdup_printf(TITLE_STRING);
	gtk_tooltips_set_tip            (pd->tooltip, GTK_WIDGET(pd->base), tooltip, NULL);

	if (pd->simple_title) label   = g_strdup_printf("%s %s ", tooltip, tooltip);
	else                  label   = g_strdup_printf("%s +++ %s +++", tooltip, tooltip);
	pd->labelattr->start_index    = 0;
	pd->labelattr->end_index      = strlen(label);
	gtk_label_set_attributes(GTK_LABEL(pd->label), pd->labelattrlist);
	gtk_label_set_text(GTK_LABEL(pd->label), label);
	//g_free(title);
	g_free(tooltip);
	g_free(label);
}


/*************************************************/
/* timeout function to update the plugin widgets */
/*************************************************/
gboolean pbar_label_update(gpointer data) {
	plugin_data    *pd = (plugin_data*) data;
	GtkAdjustment  *adj;
	gint           sp, len, time = 1, pl_pos = -1, play_time = 0, vl, vr;
	gchar          *timestr;
	PangoAttribute *attr;
	gboolean       running, playing;

	/* check xmms status */
	running  = pd->pctrl->player_is_running(pd->xmms_session);
	playing  = (running && pd->pctrl->player_is_playing(pd->xmms_session));
	if (running) { /* get playlist position and song length */
		pl_pos = pd->pctrl->player_get_playlist_pos(pd->xmms_session);
		time   = pd->pctrl->player_get_playlist_time(pd->xmms_session, pl_pos) / 1000;
	}

	/* update tooltip and song title */
	if (pd->playlist_position != pl_pos) {
		pd->playlist_position     = pl_pos;
		pd->title_scroll_position = 0;
		set_song_title(pd);
	}

	/* update progress bar */
       if (playing) {
               play_time = pd->pctrl->player_get_output_time(pd->xmms_session) / 1000;
               /* make sure title gets updated when new files are loaded */
               if (!play_time) {
                       pd->title_scroll_position = 0;
                       set_song_title(pd);
               }
       }	
	gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(pd->pbar), (gdouble) play_time / time);

	/* scroll song title */
	if ((pd->show_scrolledtitle) && (pd->scroll_step > 0)) {
		adj = gtk_viewport_get_hadjustment(GTK_VIEWPORT(pd->viewport));
		len = pd->label->allocation.width / 2;
		sp  = pd->title_scroll_position - pd->step_delay;
		sp  = (sp < 0) ? 0 : sp;
		sp  = (len > 0) ? sp % len : 0;
		gtk_adjustment_set_value(adj, sp);
		pd->title_scroll_position += pd->scroll_step;
	}

	if((running) && (pd->vol_pbar_visible)){
		pd->pctrl->player_get_volume(pd->xmms_session, &vl, &vr);
		if ( vl >= 0 ) /* XMMS may return -1 (Alsa/OSS output issue) */
		gtk_progress_bar_set_fraction     (GTK_PROGRESS_BAR(pd->vol_pbar), ((double)(MAX(vl,vr)))/100);    
	}

	/* set up new timer and destroy old if new scroll speed has been set */
	if (pd->timer_reset) {
		g_source_remove(pd->timeout);
		pd->timeout = g_timeout_add(1000 / pd->scroll_speed, pbar_label_update, pd);
		pd->timer_reset = FALSE;
		return FALSE;
	}
	return TRUE;
}




/******************************************************/
/* frees all resources used by the plugin (I hope :-) */
/******************************************************/
static void xmms_plugin_free(XfcePanelPlugin *plugin, plugin_data *pd) {

	g_return_if_fail(plugin != NULL);

	/* remove timeout */
	if (pd->timeout) g_source_remove(pd->timeout);

	/* destroy all widgets */
	gtk_widget_destroy(pd->boxMain);

	/* destroy text attribute for the label widget */
	pango_attribute_destroy(pd->labelattr);

	/* destroy the tooltips */
	gtk_object_destroy(GTK_OBJECT(pd->tooltip));

	/* let xmms exit if quit_xmms option is active */
	if (pd->quit_xmms) pd->pctrl->player_quit(pd->xmms_session);

	// unload library
	player_control_close(pd->pctrl);

	/* free the plugin data structure */
	g_free(pd);
}

/*************************************/
/* Adjust the position of volume bar */
/*************************************/

void adjust_vol_pbar(plugin_data *pd) {

	gtk_widget_hide(pd->vol_pbar);

	if(pd->vol_pbar){
		g_object_ref(G_OBJECT(pd->vol_pbar));
		gtk_container_remove(GTK_CONTAINER(gtk_widget_get_parent(pd->vol_pbar)),
				pd->vol_pbar);
	}  

	/* use horizontal volume bar regardless of panel orientation */
	if (pd->hor_vol_if_vertical) {
		gtk_progress_bar_set_orientation(GTK_PROGRESS_BAR(pd->vol_pbar),
				GTK_PROGRESS_LEFT_TO_RIGHT);
		gtk_widget_set_size_request(pd->vol_pbar, 0, 4);
		gtk_box_pack_start(GTK_BOX(pd->boxV), pd->vol_pbar, DOEXPAND, DOFILL, PADDING);
	} else {
		gtk_progress_bar_set_orientation(GTK_PROGRESS_BAR(pd->vol_pbar),
				GTK_PROGRESS_BOTTOM_TO_TOP);
		gtk_widget_set_size_request(pd->vol_pbar, 6, 0);
		gtk_box_pack_start(GTK_BOX(pd->boxMain),pd->vol_pbar, FALSE, FALSE, PADDING);
	}
	if (pd->vol_pbar_visible)
		gtk_widget_show(pd->vol_pbar);

	if( xfce_panel_plugin_get_orientation(pd->base)==GTK_ORIENTATION_VERTICAL)
		gtk_widget_set_size_request(GTK_WIDGET(pd->base),
				xfce_panel_plugin_get_size(pd->base), -1);
	else
		gtk_widget_set_size_request(GTK_WIDGET(pd->base),
				-1, xfce_panel_plugin_get_size(pd->base));  

}

/**
 * panel resize callback
 */
static gboolean xmms_plugin_set_size(XfcePanelPlugin *plugin, int size, plugin_data *pd) {
	adjust_vol_pbar(pd);
	return TRUE;
}

/*************************************/
/* Panel orientation change callback */
/*************************************/
static void orient_change(XfcePanelPlugin *plugin, GtkOrientation orient, plugin_data *pd){
	adjust_vol_pbar(pd);
}

/*****************************/
/* apply visibility settings */
/*****************************/
void apply_visibility_settings(plugin_data *pd) {

	PangoAttrSize *attr = (PangoAttrSize*) pd->labelattr;

	attr->size = pd->titletextsize * PANGO_SCALE;
	gtk_label_set_attributes(GTK_LABEL(pd->label), pd->labelattrlist);

	if(!pd->show_scrolledtitle)
		gtk_widget_hide_all (pd->viewport);
	/* show/hide the event box parent of the scrolled title */
	if (pd->show_scrolledtitle)
		gtk_widget_show_all(gtk_widget_get_parent(pd->label));
	else                          
		gtk_widget_hide_all(gtk_widget_get_parent(pd->label));


	if(pd->pbar_visible)
		gtk_widget_show_all  (pd->pbar);
	else
		gtk_widget_hide_all  (pd->pbar);   

	if(pd->vol_pbar_visible)
		gtk_widget_show_all  (pd->vol_pbar);
	else
		gtk_widget_hide_all  (pd->vol_pbar); 								 	

	if (pd->menu_btn_visible)
		gtk_widget_show(pd->menu_btn);
	else
		gtk_widget_hide(pd->menu_btn);
}

/***************************************************************/
/* adds a new button with image and callback to the parent box */
/***************************************************************/
static void new_button_with_img(GtkWidget *box, gchar *filename, gpointer cb, gpointer data) {
	GtkWidget *image, *eventbox;

	eventbox            = gtk_event_box_new();
	image               = gtk_image_new_from_file(filename);
	gtk_widget_show       (image);
	gtk_container_add     (GTK_CONTAINER(eventbox), image);
	gtk_widget_set_events (eventbox, GDK_BUTTON_PRESS_MASK);
	g_signal_connect      (G_OBJECT(eventbox), "button_press_event", G_CALLBACK(cb), data);
	gtk_box_pack_start    (GTK_BOX(box), eventbox, DOEXPAND, DOFILL, PADDING);
}

/******************************/
/* creates the plugin widgets */
/******************************/
gboolean xmms_plugin_control_new(XfcePanelPlugin *plugin) {
	GtkWidget *button, *box, *boxV, *boxMain, *pbar, *vol_pbar, *viewport, *eventbox, *label, *menu_btn,*image;
	plugin_data *pd;
	gchar *title = TITLE_STRING" +++ "TITLE_STRING" +++ ";
	gint vl, vr;
	GtkRcStyle *rc;
	GdkColor color;

	xfce_textdomain (GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR, "UTF-8");
	
	pd = g_new(plugin_data, 1);

	/* These defaults will be overwritten by read config */
	pd->titletextsize          = TITLE_SIZE;
	pd->title_scroll_position  = 0;
	pd->scroll_speed           = SCROLL_SPEED;
	pd->scroll_step            = SCROLL_STEP;
	pd->step_delay             = STEP_DELAY;
	pd->scroll_delay           = SCROLL_DELAY;
	pd->playlist_position      = -1;
	pd->xmmsvisible            = TRUE;
	pd->xmms_session           = 0;
	pd->timeout                = 0;
	pd->timer_reset            = FALSE;
	pd->show_scrolledtitle     = TRUE;
	pd->tooltip                = gtk_tooltips_new();
	pd->labelattrlist          = pango_attr_list_new();
	pd->labelattr              = pango_attr_size_new(pd->titletextsize * PANGO_SCALE);
	pd->labelattr->start_index = 0;
	pd->labelattr->end_index   = strlen(title);
	pd->quit_xmms              = FALSE;
	pd->simple_title           = FALSE;
	pd->pbar_visible           = TRUE;
	pd->vol_pbar_visible       = TRUE;
	pd->hor_vol_if_vertical    = TRUE;
	pd->menu_btn_visible       = TRUE;
	pango_attr_list_insert       (pd->labelattrlist, pd->labelattr);

	// check media player and set default one
	if (player_control_has_library(LIB_XMMS)) {
		pd->player = PLAYER_XMMS;
	} else if (player_control_has_library(LIB_BEEP)) {
		pd->player = PLAYER_BEEP;
	} else if (player_control_has_library(LIB_AUDACIOUS)) {
		pd->player = PLAYER_AUDACIOUS;
	} else {
		// TODO any other idea
		xfce_err(_("Cannot find any supported media player. You have to install at least one player first."));
		pango_attribute_destroy(pd->labelattr);
		g_free(pd);
		gtk_widget_destroy(GTK_WIDGET(plugin));
		return FALSE;
	}

	xfce_panel_plugin_set_expand(plugin,FALSE);


	g_signal_connect (plugin, "free-data", 
			G_CALLBACK (xmms_plugin_free), pd);

	g_signal_connect (plugin, "save", 
			G_CALLBACK (xmms_plugin_write_config), pd);

	xfce_panel_plugin_menu_show_configure (plugin);
	g_signal_connect (plugin, "configure-plugin", 
			G_CALLBACK (xmms_plugin_create_options), pd);

	g_signal_connect (plugin, "orientation-changed", 
			G_CALLBACK (orient_change), pd);

	g_signal_connect(plugin, "size-changed",
			G_CALLBACK(xmms_plugin_set_size), pd);

	/* add scrolling callback for the plugin base widget */
	pd->base                     = plugin;
	gtk_widget_add_events(GTK_WIDGET(plugin), GDK_SCROLL_MASK);
	g_signal_connect(G_OBJECT(plugin),"scroll_event",G_CALLBACK(box_scroll), pd);
	gtk_tooltips_set_tip           (pd->tooltip, GTK_WIDGET(plugin), TITLE_STRING, NULL);

	/* main container for the plugin widgets */
	boxMain                      = gtk_hbox_new(FALSE, 0);
	boxV                         = gtk_vbox_new(FALSE, 0);

	/* label for the song title */
	eventbox                     = gtk_event_box_new();
	label                        = gtk_label_new(title);
	gtk_label_set_line_wrap        (GTK_LABEL(label), FALSE);
	gtk_container_add              (GTK_CONTAINER(eventbox), label);
	gtk_widget_set_events          (eventbox, GDK_BUTTON_PRESS_MASK);
	gtk_label_set_attributes       (GTK_LABEL(label), pd->labelattrlist);

	/* viewport widget that manages the scrolling */
	viewport                     = gtk_viewport_new(NULL, NULL);
	gtk_viewport_set_shadow_type   (GTK_VIEWPORT(viewport), GTK_SHADOW_NONE);
	gtk_container_add              (GTK_CONTAINER(viewport), eventbox);
	gtk_widget_set_size_request    (viewport, 0, -1);
	gtk_box_pack_start             (GTK_BOX(boxV), viewport, DOEXPAND, DOFILL, PADDING);

	/* the progress bar */
	pbar                         = gtk_progress_bar_new();
	gtk_progress_bar_set_bar_style (GTK_PROGRESS_BAR(pbar), GTK_PROGRESS_CONTINUOUS);
	gtk_widget_set_size_request    (pbar, 0, 4);
	gtk_widget_set_events          (pbar, GDK_BUTTON_PRESS_MASK);
	g_signal_connect               (G_OBJECT(pbar), "button_press_event",
			G_CALLBACK(pbar_click), pd);
	gtk_box_pack_start             (GTK_BOX(boxV), pbar, DOEXPAND, DOFILL, PADDING);

	pd->boxMain  = boxMain;
	pd->boxV     = boxV;
	pd->viewport = viewport;
	pd->label    = label;
	pd->pbar     = pbar;

	/* box that contains the xmms control buttons */
	box                          = gtk_hbox_new(FALSE, 2);
	gtk_container_set_border_width(GTK_CONTAINER(box), 1);
	new_button_with_img(box, PREV, G_CALLBACK(prev), pd);
	new_button_with_img(box, PLAY, G_CALLBACK(play), pd);
	new_button_with_img(box, PAUS, G_CALLBACK(paus), pd);
	new_button_with_img(box, STOP, G_CALLBACK(stop), pd);
	new_button_with_img(box, NEXT, G_CALLBACK(next), pd);
	// create menu button
	menu_btn = gtk_event_box_new();
	image = gtk_image_new_from_file(MENU);
	gtk_widget_show(image);
	gtk_container_add(GTK_CONTAINER(menu_btn), image);
	gtk_widget_set_events(menu_btn, GDK_BUTTON_PRESS_MASK);
	g_signal_connect(G_OBJECT(menu_btn), "button_press_event",
			G_CALLBACK(menu_btn_pressed), pd);
	gtk_box_pack_start(GTK_BOX(box), menu_btn, DOEXPAND, DOFILL, PADDING);

	pd->menu_btn = menu_btn;

	gtk_box_pack_start(GTK_BOX(boxV), box, DOEXPAND, DOFILL, PADDING);    
	gtk_container_set_border_width(GTK_CONTAINER(boxMain), 2);

	/* the volume progress bar */
	vol_pbar                          = gtk_progress_bar_new();
	gtk_progress_bar_set_orientation  (GTK_PROGRESS_BAR(vol_pbar)
			,GTK_PROGRESS_BOTTOM_TO_TOP);
	gtk_progress_bar_set_bar_style    (GTK_PROGRESS_BAR(vol_pbar), 
			GTK_PROGRESS_CONTINUOUS);
	gtk_widget_set_size_request       (vol_pbar, 6, 0);
	gtk_widget_set_events(vol_pbar, GDK_BUTTON_PRESS_MASK);
	g_signal_connect(G_OBJECT(vol_pbar), "button_press_event",
			G_CALLBACK(vol_pbar_pressed), pd);

	rc =                              gtk_widget_get_modifier_style 
		(GTK_WIDGET (vol_pbar));

	if (!rc)
		rc = gtk_rc_style_new ();

	gdk_color_parse ("#00c000", &color);

	if (rc) {
		rc->color_flags[GTK_STATE_PRELIGHT] |= GTK_RC_BG;
		rc->bg[GTK_STATE_PRELIGHT] = color;
	}

	gtk_widget_modify_style (GTK_WIDGET (vol_pbar), rc);
	pd->vol_pbar                      = vol_pbar;

	gtk_box_pack_start             (GTK_BOX(boxMain), boxV, TRUE, TRUE, 1);
	gtk_box_pack_start             (GTK_BOX(boxMain), vol_pbar, FALSE, FALSE, 1);


	gtk_container_add              (GTK_CONTAINER(plugin), boxMain);

	xmms_plugin_read_config(pd);
	adjust_vol_pbar(pd);
	pd->pctrl = player_control_get_instance(pd->player);
	gtk_widget_show_all(boxMain);
	apply_visibility_settings(pd);

	pd->timeout                  = g_timeout_add(1000 / pd->scroll_speed, 						pbar_label_update, pd);

	return(TRUE);
}
