/*
 *  libghal
 *
 *  Copyright (c) 2007 Brian Tarricone <bjt23@cornell.edu>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 of the License ONLY.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <exo-hal/exo-hal.h>

#include "ghal-drive.h"
#include "ghal-context.h"
#include "ghal-private.h"

struct _GHalDrivePrivate
{
    LibHalDrive *hal_drive;
};

enum
{
    SIG_VOLUME_ADDED = 0,
    SIG_VOLUME_REMOVED,
    N_SIGS,
};

static void ghal_drive_class_init(GHalDriveClass *klass);

static void ghal_drive_init(GHalDrive *instance);
static void ghal_drive_finalize(GObject *obj);

static void ghal_drive_property_changed(GHalDevice *device,
                                        const gchar *property,
                                        GhalPropertyChangeType change_type);

guint ghd_sigs[N_SIGS] = { 0, };


G_DEFINE_TYPE(GHalDrive, ghal_drive, GHAL_TYPE_DEVICE)


static void
ghal_drive_class_init(GHalDriveClass *klass)
{
    GObjectClass *object_class = (GObjectClass *)klass;
    
    g_type_class_add_private(klass, sizeof(GHalDrivePrivate));
    
    object_class->finalize = ghal_drive_finalize;
    
    ghd_sigs[SIG_VOLUME_ADDED] = g_signal_new("volume-added",
                                              GHAL_TYPE_DRIVE,
                                              G_SIGNAL_RUN_LAST,
                                              G_STRUCT_OFFSET(GHalDriveClass,
                                                              volume_added),
                                              NULL, NULL,
                                              g_cclosure_marshal_VOID__OBJECT,
                                              G_TYPE_NONE, 1,
                                              GHAL_TYPE_VOLUME);
    ghd_sigs[SIG_VOLUME_REMOVED] = g_signal_new("volume-removed",
                                                GHAL_TYPE_DRIVE,
                                                G_SIGNAL_RUN_LAST,
                                                G_STRUCT_OFFSET(GHalDriveClass,
                                                                volume_removed),
                                                NULL, NULL,
                                                g_cclosure_marshal_VOID__OBJECT,
                                                G_TYPE_NONE, 1,
                                                GHAL_TYPE_VOLUME);
}

static void
ghal_drive_init(GHalDrive *drive)
{
    drive->priv = G_TYPE_INSTANCE_GET_PRIVATE(drive, GHAL_TYPE_DRIVE,
                                              GHalDrivePrivate);
    
    g_signal_connect(G_OBJECT(drive), "property-changed",
                     G_CALLBACK(ghal_drive_property_changed), NULL);
}

static void
ghal_drive_finalize(GObject *obj)
{
    GHalDrive *drive = GHAL_DRIVE(obj);
    
    if(drive->priv->hal_drive)
        libhal_drive_free(drive->priv->hal_drive);
    
    G_OBJECT_CLASS(ghal_drive_parent_class)->finalize(obj);
}



static void
ghal_drive_property_changed(GHalDevice *device,
                            const gchar *property,
                            GhalPropertyChangeType change_type)
{
    GHalDrive *drive = GHAL_DRIVE(device);
    
    /* ditch the old LibHalDrive, as it's no longer valid */
    if(G_LIKELY(drive->priv->hal_drive)) {
        libhal_drive_free(drive->priv->hal_drive);
        drive->priv->hal_drive = NULL;
    }
}

static gboolean
ghal_drive_ensure_hal_drive(GHalDrive *drive)
{
    g_return_val_if_fail(GHAL_IS_DRIVE(drive), FALSE);
    
    if(G_UNLIKELY(!drive->priv->hal_drive)) {
        GHalContext *context = NULL;
        const gchar *udi = NULL;
        LibHalContext *hal_ctx = NULL;
        
        g_return_val_if_fail((context = _ghal_device_peek_context(GHAL_DEVICE(drive)))
                             && (udi = ghal_device_peek_udi(GHAL_DEVICE(drive)))
                             && (hal_ctx = _ghal_context_peek_libhal_context(context)),
                             FALSE);
        
        drive->priv->hal_drive = libhal_drive_from_udi(hal_ctx, udi);
    }
    
    return drive->priv->hal_drive ? TRUE : FALSE;
}



void
_ghal_drive_volume_added(GHalDrive *drive,
                         GHalVolume *volume)
{
    g_signal_emit(G_OBJECT(drive), ghd_sigs[SIG_VOLUME_ADDED], 0, volume);
}

void
_ghal_drive_volume_removed(GHalDrive *drive,
                           GHalVolume *volume)
{
    g_signal_emit(G_OBJECT(drive), ghd_sigs[SIG_VOLUME_REMOVED], 0, volume);
}

LibHalDrive *
_ghal_drive_peek_libhal_drive(GHalDrive *drive)
{
    ghal_drive_ensure_hal_drive(drive);
    return drive->priv->hal_drive;
}



gboolean
ghal_drive_is_hotpluggable(GHalDrive *drive)
{
    g_return_val_if_fail(ghal_drive_ensure_hal_drive(drive), FALSE);
    
    return libhal_drive_is_hotpluggable(drive->priv->hal_drive);
}

gboolean
ghal_drive_uses_removable_media(GHalDrive *drive)
{
    g_return_val_if_fail(ghal_drive_ensure_hal_drive(drive), FALSE);
    
    return libhal_drive_uses_removable_media(drive->priv->hal_drive);
}

gboolean
ghal_drive_requires_eject(GHalDrive *drive)
{
    g_return_val_if_fail(ghal_drive_ensure_hal_drive(drive), FALSE);
    
    return libhal_drive_requires_eject(drive->priv->hal_drive);
}

LibHalDriveType
ghal_drive_get_drive_type(GHalDrive *drive)
{
    g_return_val_if_fail(ghal_drive_ensure_hal_drive(drive), -1);
    
    return libhal_drive_get_type(drive->priv->hal_drive);
}

LibHalDriveBus
ghal_drive_get_bus(GHalDrive *drive)
{
    g_return_val_if_fail(ghal_drive_ensure_hal_drive(drive), -1);
    
    return libhal_drive_get_bus(drive->priv->hal_drive);
}

LibHalDriveCdromCaps
ghal_drive_get_cdrom_caps(GHalDrive *drive)
{
    g_return_val_if_fail(ghal_drive_ensure_hal_drive(drive), 0);
    
    return libhal_drive_get_cdrom_caps(drive->priv->hal_drive);
}

const gchar *
ghal_drive_get_vendor(GHalDrive *drive)
{
    g_return_val_if_fail(ghal_drive_ensure_hal_drive(drive), NULL);
    
    return libhal_drive_get_vendor(drive->priv->hal_drive);
}

const gchar *
ghal_drive_get_model(GHalDrive *drive)
{
    g_return_val_if_fail(ghal_drive_ensure_hal_drive(drive), NULL);
    
    return libhal_drive_get_model(drive->priv->hal_drive);
}

const gchar *
ghal_drive_get_serial(GHalDrive *drive)
{
    g_return_val_if_fail(ghal_drive_ensure_hal_drive(drive), NULL);
    
    return libhal_drive_get_serial(drive->priv->hal_drive);
}

const gchar *
ghal_drive_get_firmware_version(GHalDrive *drive)
{
    g_return_val_if_fail(ghal_drive_ensure_hal_drive(drive), NULL);
    
    return libhal_drive_get_firmware_version(drive->priv->hal_drive);
}

GList *
ghal_drive_list_volumes(GHalDrive *drive)
{
    GHalContext *context = NULL;
    LibHalContext *hal_ctx = NULL;
    GList *volumes = NULL;
    char **udis;
    gint i, num_volumes = 0;
    
    g_return_val_if_fail(ghal_drive_ensure_hal_drive(drive)
                         && (context = _ghal_device_peek_context(GHAL_DEVICE(drive)))
                         && (hal_ctx = _ghal_context_peek_libhal_context(context)),
                         NULL);
    
    udis = libhal_drive_find_all_volumes(hal_ctx, drive->priv->hal_drive,
                                         &num_volumes);
    for(i = 0; i < num_volumes; ++i) {
        GHalDevice *device = ghal_context_get_device_for_udi(context, udis[i]);
        
        if(device)
            volumes = g_list_prepend(volumes, device);
        else
            g_printerr("Failed to get volume device for UDI %s\n", udis[i]);
    }
    
    libhal_free_string_array(udis);
    
    return g_list_reverse(volumes);
}

gchar *
ghal_drive_get_display_name(GHalDrive *drive)
{
    GHalContext *context = NULL;
    LibHalContext *hal_ctx = NULL;
    
    exo_hal_init();
    
    g_return_val_if_fail(ghal_drive_ensure_hal_drive(drive)
                         && (context = _ghal_device_peek_context(GHAL_DEVICE(drive)))
                         && (hal_ctx = _ghal_context_peek_libhal_context(context)),
                         NULL);
    
    return exo_hal_drive_compute_display_name(hal_ctx, drive->priv->hal_drive);
}

GList *
ghal_drive_get_icon_list(GHalDrive *drive)
{
    GHalContext *context = NULL;
    LibHalContext *hal_ctx = NULL;
    
    exo_hal_init();
    
    g_return_val_if_fail(ghal_drive_ensure_hal_drive(drive)
                         && (context = _ghal_device_peek_context(GHAL_DEVICE(drive)))
                         && (hal_ctx = _ghal_context_peek_libhal_context(context)),
                         NULL);
    
    return exo_hal_drive_compute_icon_list(hal_ctx, drive->priv->hal_drive);
}
