####################################
#   Burncenter
#   Copyright(c) 2000-2005 Alexandre Janon <alex14fr *at* yahoo *dot* fr>
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
##   
# Burncenter.pm
##

# Joey Schulze <joey *at* infodrom *dot* org> modified copyalltracks to copy only
# selected tracks and reworked configuration interface. 

package Burncenter;

#
# Configuration
#
$conf_global = "/etc/burncenter.conf";
$conf_local = $ENV{'HOME'}."/.burncenter";

## Ask for something

sub getline {
	my $question = shift;
	my $default = shift;

	print "$question ";
	print "[default=$default] " if $default ne '';

	$answer = <STDIN>;
	
	if($answer eq "\n") {
		$answer = $default;
	}
	chomp $answer;
	return $answer;
}

sub getlinebasedonconfig {
	my $question = shift;
	my $confvar = shift;
	my $default = shift;

	my $var = getconfig($confvar);
	if($var eq '') {
		return getline($question,$default);
	} else {
		return getline($question,$var);
	}
}

## Gets a config variable

sub getconfig {
	my $what = shift;

	open(CONFFILE, $conf_global);
	while(<CONFFILE>) {
		/^$what=(.+)/;
		if($1 ne '') {
			$ret = $1;
		}
	}
	close(CONFFILE);

	open(CONFFILE, $conf_local);
	while(<CONFFILE>) {
		/^$what=(.+)/;
		if($1 ne '') {
			$ret = $1;
		}
	}
	close(CONFFILE);

	return $ret;
}

## Writes configuration file

# Print a menu line for the reconfigure menu
#
# Arguments:
#  1 - id/nr
#  2 - default
#  3 - text 
sub print_conf_menu {
    printf "%2s. %s [%s]\n", $_[0], $_[2], $_[1];
}

sub reconfigure {
	my %config;
	my $save_and_exit = 0;
	if (-e $Burncenter::conf_global || -e $Burncenter::conf_local) {
		foreach my $var (qw/lang mkisofs cdrecord cdda2wav mpg123 ogg123
				 normalize cdreader cdwriter dev speed tempfiles
				 burndir/) {
	    		$config{$var} = Burncenter::getconfig($var);
		}
	} else {
		$config{lang} = "en";
		$config{cdreader} = "/dev/cdrom";
		$config{cdwriter} = "/dev/cdrom";
		$config{dev} = "/dev/cdrom";
		$config{speed} = "4";
		$config{tempfiles} = "/tmp";
		$config{burndir} = "/tmp/ToBurn";

		foreach my $var (qw/mkisofs cdrecord cdda2wav mpg123 ogg123 normalize/) {
		    $config{$var} = $var;
		}
  	  }

	while (!$save_and_exit) {
		print "Burncenter configuration --\n\n";
		print_conf_menu( 1, $config{mkisofs}, "Path to mkisofs");
		print_conf_menu( 2, $config{cdrecord}, "Path to cdrecord");
		print_conf_menu( 3, $config{cdda2wav}, "Path to cdda2wav");
		print_conf_menu( 4, $config{mpg123}, "Path to mpg123");
		print_conf_menu( 5, $config{ogg123}, "Path to ogg123");
		print_conf_menu( 6, $config{normalize}, "Path to normalize");
		print_conf_menu( 7, $config{cdreader}, "CD reader");
		print_conf_menu( 8, $config{cdwriter}, "CD writer");
		print_conf_menu( 9, $config{speed}, "Speed to use");
		print_conf_menu(10, $config{tempfiles}, "Temporary directory");
		print_conf_menu(11, $config{burndir}, "Burn source directory");
		print " Q. Save changes\n";
		print "\n";
		$choice = Burncenter::getline("Your choice? ");

		if ($choice =~ /^[qx]$/i) { $save_and_exit = 1 }
		elsif ($choice =~ /^1$/) {
	  		$config{mkisofs} = Burncenter::getlinebasedonconfig("Location of mkisofs?",
									"mkisofs", $config{mkisofs});
		} elsif ($choice =~ /^2$/) {
	    		$config{cdrecord} = Burncenter::getlinebasedonconfig("Location of cdrecord?",
									 "cdrecord", $config{cdrecord});
		} elsif ($choice =~ /^3$/) {
	 		$config{cdda2wav} = Burncenter::getlinebasedonconfig("Location of cdda2wav?",
									 "cdda2wav", $config{cdda2wav});
		} elsif ($choice =~ /^4$/) {
			$config{mpg123} = Burncenter::getlinebasedonconfig("Location of mpg123?",
								       "mpg123", $config{mpg123});
		} elsif ($choice =~ /^5$/) {
   			 $config{ogg123} = Burncenter::getlinebasedonconfig("Location of ogg123?",
							 	      "ogg123", $config{ogg123});
		} elsif ($choice =~ /^6$/) {
	  	  	$config{normalize} = Burncenter::getlinebasedonconfig("Location of normalize?",
									  "normalize", $config{normalize});
		} elsif ($choice =~ /^7$/) {
		    	$config{cdreader} = Burncenter::getlinebasedonconfig("Location of your CD-Reader?",
									 "cdreader",$config{cdreader});
		} elsif ($choice =~ /^8$/) {
		    	$config{cdwriter} = Burncenter::getlinebasedonconfig("Location of your CD-Recorder?",
									 "cdwriter",$config{cdwriter});
		} elsif ($choice =~ /^9$/) {
	   		$config{speed}=Burncenter::getlinebasedonconfig("Which speed should your CD-Writer use?",
							   		 "speed",$config{speed});
		} elsif ($choice =~ /^10$/) {
	    		$config{tempfiles}=Burncenter::getlinebasedonconfig("Location of the temporary files?",
								         "tempfiles",$config{tempfiles});
		} elsif ($choice =~ /^11$/) {
	    		$config{burndir}=Burncenter::getlinebasedonconfig("What will be the default path for files to burn?",
								      "burndir",$config{burndir});
		}
	}
 	print "Configuration done !\n Writing configuration file: ";
	if ($> == 0 && ! -e $Burncenter::conf_local) {
		$conffile = $Burncenter::conf_global;
	} else {
		$conffile = $Burncenter::conf_local;
	}
	print "$conffile\n";
	open(CONF, ">$conffile");
	$config{dev}=$config{cdwriter};
	foreach my $var (qw/lang mkisofs cdrecord cdda2wav mpg123 ogg123
			 normalize cdreader cdwriter dev speed tempfiles
			 burndir/) {
		printf CONF "%s=%s\n", $var, $config{$var};
	}
	close CONF;
}

## Makes an image of the CD currently in the cdreader and returns the name of the image

sub makecdimage {
	my $tmpfiles=getconfig("tempfiles");
	my $name = "$tmpfiles/cdimage.$$";
	$cdreader=getconfig("cdreader");
	system("cp $cdreader $name");
	return $name;
}	

## Copies desired tracks on an audio CD to a directory, returns the name of this dir

sub copyalltracks {
	my $dirname=getconfig("tempfiles")."/audiofiles.$$";
	mkdir $dirname;
	my $cdda2wav=getconfig("cdda2wav");
	my $dev=getconfig("cdreader");
	if ($_[0]) {
	    # Should understand the following
	    # 1       1-5    1 - 5         1-4,6         1-4 6     1 - 4 7
	    my $args = $_[0];
	    $args =~ s/(\d)\s+-\s+(\d)/$1-$2/g;
	    $args =~ s/,/ /g;
	    $args =~ s/\s+/ /g;
	    foreach my $part (split(/ /, $args)) {
		if ($part =~ /-/) {
		    $part =~ s/-/+/;
		    system("cd $dirname ; $cdda2wav -t$part -B -D$dev ; rm -f *.inf ; cd -");
		} else {
		    system("cd $dirname ; $cdda2wav -t$part+$part -B -D$dev ; rm -f *.inf ; cd -");
		    rename ("$dirname/audio.wav", sprintf ("$dirname/audio_%02d.wav", $part))
			if (-e "$dirname/audio.wav");
		}
	    }

	} else {
	    system("cd $dirname ; $cdda2wav -t1 -B -D$dev ; rm -f *.inf ; cd -");
	}

	return $dirname;
}

## Converts an entire directory of MP3s or OGGs to WAVes files, and then normalize them

sub mp3towav {
	my $dirname=shift;

	my $mpg123=getconfig("mpg123");
	my $ogg123=getconfig("ogg123");
	my $normalize=getconfig("normalize");
	opendir(DIR,$dirname);
	@mp3files = grep(/\.mp3$/i,readdir(DIR));
	for $file(@mp3files) {
		system("cd $dirname ; $mpg123 -r44100 --stereo -w '$file.wav' '$file' ; cd -");
	}
	closedir(DIR);
	opendir(DIR,$dirname);
	@oggfiles = grep(/\.ogg$/i,readdir(DIR));
	for $file(@oggfiles) {
		system("cd $dirname ; $ogg123 -d wav -f '$file.wav' '$file' ; cd -");
	}
	closedir(DIR);
	system("cd $dirname ; $normalize -m *.wav ; cd -"); 
}

## Burns an image on CD. It is possible to specify additionnal cdrecord flags (eg -multi)

sub burncdimage {
	my $image = shift;
	my $cdrecordflags = shift;

	my $cdrecord=getconfig("cdrecord");
	my $speed=getconfig("speed");
	my $dev=getconfig("dev");

	system("$cdrecord -eject -v speed=$speed dev=$dev $cdrecordflags $image");
}

## Burns all the WAVes files in a given directory

sub burnwavfiles {
	my $directory = shift;

	my $cdrecord=getconfig("cdrecord");
	my $dev=getconfig("dev");
	my $speed=getconfig("speed");

       	system("$cdrecord -eject -v speed=$speed dev=$dev -pad -audio $directory/*.wav");
}

## Blanks a CDRW 

sub blankcdrw {
	my $blank = shift;

	my $cdrecord=getconfig("cdrecord");
	my $speed=getconfig("speed");
	my $dev=getconfig("dev");

	system("$cdrecord -eject -v speed=$speed dev=$dev blank=$blank");
}

## Creates a (first-session) Rockridge/Joliet ISO image of a given directory, and returns the name of this image

sub makefirstisoimage {
	my $directory = shift;

	my $name=getconfig("tempfiles")."/isoimage.$$";
	my $mkisofs=getconfig("mkisofs");

	system("$mkisofs -v -o $name -J -R $directory");

	return $name;
}

## Same as makefirstisoimage, except we import the precedent sessions first 

sub makesecondisoimage {
	my $directory = shift;

	my $name=getconfig("tempfiles")."/isoimage.$$";
	my $cdrecord=getconfig("cdrecord");
	my $dev=getconfig("dev");
	my $cdwriter=getconfig("cdwriter");
	my $mkisofs=getconfig("mkisofs");

	my $msinfo=`$cdrecord dev=$dev -msinfo`;
	chomp $msinfo;
	system("$mkisofs -v -o $name -J -R -M$cdwriter -C$msinfo $directory");

	return $name;
}

## Check if the disc has other sessions written. Returns 0 if no other session
## was written, and 1 if other sessions have been written.

sub checkothersessions {
	my $cdrecord=getconfig("cdrecord");
	my $dev=getconfig("dev");

	my $msinfo=`$cdrecord dev=$dev -msinfo 2>/dev/null`;

	return 0 if $msinfo eq '';
	return 1 if $msinfo ne '';
}

## Copy a given playlist to a directory; returns the name of this directory

sub playlist2dir {
	my $playlist = shift;

	my $dirname = getconfig("tempfiles")."/playlist.$$";
	mkdir $dirname;
	open(PLAYLIST, $playlist) || return "";
	while(<PLAYLIST>) {
		next if /^#/;
		next if ($_ eq '');
		chomp;
		system("cp $_ $dirname");
	}

	return $dirname;
}

1;
