/*
 *      Common routines for tag and audio information retrieval.
 *
 *      Copyright (c) 2005-2007 Naoaki Okazaki
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA, or visit
 * http://www.gnu.org/copyleft/gpl.html .
 *
 */

/* $Id: gmi.c 328 2007-02-10 17:50:11Z nyaochi $ */

#ifdef	HAVE_CONFIG_H
#include <config.h>
#endif/*HAVE_CONFIG_H*/

#include <os.h>
#include <pmplib/ucs2char.h>
#include <pmplib/filepath.h>
#include <gmi.h>

/**
 * \addtogroup gmi
 * @{
 */

int gmi_mp3(media_info_t* info, const ucs2char_t *filename, const char *charset);
static const ucs2char_t ucs2cs_ext_mp3[] = {'.','m','p','3',0};

int gmi_wma(media_info_t* info, const ucs2char_t *filename, const char *charset);
static const ucs2char_t ucs2cs_ext_wma[] = {'.','w','m','a',0};

int gmi_vorbis(media_info_t* info, const ucs2char_t *filename, const char *charset);
static const ucs2char_t ucs2cs_ext_ogg[] = {'.','o','g','g',0};

int gmi_wav(media_info_t* info, const ucs2char_t *filename, const char *charset);
static const ucs2char_t ucs2cs_ext_wav[] = {'.','w','a','v',0};

typedef struct {
	int (*func)(media_info_t* info, const ucs2char_t *filename, const char *charset);
	const ucs2char_t* ext;
} gmi_exports_t;

static gmi_exports_t gmi_exports[] = {
	{gmi_mp3, ucs2cs_ext_mp3},
	{gmi_wma, ucs2cs_ext_wma},
	{gmi_vorbis, ucs2cs_ext_ogg},
	{gmi_wav, ucs2cs_ext_wav},
	{NULL, NULL},
};


static void replace_slash(ucs2char_t* path, ucs2char_t c)
{
	while (*path) {
		if (*path == PATHCHAR) {
			*path = c;
		}
		path++;
	}
}


static int gmi_get_from_pathname(
	media_info_t* info,
	const ucs2char_t *filename,
	const ucs2char_t* path_to_music,
	int flag
	)
{
	const ucs2char_t *p = NULL, *q = NULL;
	ucs2char_t file[MAX_PATH], path[MAX_PATH];

	ucs2cpy(file, filepath_skippath(filename));
	ucs2cpy(path, filename);
	filepath_remove_filespec(path);

	if (flag & GMIF_PARSEMASK) {
		/* Clear fields according to the type. */
		if (flag & GMIF_TITLE) {
			ucs2free(info->title);	info->title = NULL;
		}
		if (flag & GMIF_ALBUM) {
			ucs2free(info->album);	info->album = NULL;
		}
		if (flag & GMIF_GENRE) {
			ucs2free(info->genre);	info->genre = NULL;
		}
		if (flag & GMIF_ARTIST) {
			ucs2free(info->artist);	info->artist = NULL;
		}

		/* Obtain path name to the music. */
		p = ucs2str(path, path_to_music);
		if (p) {
			p += ucs2len(path_to_music);
		} else {
			p = path;
		}

		/* Obtain genre, artist, and album values from the path if necessary. */
		if (flag & GMIF_GENRE) {
			q = filepath_skipadirectory(p);
			if (q) {
				info->genre = ucs2ndup(p, (q-p) - 1);
				p = q;
				q = filepath_skipadirectory(p);
				if (q) {
					info->artist = ucs2ndup(p, (q-p) - 1);
					if (flag & GMIF_ALBUM) {
						info->album = ucs2dup(q);
						filepath_removeslash(info->album);
						replace_slash(info->album, (ucs2char_t)' ');
					}
				}
			}
		} else  {
			q = filepath_skipadirectory(p);
			if (q) {
				info->artist = ucs2ndup(p, (q-p) - 1);
				if (flag & GMIF_ALBUM) {
					info->album = ucs2dup(q);
					filepath_removeslash(info->album);
					replace_slash(info->album, (ucs2char_t)' ');
				}
			}
		}

		if (flag & GMIF_NUMBER) {
			/* Use filename as a title and track number. */
			info->track_number = 0;
			p = file;
			if (0x0030 <= *p && *p <= 0x0039) {
				p++;
				if (0x0030 <= *p && *p <= 0x0039) {
					/* The filename begins with two numerical letters. */
					p++;

					/* Obtain track number. */
					info->track_number = (uint32_t)ucs2toi(file);

					if (flag & GMIF_TITLE) {
						/* Skip space and '_' letters. */
						while (*p == 0x0020 || *p == 0x005F) {
							p++;
						}

						/* Obtain track title. */
						info->title = ucs2dup(p);
						filepath_remove_extension(info->title);
						if (!info->title[0]) {
							/* If track title becomes empty (e.g., 01.mp3), set the filename. */
							ucs2free(info->title);
							info->title = ucs2dup(file);
							filepath_remove_extension(info->title);
						}
					}
				} else {
					/* The filename begins with one numerical letter. */
					if (flag & GMIF_TITLE) {
						info->title = ucs2dup(file);
						filepath_remove_extension(info->title);
					}
				}
			} else {
				/* The filename does not begin with a numerical letter. */
				if (flag & GMIF_TITLE) {
					info->title = ucs2dup(file);
					filepath_remove_extension(info->title);
				}
			}
		} else {
			if (flag & GMIF_TITLE) {
				/* Use filename as a title. */
				info->title = ucs2dup(file);
				filepath_remove_extension(info->title);
			}
		}
	}

	return 0;
}

static void strip_head_letters(ucs2char_t* str, size_t offset)
{
	if (offset <= ucs2len(str)) {
		size_t i, n = ucs2len(str) + 1 - offset;
		for (i = 0;i < n;++i) {
			str[i] = str[i+offset];
		}
	}
}

int gmi_get(
	media_info_t* info,
	const ucs2char_t *filename,
	const ucs2char_t* path_to_music,
	int flag,
	const char *charset,
	ucs2char_t **strip_words,
	int num_strip_words
	)
{
	int ret = 1;
	gmi_exports_t* exp = gmi_exports;

	/* Set the pathname and filename. */
	info->filename = ucs2dup(filename);
	info->filesize = filepath_size(filename);

	if (flag & GMIF_TAG) {
		while (exp->func) {
			if (filepath_hasext(filename, exp->ext)) {
				if (exp->func(info, filename, charset) == 0) {
					ret = 0;
					break;
				}
			}
			exp++;
		}
	}

	if (flag & GMIF_PARSEMASK) {
		gmi_get_from_pathname(info, filename, path_to_music, flag);
	}

	/* Strip specific words in artist names. */
	if ((flag & GMIF_STRIP_ARTIST) && strip_words && info->artist) {
		int i;
		for (i = 0;i < num_strip_words;++i) {

			/* If the artist name doesn't start with this
			   word, then ignore it. */
			size_t artist_length = ucs2len(info->artist);
			size_t length = ucs2len(strip_words[i]);
			if (ucs2incmp(info->artist, strip_words[i], length) != 0) {
				break;
			}
			/* OK. The artist name starts with this word.
			   Suppose the 'strip word' is 'the' - we
			   don't want to remove it if there's no
			   subsequent space.  For example, we don't
			   want to strip 'The' from 'Therapy?'.

			   So we need to check the next character in
			   the artist name for a space, remembering
			   that there might not actually be a next
			   character...
			 */
			if (artist_length == length) {
				/* Artist name is (exactly) the word
				   to be stripped */
				info->artist[0] = 0;
			} else if (artist_length > length && info->artist[length] == ' ') {
				/* Remove the subsequent space too. */
				strip_head_letters(info->artist, length + 1);
			}
		}
	}

	return ret;
}

/** @} */
