/*
 *      Generic exports for idx.
 *
 *      Copyright (c) 2005-2007 Naoaki Okazaki
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 *
 */

/* $Id: idx_exports.c 328 2007-02-10 17:50:11Z nyaochi $ */

#ifdef	HAVE_CONFIG_H
#include <config.h>
#endif/*HAVE_CONFIG_H*/

#include <os.h>
#include <stdio.h>
#include <stdlib.h>
#include <pmplib/ucs2char.h>

#include "ip2db.h"

#define	COMP(a, b)	((a)>(b))-((a)<(b))



/*
 * Implementation of idx_export_t::get_num_childlen.
 */
static void idx_setget_num_children_none(uint8_t* block, uint16_t* value, int is_storing)
{
}

static void idx_setget_num_children_dword_unique(uint8_t* block, uint16_t* value, int is_storing)
{
	idxnode_setget_num_children(block, value, sizeof(uint32_t) * 2, is_storing);
}

static void idx_setget_num_children_dword(uint8_t* block, uint16_t* value, int is_storing)
{
	idxnode_setget_num_children(block, value, sizeof(uint32_t) * 3, is_storing);
}

static void idx_setget_num_children_str8(uint8_t* block, uint16_t* value, int is_storing)
{
	idxnode_setget_num_children(block, value, sizeof(uint32_t) * 2 + sizeof(ucs2char_t) * 8, is_storing);
}

static void idx_setget_num_children_str16(uint8_t* block, uint16_t* value, int is_storing)
{
	idxnode_setget_num_children(block, value, sizeof(uint32_t) * 2 + sizeof(ucs2char_t) * 16, is_storing);
}

static void idx_setget_num_children_str24(uint8_t* block, uint16_t* value, int is_storing)
{
	idxnode_setget_num_children(block, value, sizeof(uint32_t) * 2 + sizeof(ucs2char_t) * 24, is_storing);
}


/*
 * Implementation of idx_export_t::get_depth.
 */
static uint8_t idx_get_height_none(uint8_t* block)
{
	return 0;
}

static uint8_t idx_get_height(uint8_t* block)
{
	uint8_t height;
	idxnode_setget_height(block, &height, 0);
	return height;
}


/*
 * Implementation of idx_export_t::record_to_key.
 */
static void idx_record_to_key_none(idx_key_t* dst, const ip2db_record_t* src)
{
}

static void idx_record_to_key_entrynumber(idx_key_t* dst, const ip2db_record_t* src)
{
	dst->type = IP2DBIDX_PAGE_NODE_ENTRYNUMBER;
	dst->dup = 0;
	dst->data.dword = src->entry_number;
}

static void idx_record_to_key_filename(idx_key_t* dst, const ip2db_record_t* src)
{
	dst->type = IP2DBIDX_PAGE_NODE_FILENAME;
	dst->dup = 0;
	ucs2ncpy(dst->data.str, src->filename, 8);
}

static void idx_record_to_key_title(idx_key_t* dst, const ip2db_record_t* src)
{
	dst->type = IP2DBIDX_PAGE_NODE_TITLE;
	dst->dup = 0;
	ucs2ncpy(dst->data.str, src->title, 8);
}

static void idx_record_to_key_artist(idx_key_t* dst, const ip2db_record_t* src)
{
	dst->type = IP2DBIDX_PAGE_NODE_ARTIST;
	dst->dup = 0;
	ucs2ncpy(dst->data.str, src->artist, 8);
}

static void idx_record_to_key_album(idx_key_t* dst, const ip2db_record_t* src)
{
	dst->type = IP2DBIDX_PAGE_NODE_ALBUM;
	dst->dup = 0;
	ucs2ncpy(dst->data.str, src->album, 8);
}

static void idx_record_to_key_genre(idx_key_t* dst, const ip2db_record_t* src)
{
	dst->type = IP2DBIDX_PAGE_NODE_GENRE;
	dst->dup = 0;
	ucs2ncpy(dst->data.str, src->genre, 8);
}

static void idx_record_to_key_genre_artist(idx_key_t* dst, const ip2db_record_t* src)
{
	dst->type = IP2DBIDX_PAGE_NODE_GENRE_ARTIST;
	dst->dup = 0;
	ucs2ncpy(dst->data.str, src->genre, 8);
	ucs2ncpy(dst->data.str+8, src->artist, 8);
}

static void idx_record_to_key_genre_album(idx_key_t* dst, const ip2db_record_t* src)
{
	dst->type = IP2DBIDX_PAGE_NODE_GENRE_ALBUM;
	dst->dup = 0;
	ucs2ncpy(dst->data.str, src->genre, 8);
	ucs2ncpy(dst->data.str+8, src->album, 8);
}

static void idx_record_to_key_genre_artist_album(idx_key_t* dst, const ip2db_record_t* src)
{
	dst->type = IP2DBIDX_PAGE_NODE_GENRE_ARTIST_ALBUM;
	dst->dup = 0;
	ucs2ncpy(dst->data.str, src->genre, 8);
	ucs2ncpy(dst->data.str+8, src->artist, 8);
	ucs2ncpy(dst->data.str+16, src->album, 8);
}

static void idx_record_to_key_artist_album(idx_key_t* dst, const ip2db_record_t* src)
{
	dst->type = IP2DBIDX_PAGE_NODE_ARTIST_ALBUM;
	dst->dup = 0;
	ucs2ncpy(dst->data.str, src->artist, 8);
	ucs2ncpy(dst->data.str+8, src->album, 8);
}

static void idx_record_to_key_rating(idx_key_t* dst, const ip2db_record_t* src)
{
	dst->type = IP2DBIDX_PAGE_NODE_RATING;
	dst->dup = 0;
	dst->data.dword = src->rating;
}

static void idx_record_to_key_playcount(idx_key_t* dst, const ip2db_record_t* src)
{
	dst->type = IP2DBIDX_PAGE_NODE_PLAYCOUNT;
	dst->dup = 0;
	dst->data.dword = src->play_count;
}

static void idx_record_to_key_recentplay(idx_key_t* dst, const ip2db_record_t* src)
{
	dst->type = IP2DBIDX_PAGE_NODE_RECENTPLAY;
	dst->dup = 0;
	dst->data.dword = src->recent_play;
}

static void idx_record_to_key_format(idx_key_t* dst, const ip2db_record_t* src)
{
	dst->type = IP2DBIDX_PAGE_NODE_FORMAT;
	dst->dup = 0;
	dst->data.dword = src->format;
}






/*
 * Implementation of idx_export_t::get_item.
 */
static void idx_setget_child_none(uint8_t* block, int i, idx_key_t* key, uint32_t* child, int is_storing)
{
}

static void idx_setget_child_dword_unique(uint8_t* block, int i, idx_key_t* key, uint32_t* child, int is_storing)
{
	key->dup = 0;
	idxnode_setget_dword(block, i, &key->data.dword, NULL, child, is_storing);
}

static void idx_setget_child_dword(uint8_t* block, int i, idx_key_t* key, uint32_t* child, int is_storing)
{
	idxnode_setget_dword(block, i, &key->data.dword, &key->dup, child, is_storing);
}

static void idx_setget_child_str8(uint8_t* block, int i, idx_key_t* key, uint32_t* child, int is_storing)
{
	idxnode_setget_str(block, i, key->data.str, 8, &key->dup, child, is_storing);
}

static void idx_setget_child_str16(uint8_t* block, int i, idx_key_t* key, uint32_t* child, int is_storing)
{
	idxnode_setget_str(block, i, key->data.str, 16, &key->dup, child, is_storing);
}

static void idx_setget_child_str24(uint8_t* block, int i, idx_key_t* key, uint32_t* child, int is_storing)
{
	idxnode_setget_str(block, i, key->data.str, 24, &key->dup, child, is_storing);
}




/*
 * Implementation of idx_export_t::comp_idxkey.
 */
static int idx_comp_idxkey_none(const idx_key_t* x, const idx_key_t* y)
{
	return 0;
}

static int idx_comp_idxkey_dword_unique(const idx_key_t* x, const idx_key_t* y)
{
	return COMP(x->data.dword, y->data.dword);
}

static int idx_comp_idxkey_dword(const idx_key_t* x, const idx_key_t* y)
{
	if (x->data.dword == y->data.dword) {
		return COMP(x->dup, y->dup);
	} else {
		return COMP(x->data.dword, y->data.dword);
	}
}

int ucs2cmp_fixed(const ucs2char_t* x, const ucs2char_t* y, size_t n)
{
	size_t i;

	for (i = 0;i < n;i++) {
		if (*x != *y) {
			break;
		}
		x++;
		y++;
	}
	return i == n ? 0 : COMP(*x, *y);
}

static int idx_comp_idxkey_str8(const idx_key_t* x, const idx_key_t* y)
{
	int ret = 0;
	ret = ucs2cmp_fixed(x->data.str, y->data.str, 8);
	return (ret == 0) ? COMP(x->dup, y->dup) : ret;
}

static int idx_comp_idxkey_str16(const idx_key_t* x, const idx_key_t* y)
{
	int ret = 0;
	ret = ucs2cmp_fixed(x->data.str, y->data.str, 16);
	return (ret == 0) ? COMP(x->dup, y->dup) : ret;
}

static int idx_comp_idxkey_str24(const idx_key_t* x, const idx_key_t* y)
{
	int ret = 0;
	ret = ucs2cmp_fixed(x->data.str, y->data.str, 24);
	return (ret == 0) ? COMP(x->dup, y->dup) : ret;
}


/*
 * Implementation of idx_export_t::dump.
 */
static void idx_dump_header(uint8_t* block, uint32_t offset, FILE *fp)
{
	idxheader_repr(block, offset, fp);
}

static void idx_dump_descriptor(uint8_t* block, uint32_t offset, FILE *fp)
{
	idxdescriptor_repr(block, offset, fp);
}

static void idx_dump_dword_unique(uint8_t* block, uint32_t offset, FILE *fp)
{
	idxnode_repr_dword(block, 0, offset, fp);
}

static void idx_dump_dword(uint8_t* block, uint32_t offset, FILE *fp)
{
	idxnode_repr_dword(block, 1, offset, fp);
}

static void idx_dump_str8(uint8_t* block, uint32_t offset, FILE *fp)
{
	idxnode_repr_str(block, 8, offset, fp);
}

static void idx_dump_str16(uint8_t* block, uint32_t offset, FILE *fp)
{
	idxnode_repr_str(block, 16, offset, fp);
}

static void idx_dump_str24(uint8_t* block, uint32_t offset, FILE *fp)
{
	idxnode_repr_str(block, 24, offset, fp);
}

static void idx_dump_leaf(uint8_t* block, uint32_t offset, FILE *fp)
{
	idxleaf_repr(block, offset, fp);
}

static int idx_comp_none(const sortitem_t *_x, const sortitem_t *_y)
{
	return 0;
}

static int idx_comp_entrynumber(const sortitem_t *_x, const sortitem_t *_y)
{
	const ip2db_record_t* x = &_x->records[_x->index];
	const ip2db_record_t* y = &_y->records[_y->index];
	return COMP(x->entry_number, y->entry_number);
}

static int idx_comp_filename(const sortitem_t *_x, const sortitem_t *_y)
{
	const ip2db_record_t* x = &_x->records[_x->index];
	const ip2db_record_t* y = &_y->records[_y->index];
	int ret = ucs2cmp(x->filename, y->filename);
	if (ret == 0) {
		ret = COMP(_x->index, _y->index);
	}
	return ret;
}

static int idx_comp_title(const sortitem_t *_x, const sortitem_t *_y)
{
	const ip2db_record_t* x = &_x->records[_x->index];
	const ip2db_record_t* y = &_y->records[_y->index];
	int ret = ucs2cmp(x->title, y->title);
	if (ret == 0) {
		ret = COMP(_x->index, _y->index);
	}
	return ret;
}

static int idx_comp_artist(const sortitem_t *_x, const sortitem_t *_y)
{
	const ip2db_record_t* x = &_x->records[_x->index];
	const ip2db_record_t* y = &_y->records[_y->index];
	int ret = ucs2cmp(x->artist, y->artist);
	if (ret == 0) {
		ret = COMP(_x->index, _y->index);
	}
	return ret;
}

static int idx_comp_album(const sortitem_t *_x, const sortitem_t *_y)
{
	const ip2db_record_t* x = &_x->records[_x->index];
	const ip2db_record_t* y = &_y->records[_y->index];
	int ret = ucs2cmp(x->album, y->album);
	if (ret == 0) {
		ret = COMP(_x->index, _y->index);
	}
	return ret;
}

static int idx_comp_genre(const sortitem_t *_x, const sortitem_t *_y)
{
	const ip2db_record_t* x = &_x->records[_x->index];
	const ip2db_record_t* y = &_y->records[_y->index];
	int ret = ucs2cmp(x->genre, y->genre);
	if (ret == 0) {
		ret = COMP(_x->index, _y->index);
	}
	return ret;
}

static int idx_comp_genre_artist(const sortitem_t *_x, const sortitem_t *_y)
{
	const ip2db_record_t* x = &_x->records[_x->index];
	const ip2db_record_t* y = &_y->records[_y->index];
	int ret = ucs2ncmp(x->genre, y->genre, 8);
	if (ret == 0) {
		ret = idx_comp_artist(_x, _y);
	}
	return ret;
}

static int idx_comp_genre_album(const sortitem_t *_x, const sortitem_t *_y)
{
	const ip2db_record_t* x = &_x->records[_x->index];
	const ip2db_record_t* y = &_y->records[_y->index];
	int ret = ucs2ncmp(x->genre, y->genre, 8);
	if (ret == 0) {
		ret = idx_comp_album(_x, _y);
	}
	return ret;
}

static int idx_comp_genre_artist_album(const sortitem_t *_x, const sortitem_t *_y)
{
	const ip2db_record_t* x = &_x->records[_x->index];
	const ip2db_record_t* y = &_y->records[_y->index];
	int ret = ucs2ncmp(x->genre, y->genre, 8);
	if (ret == 0) {
		ret = ucs2ncmp(x->artist, y->artist, 8);
	}
	if (ret == 0) {
		ret = idx_comp_album(_x, _y);
	}
	return ret;
}

static int idx_comp_artist_album(const sortitem_t *_x, const sortitem_t *_y)
{
	const ip2db_record_t* x = &_x->records[_x->index];
	const ip2db_record_t* y = &_y->records[_y->index];
	int ret = ucs2ncmp(x->artist, y->artist, 8);
	if (ret == 0) {
		ret = idx_comp_album(_x, _y);
	}
	return ret;
}

static int idx_comp_rating(const sortitem_t *_x, const sortitem_t *_y)
{
	const ip2db_record_t* x = &_x->records[_x->index];
	const ip2db_record_t* y = &_y->records[_y->index];
	int ret = COMP(x->rating, y->rating);
	if (ret == 0) {
		ret = COMP(_x->index, _y->index);
	}
	return ret;
}

static int idx_comp_playcount(const sortitem_t *_x, const sortitem_t *_y)
{
	const ip2db_record_t* x = &_x->records[_x->index];
	const ip2db_record_t* y = &_y->records[_y->index];
	int ret = COMP(x->play_count, y->play_count);
	if (ret == 0) {
		ret = COMP(_x->index, _y->index);
	}
	return ret;
}

static int idx_comp_recentplay(const sortitem_t *_x, const sortitem_t *_y)
{
	const ip2db_record_t* x = &_x->records[_x->index];
	const ip2db_record_t* y = &_y->records[_y->index];
	int ret = COMP(x->recent_play, y->recent_play);
	if (ret == 0) {
		ret = COMP(_x->index, _y->index);
	}
	return ret;
}

static int idx_comp_format(const sortitem_t *_x, const sortitem_t *_y)
{
	const ip2db_record_t* x = &_x->records[_x->index];
	const ip2db_record_t* y = &_y->records[_y->index];
	int ret = COMP(x->format, y->format);
	if (ret == 0) {
		ret = COMP(_x->index, _y->index);
	}
	return ret;
}

idx_export_t g_idx_exports[IP2DBIDX_PAGE_LEAF+1] = {
	{	/* @0 */
		NULL,
		NULL,
		NULL,
		NULL,
		NULL,
		NULL,
		NULL,
		NULL,
	},
	{	/* @1 */
		idxheader_init,
		idx_setget_num_children_none,
		idx_get_height_none,
		idx_record_to_key_none,
		idx_setget_child_none,
		idx_comp_idxkey_none,
		idx_dump_header,
		idx_comp_none,
	},
	{	/* @2 */
		idxdescriptor_init,
		idx_setget_num_children_none,
		idx_get_height_none,
		idx_record_to_key_none,
		idx_setget_child_none,
		idx_comp_idxkey_none,
		idx_dump_descriptor,
		idx_comp_none,
	},
	{	/* @3 */
		idxnode_init,
		idx_setget_num_children_dword_unique,
		idx_get_height,
		idx_record_to_key_entrynumber,
		idx_setget_child_dword_unique,
		idx_comp_idxkey_dword_unique,
		idx_dump_dword_unique,
		idx_comp_entrynumber,
	},
	{	/* @4 */
		idxnode_init,
		idx_setget_num_children_str8,
		idx_get_height,
		idx_record_to_key_filename,
		idx_setget_child_str8,
		idx_comp_idxkey_str8,
		idx_dump_str8,
		idx_comp_filename,
	},
	{	/* @5 */
		idxnode_init,
		idx_setget_num_children_str8,
		idx_get_height,
		idx_record_to_key_title,
		idx_setget_child_str8,
		idx_comp_idxkey_str8,
		idx_dump_str8,
		idx_comp_title,
	},
	{	/* @6 */
		idxnode_init,
		idx_setget_num_children_str8,
		idx_get_height,
		idx_record_to_key_artist,
		idx_setget_child_str8,
		idx_comp_idxkey_str8,
		idx_dump_str8,
		idx_comp_artist,
	},
	{	/* @7 */
		idxnode_init,
		idx_setget_num_children_str8,
		idx_get_height,
		idx_record_to_key_album,
		idx_setget_child_str8,
		idx_comp_idxkey_str8,
		idx_dump_str8,
		idx_comp_album,
	},
	{	/* @8 */
		idxnode_init,
		idx_setget_num_children_str8,
		idx_get_height,
		idx_record_to_key_genre,
		idx_setget_child_str8,
		idx_comp_idxkey_str8,
		idx_dump_str8,
		idx_comp_genre,
	},
	{	/* @9 */
		idxnode_init,
		idx_setget_num_children_str16,
		idx_get_height,
		idx_record_to_key_genre_artist,
		idx_setget_child_str16,
		idx_comp_idxkey_str16,
		idx_dump_str16,
		idx_comp_genre_artist,
	},
	{	/* @10 */
		idxnode_init,
		idx_setget_num_children_str16,
		idx_get_height,
		idx_record_to_key_genre_album,
		idx_setget_child_str16,
		idx_comp_idxkey_str16,
		idx_dump_str16,
		idx_comp_genre_album,
	},
	{	/* @11 */
		idxnode_init,
		idx_setget_num_children_str24,
		idx_get_height,
		idx_record_to_key_genre_artist_album,
		idx_setget_child_str24,
		idx_comp_idxkey_str24,
		idx_dump_str24,
		idx_comp_genre_artist_album,
	},
	{	/* @12 */
		idxnode_init,
		idx_setget_num_children_str16,
		idx_get_height,
		idx_record_to_key_artist_album,
		idx_setget_child_str16,
		idx_comp_idxkey_str16,
		idx_dump_str16,
		idx_comp_artist_album,
	},
	{	/* @13 */
		idxnode_init,
		idx_setget_num_children_dword,
		idx_get_height,
		idx_record_to_key_rating,
		idx_setget_child_dword,
		idx_comp_idxkey_dword,
		idx_dump_dword,
		idx_comp_rating,
	},
	{	/* @14 */
		idxnode_init,
		idx_setget_num_children_dword,
		idx_get_height,
		idx_record_to_key_playcount,
		idx_setget_child_dword,
		idx_comp_idxkey_dword,
		idx_dump_dword,
		idx_comp_playcount,
	},
	{	/* @15 */
		idxnode_init,
		idx_setget_num_children_dword,
		idx_get_height,
		idx_record_to_key_recentplay,
		idx_setget_child_dword,
		idx_comp_idxkey_dword,
		idx_dump_dword,
		idx_comp_recentplay,
	},
	{	/* @16 */
		idxnode_init,
		idx_setget_num_children_dword,
		idx_get_height,
		idx_record_to_key_format,
		idx_setget_child_dword,
		idx_comp_idxkey_dword,
		idx_dump_dword,
		idx_comp_format,
	},
	{	/* @17 */
		idxleaf_init,
		idx_setget_num_children_none,
		idx_get_height_none,
		idx_record_to_key_none,
		idx_setget_child_none,
		idx_comp_idxkey_none,
		idx_dump_leaf,
		idx_comp_none,
	},
};
