/*
 *      Playlist reader/writer for iRiver Plus 2.
 *
 *      Copyright (c) 2005-2007 Naoaki Okazaki
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 *
 */

/* $Id: playlist.c 328 2007-02-10 17:50:11Z nyaochi $ */

#ifdef	HAVE_CONFIG_H
#include <config.h>
#endif/*HAVE_CONFIG_H*/
#ifdef	HAVE_STRING_H
#include <string.h>
#endif/*HAVE_STRING_H*/

#include <os.h>
#include <stdio.h>
#include <stdlib.h>
#include <pmplib/ucs2char.h>
#include <pmplib/filepath.h>

#include "serialize.h"
#include "ip2db.h"

// ip2db_findfile: check that a file with path 'filename' can be found in the music database
//                 returns 0 if not found.
// db - the music database
// filename - the path given in the playlist file
// path_to_root - encoded mount point prefix
static uint32_t ip2db_findfile(ip2db_t* db, const ucs2char_t *filename, const ucs2char_t *path_to_root)
{
	uint8_t *block = NULL;
	idx_key_t key;
	ucs2char_t pathname[MAX_PATH];
	ucs2char_t filepart[MAX_PATH];
	ucs2char_t decoded_filename[MAX_PATH]; 
	ucs2char_t decoded_filepathname[MAX_PATH]; 
	ucs2char_t decoded_pathname[MAX_PATH];
	uint32_t leafid;

	// 1. Decode the file path listed in the playlist, including mount point prefix.
	ucs2cpy(decoded_filename, filename);
	filepath_decode(decoded_filename);

	// 2. As above, with mount point prefix stripped.
	ucs2cpy(decoded_filepathname, filepath_changeroot(decoded_filename, path_to_root));

	// 3. As above, but with filename stripped.
	ucs2cpy(decoded_pathname, decoded_filepathname);
	filepath_remove_filespec(decoded_pathname);
	// The path stored in the database is encoded, need do comparisons against that.
	ucs2cpy(pathname, decoded_pathname);
	filepath_encode(pathname);

	// 4. As 2, but with directory path stripped.
	ucs2cpy(filepart, filepath_skippath(decoded_filepathname));

	memset(&key, 0, sizeof(key));
	key.type = IP2DBIDX_PAGE_NODE_FILENAME;
	key.dup = 0;
	ucs2ncpy(key.data.str, filepart, 8);

	while (idx_get(db->idx_buffer, &key, &leafid, 0) == 0) {
		ip2db_record_t record;

		ip2db_init_record(&record);
		ip2db_get_record(db, leafid, &record);
		if (ucs2icmp(record.pathname, pathname) == 0 && ucs2icmp(record.filename, filepart) == 0) {
			ip2db_free_record(&record);
			return leafid;
		}
		ip2db_free_record(&record);
		++key.dup;
	}
	return 0;
}

int ip2db_playlist_write(
	ip2db_t* db,
	const ucs2char_t *name,
	ucs2char_t* const mediafiles[],
	int num_mediafiles,
	const ucs2char_t *path_to_root,
	const ucs2char_t *path_to_playlist,
	const ucs2char_t *ext
	)
{
	int i;
	FILE *fp = NULL;
	uint32_t* recids = NULL;
	uint32_t num_recids = 0;
	ucs2char_t dst[MAX_PATH];

	filepath_combinepath(dst, MAX_PATH, path_to_root, path_to_playlist);
	filepath_addslash(dst);
	ucs2cat(dst, name);
	ucs2cat(dst, ext);

	for (i = 0;i < num_mediafiles;i++) {
		if (mediafiles[i][0]) {
			uint32_t recid = 0;
			ucs2char_t mediafile[MAX_PATH];
			ucs2cpy(mediafile, mediafiles[i]);

			filepath_encode(mediafile);

			recid = ip2db_findfile(db, mediafile, path_to_root);
			if (recid == 0) {
				goto error_exit;
			}

			recids = realloc(recids, sizeof(uint32_t) * (num_recids+1));
			if (!recids) {
				goto error_exit;
			}
			recids[num_recids++] = recid;
		}
	}

	fp = ucs2fopen(dst, "wb");
	if (fp) {
		size_t size = sizeof(uint32_t) + sizeof(uint32_t) * num_recids;
		uint8_t* buffer = (uint8_t*)malloc(size);
		uint8_t* p = buffer;

		if (!buffer) {
			goto error_exit;
		}

		p += serialize_uint32le(p, &num_recids, 1);
		for (i = 0;i < (int)num_recids;++i) {
			p += serialize_uint32le(p, &recids[i], 1);
		}
		fwrite(buffer, 1, size, fp);
		free(buffer);
		fclose(fp);
	} else {
		return -1;
	}

	free(recids);
	return 0;

error_exit:
	free(recids);
	return -1;
}
