/*
 *      Media database reader/writer for iriver E10.
 *
 *      Copyright (c) 2005-2007 Naoaki Okazaki
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 *
 */

/* $Id: ip3db.c 331 2007-02-10 18:39:35Z nyaochi $ */

#ifdef	HAVE_CONFIG_H
#include <config.h>
#endif/*HAVE_CONFIG_H*/

#include <os.h>
#include <stdio.h>
#include <stdlib.h>
#include <pmplib/ucs2char.h>
#include <pmplib/filepath.h>

#include "util.h"
#include "ip3db.h"
#include "dic.h"
#include "dat.h"
#include "idx.h"

void ip3db_variant_init(ip3db_variant_t* var, int type)
{
	memset(var, 0, sizeof(*var));
	var->type = type;
	if (var->type == IP3DBVT_STRING) {
		var->value.str = mbsdupucs2("");
		if (!var->value.str) {
			var->type = IP3DBVT_NONE;
		}
	}
}

void ip3db_variant_finish(ip3db_variant_t* var)
{
	if (var->type == IP3DBVT_STRING) {
		ucs2free(var->value.str);
	}
	ip3db_variant_init(var, IP3DBVT_NONE);
}

void ip3db_variant_set_byte(ip3db_variant_t* var, uint8_t val)
{
	var->type = IP3DBVT_BYTE;
	var->value.byte = val;
}

void ip3db_variant_set_word(ip3db_variant_t* var, uint16_t val)
{
	var->type = IP3DBVT_WORD;
	var->value.word = val;
}

void ip3db_variant_set_dword(ip3db_variant_t* var, uint32_t val)
{
	var->type = IP3DBVT_DWORD;
	var->value.dword = val;
}

void ip3db_variant_set_str(ip3db_variant_t* var, const ucs2char_t* val)
{
	var->type = IP3DBVT_STRING;
	ucs2free(var->value.str);
	var->value.str = val ? ucs2dup(val) : ucs2calloc(sizeof(ucs2char_t));
}

void ip3db_variant_clone(ip3db_variant_t* dst, const ip3db_variant_t* src)
{
	switch (dst->type = src->type) {
	case IP3DBVT_BYTE:
		dst->value.byte = src->value.byte;
		break;
	case IP3DBVT_WORD:
		dst->value.word = src->value.word;
		break;
	case IP3DBVT_DWORD:
		dst->value.dword = src->value.dword;
		break;
	case IP3DBVT_STRING:
		ucs2free(dst->value.str);
		dst->value.str = src->value.str ? ucs2dup(src->value.str) : 0;
		break;
	}
}

void ip3db_init(ip3db_t* db)
{
	/* Construct db->dat, db->dic, and db->idx */
	memset(db, 0, sizeof(*db));
	db->dat = dat_new();
	db->dic = dic_new();
	db->idx = idx_new();
}

void ip3db_finish(ip3db_t* db)
{
	dic_finish(db->dic);
	dat_finish(db->dat);
	idx_finish(db->idx);
	memset(db, 0, sizeof(*db));
}

result_t ip3db_read(ip3db_t* db, const ucs2char_t* datfn, const ucs2char_t* dicfn, const ucs2char_t* idxfn)
{
	FILE *fp = 0;

	/*
	 * Read db.dic first since this file describes the structure of a database
	 * (e.g., field type and the offset values to the root nodes in db.idx).
	 */
	fp = ucs2fopen(dicfn, "rb");
	if (!fp) {
		ip3db_finish(db);
		return 1;
	}
	dic_read(db->dic, fp);
	fclose(fp);

	/* Read db.dat */
	fp = ucs2fopen(datfn, "rb");
	if (!fp) {
		ip3db_finish(db);
		return 1;
	}
	dat_read(db->dat, db->dic, fp);
	fclose(fp);

	/* Read db.idx */
	fp = ucs2fopen(idxfn, "rb");
	if (!fp) {
		ip3db_finish(db);
		return 1;
	}
	idx_read(db->idx, db->dic, fp);
	fclose(fp);

	return 0;
}

result_t ip3db_write(ip3db_t* db, const ucs2char_t* datfn, const ucs2char_t* dicfn, const ucs2char_t* idxfn)
{
	FILE *fp = 0;

	/*
	 * Write db.dat. The function dat_write() also stores offset addresses
	 * (db->dat->objects.entries[i].offset and db->dat->music.entries[i].offset)
	 * where records are stored in the file.
	 */
	fp = ucs2fopen(datfn, "wb");
	if (!fp) {
		return 1;
	}
	dat_write(db->dat, db->dic, fp);
	fclose(fp);

	/*
	 * Construct binary search trees from records in db->dat. The root offset
	 * of each search tree will be stored in db->dic.
	 */
	idx_construct(db->idx, db->dic, db->dat);

	/*
	 * Write db.idx. The function idx_write() converts values in the search trees
	 * from native byte-order to big endian. The byte-order conversion will take
	 * place in memory. Therefore, do not use db->idx after calling idx_write()
	 * function.
	 */
	fp = ucs2fopen(idxfn, "wb");
	if (!fp) {
		return 1;
	}
	idx_write(db->idx, db->dic, fp);
	fclose(fp);

	/* Write db.dic. */
	fp = ucs2fopen(dicfn, "wb");
	if (!fp) {
		return 1;
	}
	dic_write(db->dic, fp);
	fclose(fp);

	return 0;
}

result_t ip3db_dump(ip3db_t* db, FILE *fpo)
{
	/* Dump db.dic */
	dic_dump(db->dic, fpo);

	/* Dump db.dat */
	dat_dump(db->dat, db->dic, fpo);

	/* Dump db.idx */
	idx_dump(db->idx, db->dic, fpo);
	return 0;
}

result_t ip3db_set(ip3db_t* db, const ip3db_music_record_t* records, int num_records, ip3db_playlist_t* playlists, int num_playlists)
{
	/* Construct records in db->dat from the records. */
	dat_set(db->dat, db->dic, records, num_records, playlists, num_playlists);
	return 0;
}

int ip3db_num_records(ip3db_t* db)
{
	return (int)db->dat->musics.num_entries;
}

ip3db_music_record_t* ip3db_get_record(ip3db_t* db, int i)
{
	return (ip3db_music_record_t*)db->dat->musics.entries[i].fields;
}

void ip3db_record_init(ip3db_t* db, ip3db_music_record_t* record)
{
	int i;
	ip3db_variant_t* var = (ip3db_variant_t*)record;
	for (i = 0;i < IP3DBF_MUSIC_LAST;++i) {
		ip3db_variant_init(&var[i], db->dic->music.fields[i].type);
	}
}

void ip3db_record_finish(ip3db_t* db, ip3db_music_record_t* record)
{
	int i;
	ip3db_variant_t* var = (ip3db_variant_t*)record;
	for (i = 0;i < IP3DBF_MUSIC_LAST;++i) {
		ip3db_variant_finish(&var[i]);
	}
}

void ip3db_playlist_init(ip3db_playlist_t* playlist)
{
	memset(playlist, 0, sizeof(*playlist));
}

void ip3db_playlist_finish(ip3db_playlist_t* playlist)
{
	int i;
	for (i = 0;i < playlist->num_entries;++i) {
		ucs2free(playlist->entries[i]);
	}
	ucs2free(playlist->entries);
	ucs2free(playlist->filepath);
	ucs2free(playlist->filename);
}

void ip3db_playlist_copy(ip3db_playlist_t* dst, const ip3db_playlist_t* src)
{
	int i;

	dst->filepath = ucs2dup(src->filepath);
	dst->filename = ucs2dup(src->filename);
	dst->num_entries = src->num_entries;
	dst->entries = (ucs2char_t**)ucs2calloc(sizeof(ucs2char_t*) * src->num_entries);
	for (i = 0;i < src->num_entries;++i) {
		dst->entries[i] = ucs2dup(src->entries[i]);
	}
}
