/*
 *      MEDION MDJuke220 specific routines and header templates.
 *
 *      Copyright (c) 2005-2007 Naoaki Okazaki
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 *
 */

/* $Id: model_medion_mdjuke220.c 342 2007-02-11 18:11:43Z nyaochi $ */

#ifdef	HAVE_CONFIG_H
#include <config.h>
#endif/*HAVE_CONFIG_H*/
#ifdef	HAVE_STRING_H
#include <string.h>
#endif/*HAVE_STRING_H*/

#include <os.h>
#include <stdio.h>
#include <stdlib.h>
#include <pmplib/ucs2char.h>
#include <pmplib/filepath.h>

#include "serialize.h"
#include "util.h"
#include "pp1db.h"
#include "hdr_template.h"

enum {
	PP1DB_DATFIELD_UNKNOWN2 = 0,// @0:	(INT)	
	PP1DB_DATFIELD_PATHNAME,	// @1:	(STR)	
	PP1DB_DATFIELD_FILENAME,	// @2:	(STR)	
	PP1DB_DATFIELD_FORMAT,		// @3:	(INT)
	PP1DB_DATFIELD_BITRATE,		// @4:	(INT)
	PP1DB_DATFIELD_SAMPLERATE,	// @5:	(INT)
	PP1DB_DATFIELD_DURATION,	// @6:	(STR)
	PP1DB_DATFIELD_ARTIST,		// @7:	(STR)
	PP1DB_DATFIELD_ALBUM,		// @8:	(STR)
	PP1DB_DATFIELD_GENRE,		// @9:	(STR)
	PP1DB_DATFIELD_TITLE,		// @10:	(STR)
	PP1DB_DATFIELD_TRACKNUMBER,	// @11:	(INT)
	PP1DB_DATFIELD_YEAR,		// @12: (INT)
	PP1DB_DATFIELD_FILESIZE,	// @13: (INT)
	PP1DB_DATFIELD_COMPOSER,	// @14: (STR)
};

void medion_mdjuke220_dat_repr(const dat_t* record, FILE *fp)
{
	fprintf(fp, "  inactive: %d\n", record->status);
	fprintf(fp, "  unknown1: %d\n", record->unknown1);
	fprintf(fp, "  unknown2: %d\n", record->fields[PP1DB_DATFIELD_UNKNOWN2].value.dword);
	fprints(fp, "  file_path: %s\n", record->fields[PP1DB_DATFIELD_PATHNAME].value.str);
	fprints(fp, "  file_name: %s\n", record->fields[PP1DB_DATFIELD_FILENAME].value.str);
	fprintf(fp, "  media_type: %d\n", record->fields[PP1DB_DATFIELD_FORMAT].value.dword);
	fprintf(fp, "  bitrate: %d\n", record->fields[PP1DB_DATFIELD_BITRATE].value.dword);
	fprintf(fp, "  samplerate: %d\n", record->fields[PP1DB_DATFIELD_SAMPLERATE].value.dword);
	fprints(fp, "  duration: %s\n", record->fields[PP1DB_DATFIELD_DURATION].value.str);
	fprints(fp, "  artist: %s\n", record->fields[PP1DB_DATFIELD_ARTIST].value.str);
	fprints(fp, "  album: %s\n", record->fields[PP1DB_DATFIELD_ALBUM].value.str);
	fprints(fp, "  genre: %s\n", record->fields[PP1DB_DATFIELD_GENRE].value.str);
	fprints(fp, "  title: %s\n", record->fields[PP1DB_DATFIELD_TITLE].value.str);
	fprintf(fp, "  number: %d\n", record->fields[PP1DB_DATFIELD_TRACKNUMBER].value.dword);
	fprintf(fp, "  year: %d\n", record->fields[PP1DB_DATFIELD_YEAR].value.dword);
	fprintf(fp, "  filesize: %d\n", record->fields[PP1DB_DATFIELD_FILESIZE].value.dword);
	fprints(fp, "  composer: %s\n", record->fields[PP1DB_DATFIELD_COMPOSER].value.str);
}

int medion_mdjuke220_dat_set(dat_t* dst, const pmp_music_record_t* src, const ucs2char_t* path_to_root)
{
	ucs2char_t duration[128];
	static const ucs2char_t ucs2cs_unknown[] = {'0',0};

	// Set fields.
	dst->status = 0;
	dst->unknown1 = 0;
	dst->fields[PP1DB_DATFIELD_PATHNAME].value.str = ucs2dup(filepath_skiproot(src->filename, path_to_root));
	filepath_remove_filespec(dst->fields[PP1DB_DATFIELD_PATHNAME].value.str);
	filepath_addslash(dst->fields[PP1DB_DATFIELD_PATHNAME].value.str);
	filepath_encode(dst->fields[PP1DB_DATFIELD_PATHNAME].value.str);
	dst->fields[PP1DB_DATFIELD_FILENAME].value.str = ucs2dup(filepath_skippath(src->filename));
	dst->fields[PP1DB_DATFIELD_FORMAT].value.dword = 0;
	dst->fields[PP1DB_DATFIELD_BITRATE].value.dword = src->bitrate;
	dst->fields[PP1DB_DATFIELD_SAMPLERATE].value.dword = src->sample_rate;

	memset(duration, 0, sizeof(duration));
	itoucs2(src->duration, duration, 10);
	dst->fields[PP1DB_DATFIELD_DURATION].value.str = ucs2dup(duration);

	dst->fields[PP1DB_DATFIELD_ARTIST].value.str = ucs2dup(src->artist ? src->artist : ucs2cs_unknown);
	dst->fields[PP1DB_DATFIELD_ALBUM].value.str = ucs2dup(src->album ? src->album : ucs2cs_unknown);
	dst->fields[PP1DB_DATFIELD_GENRE].value.str = ucs2dup(src->genre ? src->genre : ucs2cs_unknown);
	dst->fields[PP1DB_DATFIELD_TITLE].value.str = ucs2dup(src->title ? src->title : dst->fields[PP1DB_DATFIELD_FILENAME].value.str);
	dst->fields[PP1DB_DATFIELD_TRACKNUMBER].value.dword = src->track_number;
	if (src->date) {
		dst->fields[PP1DB_DATFIELD_YEAR].value.dword = ucs2toi(src->date);
	}
	dst->fields[PP1DB_DATFIELD_FILESIZE].value.dword = src->filesize;
	dst->fields[PP1DB_DATFIELD_COMPOSER].value.str = ucs2dup(src->composer ? src->composer : ucs2cs_unknown);
	return 0;
}

int medion_mdjuke220_dat_get(pmp_music_record_t* dst, const dat_t* src, const ucs2char_t* path_to_root)
{
	static const ucs2char_t ucs2cs_mp3[] = {'.','m','p','3',0};
	static const ucs2char_t ucs2cs_wma[] = {'.','w','m','a',0};
	static const ucs2char_t ucs2cs_wav[] = {'.','w','a','v',0};
	ucs2char_t tmp[128];
	size_t length = 0;

	length  = ucs2len(path_to_root);
	length += ucs2len(src->fields[PP1DB_DATFIELD_PATHNAME].value.str);
	length += ucs2len(src->fields[PP1DB_DATFIELD_FILENAME].value.str);
	length += 3;

	dst->filename = (ucs2char_t*)ucs2malloc(sizeof(ucs2char_t) * length);
	if (dst->filename) {
		filepath_combinepath(dst->filename, length, path_to_root, src->fields[PP1DB_DATFIELD_PATHNAME].value.str);
		ucs2cat(dst->filename, src->fields[PP1DB_DATFIELD_FILENAME].value.str);
		filepath_decode(dst->filename);
	}

	dst->bitrate = src->fields[PP1DB_DATFIELD_BITRATE].value.dword;
	dst->sample_rate = src->fields[PP1DB_DATFIELD_SAMPLERATE].value.dword;
	dst->duration = ucs2toi(src->fields[PP1DB_DATFIELD_DURATION].value.str);

	dst->artist = ucs2dup(src->fields[PP1DB_DATFIELD_ARTIST].value.str);
	dst->album = ucs2dup(src->fields[PP1DB_DATFIELD_ALBUM].value.str);
	dst->genre = ucs2dup(src->fields[PP1DB_DATFIELD_GENRE].value.str);
	dst->title = ucs2dup(src->fields[PP1DB_DATFIELD_TITLE].value.str);

	// Set codec information according to the file extensions.
	if (filepath_hasext(dst->filename, ucs2cs_mp3)) {
		dst->codec = PMPCODEC_MPEGLAYER3;
	} else if (filepath_hasext(dst->filename, ucs2cs_wma)) {
		dst->codec = PMPCODEC_WMA;
	} else if (filepath_hasext(dst->filename, ucs2cs_wav)) {
		dst->codec = PMPCODEC_WAV;
	}

	dst->track_number = src->fields[PP1DB_DATFIELD_TRACKNUMBER].value.dword;
	itoucs2(src->fields[PP1DB_DATFIELD_YEAR].value.dword, tmp, 10);
	dst->date = ucs2dup(tmp);
	dst->filesize = src->fields[PP1DB_DATFIELD_FILESIZE].value.dword;
	dst->composer = ucs2dup(src->fields[PP1DB_DATFIELD_COMPOSER].value.str);

	return 0;
}

int medion_mdjuke220_parse_model(const ucs2char_t* firmware, pp1model_t* model)
{
	/*
	Medion MDJUKE 220 1.4.2

	 ADDRESS   00 01 02 03 04 05 06 07 08 09 0A 0B 0C 0D 0E 0F   0123456789ABCDEF 
	------------------------------------------------------------------------------
	 00000200  88 05 00 EA 75 05 00 EA 0E F0 B0 E1 79 05 00 EA   ....u.......y... 
	 00000210  7E 05 00 EA FE FF FF EA 66 05 00 EA 5C 05 00 EA   ~.......f...\... 
	 00000220  70 6F 72 74 61 6C 70 6C 61 79 65 72 00 30 2E 30   portalplayer.0.0 
	 00000230  C0 00 22 59 4C 00 CA 43 00 00 00 00 00 00 00 00   .."YL..C........ 
	 00000240  50 50 35 30 32 30 41 46 2D 30 35 2E 31 31 2D 50   PP5020AF-05.11-P 
	 00000250  50 30 37 2D 30 35 2E 31 31 2D 4D 47 30 31 2D 30   P07-05.11-MG01-0 
	 00000260  31 2E 30 30 2D 44 54 00 4D 44 4A 55 4B 45 32 31   1.00-DT.MDJUKE21 
	 00000270  35 2F 32 32 30 00 00 00 31 2E 34 2E 32 00 00 00   5/220...1.4.2... 
	 00000280  44 69 67 69 74 61 6C 20 4D 65 64 69 61 20 50 6C   Digital Media Pl 
	 00000290  61 74 66 6F 72 6D 00 00 43 6F 70 79 72 69 67 68   atform..Copyrigh 
	 000002A0  74 28 63 29 20 31 39 39 39 20 2D 20 32 30 30 33   t(c) 1999 - 2003 
	 000002B0  20 50 6F 72 74 61 6C 50 6C 61 79 65 72 2C 20 49    PortalPlayer, I 
	 000002C0  6E 63 2E 20 20 41 6C 6C 20 72 69 67 68 74 73 20   nc.  All rights  
	 000002D0  72 65 73 65 72 76 65 64 2E 00 00 00 00 00 00 00   reserved........ 
	 000002E0  00 01 00 00 EC 00 00 00 4C 59 22 00 80 10 A0 E1   ........LY"..... 
	 000002F0  80 04 A0 E1 A0 08 B0 E1 7D 20 E0 E3 21 2C 82 E0   ........} ..!,.. 
	*/
	FILE *fp = ucs2fopen(firmware, "rb");
	if (fp) {
		char line[0x81], *p = NULL, *q = NULL, *e = line + 0x80;
		memset(line, 0, sizeof(line));

		// Seek to the firmware information.
		if (fseek(fp, 0x0240, SEEK_SET) != 0) {
			fclose(fp);
			return 0;
		}

		// Read the firmware information.
		if (fread(line, sizeof(char), 0x80, fp) != 0x80) {
			fclose(fp);
			return 0;
		}

		fclose(fp);

		// Parse product identifier.
		p = line + strlen(line) + 1;
		if (e <= p) {
			return 0;
		}
		strcpy(model->name, p);

		// Parse version.
		q = p + strlen(p) + 3;
		if (e <= p) {
			return 0;
		}
		strcpy(model->version, q);

		strcpy(model->mode, "UM");

		return 1;
	}
	return 0;
}




static fd_template_t hdrtmpl_fd_mdjuke220[] = {
	{0x0000F001, 2,   4, 0, 0, 1, 0, 0, "SYSTEM\\DATA\\PP5000_@DEV.IDX"},
	{0x0000F002, 1, 128, 0, 0, 1, 0, 0, "SYSTEM\\DATA\\PP5000_FPTH.IDX"},
	{0x0000F003, 1, 128, 0, 0, 1, 0, 0, "SYSTEM\\DATA\\PP5000_FNAM.IDX"},
	{0x0000F00A, 2,   4, 0, 0, 1, 0, 0, "SYSTEM\\DATA\\PP5000_FRMT.IDX"},
	{0x0000F005, 2,   4, 0, 0, 0, 0, 0, ""},
	{0x0000F006, 2,   4, 0, 0, 0, 0, 0, ""},
	{0x0000F007, 1,   4, 0, 0, 0, 0, 0, ""},
	{0x0000003C, 1,  40, 0, 0, 1, 0, 0, "SYSTEM\\DATA\\PP5000_TPE1.IDX"},
	{0x0000001C, 1,  40, 0, 0, 1, 0, 0, "SYSTEM\\DATA\\PP5000_TALB.IDX"},
	{0x0000001F, 1,  20, 0, 0, 1, 0, 0, "SYSTEM\\DATA\\PP5000_TCON.IDX"},
	{0x0000002E, 1,  40, 0, 0, 1, 0, 0, "SYSTEM\\DATA\\PP5000_TIT2.IDX"},
	{0x00000043, 2,   4, 0, 0, 1, 0, 0, "SYSTEM\\DATA\\PP5000_TRCK.IDX"},
	{0x0000004E, 2,   4, 0, 0, 0, 0, 0, ""},
	{0x0000F009, 2,   4, 0, 0, 0, 0, 0, ""},
	{0x0000001E, 1,  40, 0, 0, 1, 0, 0, "SYSTEM\\DATA\\PP5000_TCOM.IDX"},
	{0x00000000, 0,	  0, 0, 0, 0, 0, 0, ""},
	{0x00000000, 0,   0, 0, 0, 0, 0, 0, ""},
	{0x00000000, 0,	  0, 0, 0, 0, 0, 0, ""},
	{0x00000000, 0,   0, 0, 0, 0, 0, 0, ""},
	{0x00000000, 0,   0, 0, 0, 0, 0, 0, ""},
};

static uint32_t hdrtmpl_max_dat_field_size_mdjuke220[] =
	{8, 12, 268, 524, 528, 532, 536, 544, 624, 704, 744, 824, 828, 832, 836, 0, 0, 0, 0, 0};


/********** MEDION MDJUKE220 firmware 1.4.2 **********/
static hdr_template_t hdrtmpl_medion_mdjuke220 = {
	0, 0, "SYSTEM\\DATA\\PP5000.DAT", 0, "SYSTEM\\DATA\\PP5000.HDR", 0x00000394, 0, 0, 15,
	hdrtmpl_fd_mdjuke220,
	hdrtmpl_max_dat_field_size_mdjuke220,
	0, 0,
	{189052, 20, 4000, 1032, 0, medion_mdjuke220_dat_repr, medion_mdjuke220_dat_set, medion_mdjuke220_dat_get},
};
int hdr_init_medion_mdjuke220(hdr_t* hdr)
{
	return apply_template(hdr, &hdrtmpl_medion_mdjuke220);
}

