/*
 *      Philips HDD6320 specific routines and header templates.
 *
 *      Copyright (c) 2005-2007 Naoaki Okazaki
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 *
 */

/* $Id: model_philips_hdd6320.c 342 2007-02-11 18:11:43Z nyaochi $ */

#ifdef	HAVE_CONFIG_H
#include <config.h>
#endif/*HAVE_CONFIG_H*/
#ifdef	HAVE_STRING_H
#include <string.h>
#endif/*HAVE_STRING_H*/

#include <os.h>
#include <stdio.h>
#include <stdlib.h>
#include <pmplib/ucs2char.h>
#include <pmplib/filepath.h>

#include "serialize.h"
#include "util.h"
#include "pp1db.h"
#include "hdr_template.h"

enum {
	PP1DB_DATFIELD_UNKNOWN2 = 0,// @0:	(INT)	
	PP1DB_DATFIELD_PATHNAME,	// @1:	(STR)	
	PP1DB_DATFIELD_FILENAME,	// @2:	(STR)
	PP1DB_DATFIELD_FORMAT,		// @3:	(INT)
	PP1DB_DATFIELD_ARTIST,		// @4:	(STR)
	PP1DB_DATFIELD_ALBUM,		// @5:	(STR)
	PP1DB_DATFIELD_GENRE,		// @6:	(STR)
	PP1DB_DATFIELD_TITLE,		// @7:	(STR)
	PP1DB_DATFIELD_TRACKNUMBER,	// @8:	(INT)
	PP1DB_DATFIELD_COMPOSER,	// @9:  (STR)
	PP1DB_DATFIELD_UNKNOWN3,	// @10:	(INT)	
	PP1DB_DATFIELD_UNKNOWN4,	// @11:	(STR)
	PP1DB_DATFIELD_UNKNOWN5,	// @12:	(STR)
	PP1DB_DATFIELD_UNKNOWN6,	// @13:	(STR)
	PP1DB_DATFIELD_UNKNOWN7,	// @14:	(INT)	
	PP1DB_DATFIELD_UNKNOWN8,	// @15:	(STR)
	PP1DB_DATFIELD_UNKNOWN9,	// @16:	(STR)
};

static void philips_hdd6320_dat_repr(const dat_t* record, FILE *fp)
{
	fprintf(fp, "  inactive: %d\n", record->status);
	fprintf(fp, "  unknown1: %d\n", record->unknown1);
	fprintf(fp, "  unknown2: %d\n", record->fields[PP1DB_DATFIELD_UNKNOWN2].value.dword);
	fprints(fp, "  file_path: %s\n", record->fields[PP1DB_DATFIELD_PATHNAME].value.str);
	fprints(fp, "  file_name: %s\n", record->fields[PP1DB_DATFIELD_FILENAME].value.str);
	fprintf(fp, "  media_type: %d\n", record->fields[PP1DB_DATFIELD_FORMAT].value.dword);
	fprints(fp, "  artist: %s\n", record->fields[PP1DB_DATFIELD_ARTIST].value.str);
	fprints(fp, "  album: %s\n", record->fields[PP1DB_DATFIELD_ALBUM].value.str);
	fprints(fp, "  genre: %s\n", record->fields[PP1DB_DATFIELD_GENRE].value.str);
	fprints(fp, "  title: %s\n", record->fields[PP1DB_DATFIELD_TITLE].value.str);
	fprintf(fp, "  number: %d\n", record->fields[PP1DB_DATFIELD_TRACKNUMBER].value.dword);
	fprints(fp, "  composer: %s\n", record->fields[PP1DB_DATFIELD_COMPOSER].value.str);
	fprintf(fp, "  unknown3: %d\n", record->fields[PP1DB_DATFIELD_UNKNOWN3].value.dword);
	fprintf(fp, "  unknown4: %d\n", record->fields[PP1DB_DATFIELD_UNKNOWN4].value.dword);
	fprintf(fp, "  unknown5: %d\n", record->fields[PP1DB_DATFIELD_UNKNOWN5].value.dword);
	fprintf(fp, "  unknown6: %d\n", record->fields[PP1DB_DATFIELD_UNKNOWN6].value.dword);
	fprintf(fp, "  unknown7: %d\n", record->fields[PP1DB_DATFIELD_UNKNOWN7].value.dword);
	fprintf(fp, "  unknown8: %d\n", record->fields[PP1DB_DATFIELD_UNKNOWN8].value.dword);
	fprintf(fp, "  unknown9: %d\n", record->fields[PP1DB_DATFIELD_UNKNOWN9].value.dword);
}

static int philips_hdd6320_dat_set(dat_t* dst, const pmp_music_record_t* src, const ucs2char_t* path_to_root)
{
	static const ucs2char_t ucs2cs_unknown[] = {0};
	static const ucs2char_t ucs2cs_empty[] = {0};

	// Set fields.
	dst->status = 0;
	dst->unknown1 = 0;
	dst->fields[PP1DB_DATFIELD_PATHNAME].value.str = ucs2dup(filepath_skiproot(src->filename, path_to_root));
	filepath_remove_filespec(dst->fields[PP1DB_DATFIELD_PATHNAME].value.str);
	filepath_addslash(dst->fields[PP1DB_DATFIELD_PATHNAME].value.str);
	filepath_encode(dst->fields[PP1DB_DATFIELD_PATHNAME].value.str);
	dst->fields[PP1DB_DATFIELD_FILENAME].value.str = ucs2dup(filepath_skippath(src->filename));
	dst->fields[PP1DB_DATFIELD_FORMAT].value.dword = 0;
	dst->fields[PP1DB_DATFIELD_ARTIST].value.str = ucs2dup(src->artist ? src->artist : ucs2cs_unknown);
	dst->fields[PP1DB_DATFIELD_ALBUM].value.str = ucs2dup(src->album ? src->album : ucs2cs_unknown);
	dst->fields[PP1DB_DATFIELD_GENRE].value.str = ucs2dup(src->genre ? src->genre : ucs2cs_unknown);
	dst->fields[PP1DB_DATFIELD_TITLE].value.str = ucs2dup(src->title ? src->title : dst->fields[PP1DB_DATFIELD_FILENAME].value.str);
	dst->fields[PP1DB_DATFIELD_TRACKNUMBER].value.dword = src->track_number;
	dst->fields[PP1DB_DATFIELD_COMPOSER].value.str = ucs2dup(src->composer ? src->composer : ucs2cs_unknown);

	dst->fields[PP1DB_DATFIELD_UNKNOWN3].value.dword = 0;
	dst->fields[PP1DB_DATFIELD_UNKNOWN4].value.dword = 0;
	dst->fields[PP1DB_DATFIELD_UNKNOWN5].value.dword = 0;
	dst->fields[PP1DB_DATFIELD_UNKNOWN6].value.dword = 0;
	dst->fields[PP1DB_DATFIELD_UNKNOWN7].value.dword = 0;
	dst->fields[PP1DB_DATFIELD_UNKNOWN8].value.dword = 0;
	dst->fields[PP1DB_DATFIELD_UNKNOWN9].value.dword = 0;
	return 0;
}

static int philips_hdd6320_dat_get(pmp_music_record_t* dst, const dat_t* src, const ucs2char_t* path_to_root)
{
	static const ucs2char_t ucs2cs_mp3[] = {'.','m','p','3',0};
	static const ucs2char_t ucs2cs_wma[] = {'.','w','m','a',0};
	static const ucs2char_t ucs2cs_wav[] = {'.','w','a','v',0};
	size_t length = 0;

	memset(dst, 0, sizeof(*dst));

	length  = ucs2len(path_to_root);
	length += ucs2len(src->fields[PP1DB_DATFIELD_PATHNAME].value.str);
	length += ucs2len(src->fields[PP1DB_DATFIELD_FILENAME].value.str);
	length += 3;

	dst->filename = (ucs2char_t*)ucs2malloc(sizeof(ucs2char_t) * length);
	if (dst->filename) {
		filepath_combinepath(dst->filename, length, path_to_root, src->fields[PP1DB_DATFIELD_PATHNAME].value.str);
		ucs2cat(dst->filename, src->fields[PP1DB_DATFIELD_FILENAME].value.str);
		filepath_decode(dst->filename);
	}

	dst->artist = ucs2dup(src->fields[PP1DB_DATFIELD_ARTIST].value.str);
	dst->album = ucs2dup(src->fields[PP1DB_DATFIELD_ALBUM].value.str);
	dst->genre = ucs2dup(src->fields[PP1DB_DATFIELD_GENRE].value.str);
	dst->title = ucs2dup(src->fields[PP1DB_DATFIELD_TITLE].value.str);
	dst->track_number = src->fields[PP1DB_DATFIELD_TRACKNUMBER].value.dword;
	dst->composer = ucs2dup(src->fields[PP1DB_DATFIELD_COMPOSER].value.str);

	// Set codec information according to the file extensions.
	if (filepath_hasext(dst->filename, ucs2cs_mp3)) {
		dst->codec = PMPCODEC_MPEGLAYER3;
	} else if (filepath_hasext(dst->filename, ucs2cs_wma)) {
		dst->codec = PMPCODEC_WMA;
	} else if (filepath_hasext(dst->filename, ucs2cs_wav)) {
		dst->codec = PMPCODEC_WAV;
	}

	return 0;
}

int philips_hdd6320_parse_model(const ucs2char_t* firmware, pp1model_t* model)
{
	/*
		Firmware dump (FWImage.DMP)
		ADDRESS   00 01 02 03 04 05 06 07 08 09 0A 0B 0C 0D 0E 0F   0123456789ABCDEF 
		------------------------------------------------------------------------------
		00000200  58 05 00 EA 45 05 00 EA 0E F0 B0 E1 49 05 00 EA   X...E.......I... 
		00000210  4E 05 00 EA FE FF FF EA 36 05 00 EA 2C 05 00 EA   N.......6...,... 
		00000220  70 6F 72 74 61 6C 70 6C 61 79 65 72 00 30 2E 30   portalplayer.0.0 
		00000230  C0 00 65 D0 94 00 5E F6 00 00 00 00 00 00 00 00   ..e...^......... 
		00000240  50 50 35 30 32 32 41 46 2D 30 35 2E 34 30 2D 50   PP5022AF-05.40-P 
		00000250  50 30 37 2D 30 35 2E 34 30 2D 4D 47 30 32 2D 30   P07-05.40-MG02-0 
		00000260  30 2E 30 31 2D 44 54 00 41 52 49 5A 4F 4E 41 00   0.01-DT.ARIZONA. 
		00000270  32 2E 31 00 44 69 67 69 74 61 6C 20 4D 65 64 69   2.1.Digital Medi 
		00000280  61 20 50 6C 61 74 66 6F 72 6D 00 00 43 6F 70 79   a Platform..Copy 
		00000290  72 69 67 68 74 28 63 29 20 31 39 39 39 20 2D 20   right(c) 1999 -  
		000002A0  32 30 30 33 20 50 6F 72 74 61 6C 50 6C 61 79 65   2003 PortalPlaye 
		000002B0  72 2C 20 49 6E 63 2E 20 20 41 6C 6C 20 72 69 67   r, Inc.  All rig 
		000002C0  68 74 73 20 72 65 73 65 72 76 65 64 2E 00 00 00   hts reserved.... 
		000002D0  00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00   ................ 
		000002E0  00 01 00 00 EC 00 00 00 94 D0 65 00 00 10 A0 E1   ..........e..... 
		000002F0  40 25 A0 E3 00 20 81 E5 01 00 A0 E3 0E F0 A0 E1   @%... .......... 
	*/

	FILE *fp = ucs2fopen(firmware, "rb");
	if (fp) {
		char line[65], *p = NULL, *q = NULL;
		memset(line, 0, sizeof(line));

		// Seek to the firmware information.
		if (fseek(fp, 0x0240, SEEK_SET) != 0) {
			fclose(fp);
			return 0;
		}

		// Read the firmware information.
		if (fread(line, sizeof(char), 64, fp) != 64) {
			fclose(fp);
			return 0;
		}

		// Close the firmware.
		fclose(fp);

		// Obtain the model name.
		strcpy(model->name, "Philips HDD6320");
		strcpy(model->version, "05.40");
		strcpy(model->mode, "UM");
		return 1;
	} else {
		return 0;
	}
}




static fd_template_t hdrtmpl_fd_philips_hdd6320[] = {
	{0x0000F001, 2,   4, 0, 0, 1, 0, 0, "SYSTEM\\DATA\\PP5000_@DEV.IDX"},
	{0x0000F002, 1, 256, 0, 0, 1, 0, 0, "SYSTEM\\DATA\\PP5000_FPTH.IDX"},
	{0x0000F003, 1, 256, 0, 0, 1, 0, 0, "SYSTEM\\DATA\\PP5000_FNAM.IDX"},
	{0x0000F00A, 2,   4, 0, 0, 1, 0, 0, "SYSTEM\\DATA\\PP5000_FRMT.IDX"},
	{0x10B9003C, 1, 256, 0, 0, 1, 0, 0, "SYSTEM\\DATA\\PP5000_TPE1.IDX"},
	{0x2054001C, 1, 256, 0, 0, 1, 0, 0, "SYSTEM\\DATA\\PP5000_TALB.IDX"},
	{0x320A001F, 1, 256, 0, 0, 1, 0, 0, "SYSTEM\\DATA\\PP5000_TCON.IDX"},
	{0x0000002E, 1, 256, 0, 0, 1, 0, 0, "SYSTEM\\DATA\\PP5000_TIT2.IDX"},
	{0x00750043, 2,   4, 0, 0, 1, 0, 0, "SYSTEM\\DATA\\PP5000_TRCK.IDX"},
	{0x0000001E, 1,  64, 0, 0, 1, 0, 0, "SYSTEM\\DATA\\PP5000_TCOM.IDX"},
	{0x54430011, 2,   4, 0, 0, 1, 0, 0, "SYSTEM\\DATA\\PP5000_PCNT.IDX"},
	{0x061A005D, 2,   4, 0, 0, 1, 0, 0, "SYSTEM\\DATA\\PP5000_.IDX"},
	{0x0000008D, 2,   4, 0, 0, 1, 0, 0, "SYSTEM\\DATA\\PP5000_BUYF.IDX"},
	{0x0000008E, 2,   4, 0, 0, 1, 0, 0, "SYSTEM\\DATA\\PP5000_MODF.IDX"},
	{0x8000008F, 2,   4, 0, 0, 0, 0, 0, ""},
	{0x0000E000, 2,   4, 0, 0, 1, 0, 0, "SYSTEM\\DATA\\PP5000_@DU1.IDX"},
	{0x0004E001, 2,   4, 0, 0, 1, 0, 0, "SYSTEM\\DATA\\PP5000_@DU2.IDX"},
	{0x00000000, 0,   0, 0, 0, 0, 0, 0, ""},
	{0x00000000, 0,   0, 0, 0, 0, 0, 0, ""},
	{0x00000000, 0,   0, 0, 0, 0, 0, 0, ""},
	{0x00000000, 0,   0, 0, 0, 0, 0, 0, ""},
	{0x00000000, 0,   0, 0, 0, 0, 0, 0, ""},
};

static uint32_t hdrtmpl_max_dat_field_size_philips_hdd6320[] =
	{8, 12, 524, 1036, 1040, 1552, 2064, 2576, 3088, 3092, 3220, 3224, 3228, 3232, 3236, 3240, 3244, 0, 0, 0, 0, 0};


/********** Philips HDD6320 firmware x.xx **********/
static hdr_template_t hdrtmpl_philips_hdd6320 = {
	0, 0, "System\\DATA\\PP5000.DAT", 1, "System\\DATA\\PP5000.HDR", 0x00000CB0, 0, 0, 17,
	hdrtmpl_fd_philips_hdd6320,
	hdrtmpl_max_dat_field_size_philips_hdd6320,
	0, 1,
	{589120, 22, 12000, 0, 1, philips_hdd6320_dat_repr, philips_hdd6320_dat_set, philips_hdd6320_dat_get},
};

int hdr_init_philips_hdd6320(hdr_t* hdr)
{
	return apply_template(hdr, &hdrtmpl_philips_hdd6320);
}
