/*
 *      Data serializer (with byte-order consideration).
 *
 *      Copyright (c) 2005-2007 Naoaki Okazaki
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 *
 */

/* $Id: serialize.c 328 2007-02-10 17:50:11Z nyaochi $ */

#ifdef	HAVE_CONFIG_H
#include <config.h>
#endif/*HAVE_CONFIG_H*/
#ifdef	HAVE_STRING_H
#include <string.h>
#endif/*HAVE_STRING_H*/

#include <os.h>
#include <stddef.h>
#include <pmplib/ucs2char.h>

#include "serialize.h"

size_t serialize_uint8(uint8_t* buffer, uint8_t* value, int is_storing)
{
	if (is_storing) {
		*buffer = *value;
	} else {
		*value = *buffer;
	}
	return 1;
}

size_t serialize_uint8_array(uint8_t* buffer, uint8_t* array, size_t length, int is_storing)
{
	size_t i;
	for (i = 0;i < length;i++) {
		buffer += serialize_uint8(buffer, array++, is_storing);
	}
	return length;
}

size_t serialize_uint16be(uint8_t* buffer, uint16_t* value, int is_storing)
{
	if (is_storing) {
		buffer[0] = (uint8_t)(*value >> 8);
		buffer[1] = (uint8_t)(*value & 0xFF);
	} else {
		*value = (uint16_t)buffer[0] << 8 | (uint16_t)buffer[1];
	}
	return sizeof(uint16_t);
}

size_t serialize_uint16le(uint8_t* buffer, uint16_t* value, int is_storing)
{
	if (is_storing) {
		buffer[1] = (uint8_t)(*value >> 8);
		buffer[0] = (uint8_t)(*value & 0xFF);
	} else {
		*value = (uint16_t)buffer[1] << 8 | (uint16_t)buffer[0];
	}
	return sizeof(uint16_t);
}

size_t serialize_uint32be(uint8_t* buffer, uint32_t* value, int is_storing)
{
	if (is_storing) {
		buffer[0] = (uint8_t)(*value >> 24);
		buffer[1] = (uint8_t)(*value >> 16);
		buffer[2] = (uint8_t)(*value >> 8);
		buffer[3] = (uint8_t)(*value & 0xFF);
	} else {
		*value =
			(uint32_t)buffer[0] << 24 | (uint32_t)buffer[1] << 16 |
			(uint32_t)buffer[2] << 8 | (uint32_t)buffer[3];
	}
	return sizeof(uint32_t);
}

size_t serialize_uint32le(uint8_t* buffer, uint32_t* value, int is_storing)
{
	if (is_storing) {
		buffer[3] = (uint8_t)(*value >> 24);
		buffer[2] = (uint8_t)(*value >> 16);
		buffer[1] = (uint8_t)(*value >> 8);
		buffer[0] = (uint8_t)(*value & 0xFF);
	} else {
		*value =
			(uint32_t)buffer[3] << 24 | (uint32_t)buffer[2] << 16 |
			(uint32_t)buffer[1] << 8 | (uint32_t)buffer[0];
	}
	return sizeof(uint32_t);
}

size_t serialize_ucs2be(uint8_t* buffer, ucs2char_t* value, int is_storing)
{
	serialize_uint16be(buffer, (uint16_t*)value, is_storing);
	return sizeof(ucs2char_t);
}

size_t serialize_ucs2le(uint8_t* buffer, ucs2char_t* value, int is_storing)
{
	serialize_uint16le(buffer, (uint16_t*)value, is_storing);
	return sizeof(ucs2char_t);
}

size_t serialize_ucs2be_string_fixed(uint8_t* buffer, ucs2char_t* str, size_t length, int is_storing)
{
	size_t i;
	for (i = 0;i < length;i++) {
		serialize_ucs2be(buffer, str, is_storing);
		buffer += sizeof(ucs2char_t);
		++str;
	}
	return length;
}

size_t serialize_ucs2le_string_fixed(uint8_t* buffer, ucs2char_t* str, size_t length, int is_storing)
{
	size_t i;
	for (i = 0;i < length;i++) {
		serialize_ucs2le(buffer, str, is_storing);
		buffer += sizeof(ucs2char_t);
		++str;
	}
	return length;
}

size_t serialize_ucs2be_string_var(uint8_t* buffer, ucs2char_t* str, int is_storing)
{
	size_t length = 0;
	while (serialize_ucs2be(buffer, str, is_storing), *str) {
		++length;
		++str;
		buffer += sizeof(ucs2char_t);
	}
	return length;
}

size_t serialize_ucs2be_string_var_alloc(uint8_t* buffer, ucs2char_t** str)
{
	ucs2char_t c;
	size_t length = 0;
	uint8_t* p = buffer;

	/* Measure the length of the string. */
	while (serialize_ucs2be(p, &c, 0), c) {
		++length;
		p += sizeof(ucs2char_t);
	}

	/* Allocate a string buffer. */
	ucs2free(*str);
	*str = ucs2malloc(sizeof(ucs2char_t) * (length+1));
	memset(*str, 0, sizeof(ucs2char_t) * (length+1));

	/* Read the string. */
	serialize_ucs2be_string_fixed(buffer, *str, length, 0);

	return length;
}
