/*
 *      UCS-2 implementation with libiconv.
 *
 *      Copyright (c) 2005-2007 Naoaki Okazaki
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA, or visit
 * http://www.gnu.org/copyleft/gpl.html .
 *
 */

/* $Id: ucs2char_iconv.c 344 2007-02-11 21:06:58Z sucknblow $ */

#ifdef	HAVE_CONFIG_H
#include <config.h>
#endif/*HAVE_CONFIG_H*/

#include <os.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <pmplib/ucs2char.h>

#include <iconv.h>

#ifdef HAVE_LANGINFO_CODESET
#include <langinfo.h>
#endif


struct tag_ucs2conv {
  const char *from;
  const char *to;
};
typedef struct tag_ucs2conv ucs2conv_t;

#ifdef  USE_LIBICONV_GNU
#define iconv_open	libiconv_open
#define iconv_convert	libiconv_convert
#define iconv_close	libiconv_close
#endif/*USE_LIBICONV_GNU*/

#ifndef ICONV_CONST
#define ICONV_CONST
#endif/*ICONV_CONST*/

#define	MBS_CHARSET	"UTF-8"

static char g_encoding[128];
static char g_ucs2encoding[128];

static int ucs2check(ucs2conv_t* conv);

static void print_ucs2(const ucs2char_t* str)
{
  while (*str) {
    fprintf(stderr, "0x%X %c ", *str);
    str++;
  }
  fprintf(stderr, "\n");
}

static int is_bigendian(void)
{
	ucs2char_t c = 0x1234;
	uint8_t* p = (uint8_t*)&c;
	return (*p == 0x12);
}

static const char *get_ucs2encoding(void)
{
	static const char *unicode_big = "UNICODEBIG";
	static const char *unicode_little = "UNICODELITTLE";
	return is_bigendian() ? unicode_big : unicode_little;
}

int ucs2init() 
{
	ucs2conv_t conv;
	const char *encoding = getenv("CHARSET");

#ifdef	HAVE_LANGINFO_CODESET
	if (!encoding) {
		encoding = nl_langinfo(CODESET);
	}
#endif

	strcpy(g_encoding, encoding ? encoding : MBS_CHARSET);
	strcpy(g_ucs2encoding, get_ucs2encoding());
	return ucs2check(&conv);
}

static int ucs2check(ucs2conv_t* conv)
{
	int i = 0;
	iconv_t cd;
	const static char *charsets[] = {
		g_encoding, "UNICODE",
		"UNICODE", g_encoding,
		"UNICODE", "UTF-8",
		NULL, NULL,
	};

	for (i = 0;charsets[i];i += 2) { 
		const char *tocode = charsets[i];
		const char *fromcode = charsets[i+1];
		if (strcmp(tocode, "UNICODE") == 0) {
		  tocode = get_ucs2encoding();
		}
		if (strcmp(fromcode, "UNICODE") == 0) {
		  fromcode = get_ucs2encoding();
		}
        cd = iconv_open(tocode, fromcode);
		if (cd == (iconv_t)-1) {
		  if (conv) {
		    conv->from = fromcode;
		    conv->to = tocode;
		  }
			return -1;
		}
		iconv_close(cd);
	}
	return 0;
}

int ucs2setenc(const char *encoding)
{
	int ret = 0;
	ucs2conv_t conv;
	memset(g_encoding, 0, sizeof(g_encoding));
	strncpy(g_encoding, encoding, sizeof(g_encoding));
	strcpy(g_ucs2encoding, get_ucs2encoding());
	return ucs2check(&conv);
}

const char *ucs2getenc()
{
	return g_encoding;
}

void ucs2setcp(int cp)
{
}

int ucs2getcp()
{
	return 0;
}


int ucs2toi(const ucs2char_t* str)
{
	int ret;
	char *mbs = ucs2dupmbs(str);
	ret = atoi(mbs);
	free(mbs);
	return ret;
}

ucs2char_t* itoucs2(int value, ucs2char_t *string, int radix)
{
	char buff[1024];
	sprintf(buff, "%d", value);
	mbstoucs2(string, 1024, buff, strlen(buff)+1);
	return string;
}

size_t iconv_convert(iconv_t cd, char **outbuf, size_t out_size, char **inbuf, size_t in_size)
{
	size_t ret = 0;
	if (*outbuf && out_size) {
		size_t inbytesleft = in_size;
		size_t outbytesleft = out_size;
		iconv(cd, (ICONV_CONST char **)inbuf, &inbytesleft, (char **)outbuf, &outbytesleft);
		ret = (out_size - outbytesleft);
	} else {
		size_t inbytesleft = in_size;
		while (inbytesleft > 0) {
			char buffer[1024];
			char *p = buffer;
			size_t outbytesleft = 1024;
			int iconvret = iconv(cd, (ICONV_CONST char **)inbuf, &inbytesleft, &p, &outbytesleft);
			if (iconvret == -1 && errno != E2BIG) {
				return 0;
			}
			ret += (1024 - outbytesleft);
		}
	}
	return ret;
}

size_t ucs2tombs(char *mbstr, size_t mbs_size, const ucs2char_t *ucs2str, size_t ucs_size)
{
	iconv_t cd = iconv_open(g_encoding, g_ucs2encoding);
	size_t ret = iconv_convert(cd, (char **)&mbstr, mbs_size, (char **)&ucs2str, ucs_size * sizeof(ucs2char_t));
	iconv_close(cd);
	return ret;
}

size_t mbstoucs2(ucs2char_t *ucs2str, size_t ucs_size, const char *mbstr, size_t mbs_size)
{
	iconv_t cd = iconv_open(g_ucs2encoding, g_encoding);
	size_t ret = iconv_convert(cd, (char **)&ucs2str, ucs_size * sizeof(ucs2char_t), (char **)&mbstr, mbs_size);
	iconv_close(cd);
	return ret;
}

size_t mbstoucs2_charset(ucs2char_t *ucs2str, size_t ucs_size, const char *mbstr, size_t mbs_size, const char *charset)
{
	iconv_t cd = iconv_open(g_ucs2encoding, charset);
	size_t ret = iconv_convert(cd, (char **)&ucs2str, ucs_size * sizeof(ucs2char_t), (char **)&mbstr, mbs_size);
	iconv_close(cd);
	return ret;
}

size_t utf8toucs2(ucs2char_t *ucs2str, size_t ucs_size, const char *mbstr, size_t mbs_size)
{
	iconv_t cd = iconv_open(g_ucs2encoding, "UTF-8");
	size_t ret = iconv_convert(cd, (char **)&ucs2str, ucs_size * sizeof(ucs2char_t), (char **)&mbstr, mbs_size);
	iconv_close(cd);
	return ret;
}







static void filepath_decode(char *p)
{
	while (*p) {
		if (*p == 0x005C) {
			*p = 0x002F;
		}
		p++;
	}
}

int fputucs2c(ucs2char_t c, FILE *fp)
{
	ucs2char_t ucs2str[2] = {c, 0};
	size_t mbs_size = ucs2tombs(NULL, 0, ucs2str, ucs2len(ucs2str)) + 1;
	char* mbs = (char *)alloca(mbs_size * sizeof(char));
	if (mbs) {
		ucs2tombs(mbs, mbs_size, ucs2str, ucs2len(ucs2str)+1);
		return (fputs(mbs, fp) != EOF ? 0 : 1);
	} else {
		return 1;
	}
}

FILE *ucs2fopen(const ucs2char_t *filename, const char *mode)
{
	FILE *fp = NULL;
	char *mbs_filename = ucs2dupmbs(filename);

	if (mbs_filename) {
		/* Convert '\\' to '/'. */
		filepath_decode(mbs_filename);
		//fprintf(stderr, "fopen: %s\n", mbs_filename);
		fp = fopen(mbs_filename, mode);
	}
	free(mbs_filename);
	return fp;
}
