/*  GNOME Clipboard Daemon
 *  Copyright (C) 2003, 2006  Hongli Lai <h.lai@chello.nl>
 *
 *  Release under the GNU General Public License.
 */

#include <gtk/gtk.h>
#include <gdk/gdkx.h>

#include <string.h>
#include "selection-data-list.h"


static GtkClipboard *clipboard = NULL;
static GdkAtom targets_atom = GDK_NONE;
static GdkAtom no_store_atom = GDK_NONE;
static SelectionDataList *clipboard_saved = NULL;

static gboolean check_clipboard (gpointer user_data);


/* Free the saved clipboard data */
static void
free_saved_data ()
{
	if (!clipboard_saved) return;
	selection_data_list_free (clipboard_saved);
	clipboard_saved = NULL;
}


/* Save the current clipboard data */
static gboolean
save_clipboard_data (GtkSelectionData *content)
{
	GdkAtom *atoms = NULL;
	gint i, count;

	free_saved_data ();
	clipboard_saved = selection_data_list_new ();

	if (!gtk_selection_data_get_targets (content, &atoms, &count))
	    {
		/* The type generated by KDE apps is named "TARGETS", but it's 92, which does
		   not equal GDK_SELECTION_TYPE_ATOM (4). That's why gtk_selection_data_get_targets()
		   fails. */

		count = content->length / sizeof (Atom);
		atoms = g_new0 (GdkAtom, count);

		for (i = 0; i < count; i++)
			atoms[i] = gdk_x11_xatom_to_atom ((Atom) content->data[i * sizeof (Atom)]);
	    }

	for (i = 0; i < count; i++)
	    {
		GtkSelectionData *data;
		gchar *name;

		name = gdk_atom_name (atoms[i]);
		if (!name) continue;

		data = gtk_clipboard_wait_for_contents (clipboard, atoms[i]);
		selection_data_list_add (clipboard_saved, name, data);
	    }

	g_free (atoms);
	gtk_selection_data_free (content);
	return TRUE;
}


static void
clipboard_get_func (GtkClipboard *cb, GtkSelectionData *data, guint info, gpointer user_data_or_owner)
{
	SelectionDataEntry *data_entry;

	data_entry = GINT_TO_POINTER ((guint) info);
	gtk_selection_data_set (data, data_entry->data->type, data_entry->data->format,
		data_entry->data->data, data_entry->data->length);
}


/* We lost ownership */
static void
clipboard_clear_func (GtkClipboard *cb, gpointer user_data_or_owner)
{
	/* Try to reclaim it */
	check_clipboard (GINT_TO_POINTER (TRUE));
}


/* Restore the clipboard from saved data and claim ownership */
static void
claim_clipboard ()
{
	GtkTargetEntry *targets;
	guint n_targets = 0;

	targets = selection_data_list_make_targets (clipboard_saved, &n_targets);
	if (!gtk_clipboard_set_with_data (clipboard, targets, n_targets,
	    clipboard_get_func, clipboard_clear_func, NULL))
		gtk_timeout_add (1500, check_clipboard, NULL);
	g_free (targets);
}


static gboolean
check_clipboard (gpointer reclaim)
{
	GtkSelectionData *content;

	/* Check for the __NO_STORE atom. If available, don't store the
	   clipboard and wait 1 second before trying again */
	content = gtk_clipboard_wait_for_contents (clipboard, no_store_atom);
	if (content)
	    {
		gtk_selection_data_free (content);
		gtk_timeout_add (1000, check_clipboard, NULL);
		return FALSE;
	    }

	content = gtk_clipboard_wait_for_contents (clipboard, targets_atom);
	if (content)
	    {
		gchar *name;
		gboolean saved = FALSE;

		/* Save the clipboard content and claim ownership */
		name = gdk_atom_name (content->type);
		if (name && (strcmp (name, "ATOM") == 0
		 || (strcmp (name, "TARGETS") == 0 && content->length > -1)))
			saved = save_clipboard_data (content);
		else
			g_print ("Unknown atom %s???\n", name);
		g_free (name);

		if (saved)
			claim_clipboard ();
		else
			/* Saving failed; try again */
			gtk_timeout_add (500, check_clipboard, NULL);
	    }
	else if (GPOINTER_TO_INT (reclaim))
		/* We lost the clipboard ownership but the new clipboard doesn't have data.
		   Mozilla does this. Reclaim the clipboard and restore the previously saved
		   data. */
		claim_clipboard ();
	else
		/* No clipboard data; try again in 500 msec */
		gtk_timeout_add (500, check_clipboard, NULL);

	return FALSE;
}


gboolean
clipboard_daemon_start ()
{
	GdkAtom daemon;

	/* Check if a clipboard daemon is already started */
	daemon = gdk_atom_intern ("CLIPBOARD_MANAGER", FALSE);
	if (XGetSelectionOwner (gdk_x11_get_default_xdisplay (), gdk_x11_atom_to_xatom (daemon)) != None)
	    {
		/* Yes; try to start the daemon again in 3 seconds */
		gtk_timeout_add (3000, (GtkFunction) clipboard_daemon_start, NULL);
		return FALSE;
	    }

	/* Claim the daemon atom as our own so no other clipboard daemons
	   can start. */
	gtk_selection_owner_set (gtk_invisible_new (), daemon, GDK_CURRENT_TIME);


	/* Initialize */
	clipboard = gtk_clipboard_get (GDK_SELECTION_CLIPBOARD);
	targets_atom = gdk_atom_intern ("TARGETS", FALSE);
	no_store_atom = gdk_atom_intern ("__NO_STORE", FALSE);

	check_clipboard (NULL);

	return FALSE;
}
