/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2010 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**
   \file mkisofs_interface.c

   \brief Functions creating calls to mkisofs and calling mkisofs

   Functions for creating mkisofs command line strings, creating string
   with cdrecord parameters, and for calling run_command()
   with prepared command line string.
*/

#define _GNU_SOURCE /* asprintf() */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <stdbool.h>

#include "gettext.h"
#include "cdw_thread.h"
#include "cdw_config.h"
#include "cdw_string.h"
#include "cdw_mkisofs.h"
#include "cdw_graftpoints.h"
#include "cdw_debug.h"
#include "cdw_ext_tools.h"
#include "cdw_drive.h"

/* global variable with cdw configuration */
extern cdw_config_t global_config;


static ssize_t cdw_mkisofs_create_multi_string(const cdw_disc_t *disc, char **multi_string);


/**
   \brief Copy values of all relevant mkisofs parameters into one string

   Go through (almost) all currently supported mkisofs options and if
   given option is enabled then put its value in one result string.
   Some of appended options will also depend on \p task.

   This function will put mkisofs name at the beginning of result
   string only if the string (command) is intended to be used as
   command for creating image on hard disc (this depends on
   task->burning.tool.id value).

   Currently this function supports following options:

   -iso-level <config.iso_level>
   -J (config.joliet)
   -R (config.rockridge)
   -r (config.usefulRR)
   -joliet-long (config.joliet_long)
   -f (config.follow_symlinks)

   -C X,Y -prev-session device-name

   -V <config.volumeid>

   -graft-points -path-list <path to gratfpoints file> (hardwired)
   -o <config.tempdir>

   If resulting ISO filesystem will be piped into burning software's stdio
   (which is guessed by checking if task->id == CDW_TASK_BURN_FROM_FILES),
   and the burning software is cdrecord (which is guessed by checking if
   task->burning.tool.id == CDW_TOOL_CDRECORD), the function will add following
   options to resulting string:
   '-C disc->cdrecord.last_sess_start,disc->cdrecord.next_sess_start -prev-session config.cdrw_device'

   Value of config.other_mkisofs_options variable is also appended.

   \param task - variable describing current task
   \param disc - variable describing current disc

   \return non-NULL string with command on success
   \return NULL on errors
*/
char *cdw_mkisofs_create_command(cdw_task_t *task, cdw_disc_t *disc)
{
	cdw_assert (task->id == CDW_TASK_BURN_FROM_FILES || task->id == CDW_TASK_CREATE_IMAGE,
		    "ERROR: incorrect task id that is neither CDW_TASK_BURN_FROM_FILES nor CDW_TASK_CREATE_IMAGE: %d\n", task->id);

	bool include_mkisofs = false;
	bool write_to_stdout = false;
	bool append_multi = false;

	if (task->id == CDW_TASK_BURN_FROM_FILES) {
		cdw_assert (task->burn.tool.id == CDW_TOOL_CDRECORD
			    || task->burn.tool.id == CDW_TOOL_GROWISOFS,
			    "ERROR: incorrect burn tool id in \"burn from files\" task: %d\n", task->burn.tool.id);

		if (task->burn.tool.id == CDW_TOOL_CDRECORD) {
			include_mkisofs = true;

			/* mkisofs will stream data to its stdout - into cdrecord */
			write_to_stdout = true;

			/* cdrecord requires information about existing sessions */
			append_multi = true;
		} else { /* task->tool == CDW_TOOL_GROWISOFS */
			/* growisofs needs only mkisofs' options,
			   no need for us to call mkisofs */
			include_mkisofs = false;

			/* mkisofs will stream data to its stdout - into growisofs */
			write_to_stdout = true;

			/* growisofs doesn't need multisession info */
			append_multi = false;
		}

	} else { /* task->id == CDW_TASK_CREATE_IMAGE */
		cdw_assert (task->create_image.tool.id == CDW_TOOL_MKISOFS,
			    "ERROR: incorrect tool id in \"create image\" task: %d\n", task->create_image.tool.id);
		/* command for creating ISO image only - obviously mkisofs
		   program name is required */
		include_mkisofs = true;

		/* mkisofs will stream data to some physical file */
		write_to_stdout = false;

		/* creating ISO image alone does not require multisession info */
		append_multi = false;
	}

	char *multi_string = (char *) NULL;
	/* multisession information extracted from from media currently
	   in drive is relevant only if mkisofs will produce image that
	   is immediately piped to cdrecord; produce and append
	   multisession info only if we are writing image to stdout
	   and disk is multisession (already has at least one session) */
	if (append_multi) {
		cdw_sdm ("attempting to collect multisession information for cdrecord\n");

		/* sets multi_string only if disc is non-empty and appendable */
		ssize_t multi_len = cdw_mkisofs_create_multi_string(disc, &multi_string);
		if (multi_len == 0) {
			append_multi = false;
		} else if (multi_len > 0) {
			append_multi = true;
			cdw_assert (multi_string != (char *) NULL,
				    "ERROR: failed to initialize multi_string\n");
		} else {
			cdw_vdm ("ERROR: failed to prepare multi_string\n");
			return (char *) NULL;
		}
	}


	/* " -iso-level %zd ", '%zd' conversion specifier ensures
	   that iso level value is printed using no more than necessary
	   number of characters (e.g. 1) */
	size_t iso_len_max = strlen(" -iso-level X ");
	char iso_level_string[iso_len_max + 1];
	int sn = snprintf(iso_level_string, iso_len_max + 1, " -iso-level %d ", global_config.iso_level);
	cdw_vdm ("INFO: iso level string = \"%s\", %d bytes written, %zd chars in string / %zd chars limit\n",
		 iso_level_string, sn, strlen(iso_level_string), iso_len_max);


	/* don't force volume ID, but don't let it be null either */
	cdw_assert (global_config.volumeid != (char *) NULL, "config.volumeid can't be NULL\n");
	/* e.g. ' -V "some name" ', double quotes are important in case of name
	   with spaces; volume name is no longer than VOLUME_ID_LEN_MAX chars
	   (without ending \0, cdw_config.h) */
	size_t volumeid_len_max = 5 + VOLUME_ID_LEN_MAX + 2;
	char volumeid_string[volumeid_len_max + 1];
	if (strlen(global_config.volumeid)) {
		sn = snprintf(volumeid_string, volumeid_len_max + 1, " -V \"%.*s\" ", VOLUME_ID_LEN_MAX, global_config.volumeid);
	} else {
		sn = snprintf(volumeid_string, 1 + 1, " ");
	}
	cdw_vdm ("INFO: volume id string = \"%s\", %d bytes written, %zd chars in string / %zd chars limit\n",
		 volumeid_string, sn, strlen(volumeid_string), volumeid_len_max);

	const char *graftpoints_fullpath = cdw_graftpoints_get_fullpath();
	cdw_assert (graftpoints_fullpath != (char *) NULL,
		    "ERROR: graftpoints fullpath is not initialized (is NULL)\n");

	const char *tool_fullpath = task->create_image.tool.fullpath;
	cdw_assert (tool_fullpath != (char *) NULL,
		    "ERROR: \"create image\" tool fullpath is not initialized (is NULL)\n");

	char *command = cdw_string_concat(

			include_mkisofs ? tool_fullpath : " ",
			iso_level_string, /* surrounded by spaces */
			global_config.joliet ? " -J " : " ",
			global_config.rockridge ? " -R " : " ",
			global_config.useful_rr ? " -r " : " ",
			global_config.joliet_long ? " -joliet-long " : " ",
			global_config.follow_symlinks ? " -f " : " ",

			append_multi ? multi_string : " ",

			volumeid_string, /* already surrounded by spaces, if needed */

			/* options entered by user in Configuration window
			   in 'Other mkisofs options' field */
			" ", global_config.other_mkisofs_options, " ",

			" ", (strlen(global_config.boot_disc_options)) ? global_config.boot_disc_options : " ", " ",

			/* 'graft points' means text file with
			   list of files to burn AND root
			   directories on CD that the files will
			   be put into; see also -path-list
			   argument below */
			" -graft-points -path-list ", graftpoints_fullpath, " ",

			write_to_stdout ? " " : " -o ",
			/* put file name in quotes, because the name may contain spaces */
			write_to_stdout ? " " : "\"",
			write_to_stdout ? " " : global_config.iso_image_full_path,
			write_to_stdout ? " " : "\"",

			/* NULL is obligatory last concat() argument */
			(char *) NULL);

	if (multi_string != (char *) NULL) {
		free(multi_string);
		multi_string = (char *) NULL;
	}

	if (command == (char *) NULL) {
		cdw_vdm ("ERROR: failed to create mkisofs command string\n");
		return (char *) NULL;
	} else {
		return command;
	}
}





/**
   \brief Create iso image from files selected by user

   Create command line string (program name + arguments) for creating iso
   image from selected files, and run the command.

   Call this function only if all preconditions are met - the function
   behaves as if every condition for proper execution is met.

   Currently only creating iso image from data files is supported.

   \param task - variable describing current task
   \param disc - variable describing current disc

   \return CDW_OK when iso file is created
   \return CDW_GEN_ERROR if operation_failed
*/
cdw_rv_t cdw_mkisofs_run_task(cdw_task_t *task, cdw_disc_t *disc)
{
	cdw_assert (task->id == CDW_TASK_CREATE_IMAGE,
		    "ERROR: incorrect task id is not CDW_TASK_CREATE_IMAGE: %d\n", task->id);
	cdw_assert (task->create_image.tool.id == CDW_TOOL_MKISOFS,
		    "ERROR: incorrect tool id is not CDW_TOOL_MKISOFS in \"create image\" task: %d\n", task->create_image.tool.id);

	char *command = cdw_mkisofs_create_command(task, disc);
	if (command == (char *) NULL) {
		cdw_vdm ("ERROR: failed to get command for creating image\n");
		return CDW_GEN_ERROR;
	} else {
		cdw_sdm ("INFO: created command string \"%s\"\n", command);

		run_command(command, task);

		free(command);
		command = (char *) NULL;

		return CDW_OK;
	}
}





/**
   \brief Create string with multisession parameters for cdrecord

   Create string with parameters that allow cdrecord to burn second and
   next sessions on multisession disc. The string is created only if needed,
   e.g. if two parameters describing session state on current disc (e.g.
   disc->cdrecord.last_sess_start and disc->cdrecord.next_sess_start) have
   proper values. Otherwise the string is not initialized / created.

   The function checks values of fields in disc->cdrecord, but the arguments
   in resulting string are passed to mkisofs, so this function belongs to
   mkisofs_interface.c.

   Function must be called with \p multi_string being NULL pointer. If the
   string is not created (but no error occurred), \p multi_string is not
   touched and 0 is returned. On errors -1 is returned and \p multi_string
   is not touched. If the string is initialized, function returns positive
   length of the string. Caller is responsible for free()ing the string if
   necessary.

   \param disc - variable describing current disc
   \param multi_string - string with multisession parameters

   \return -1 on errors
   \return 0 if there was no need to create parameters string
   \return string length if string was created
*/
ssize_t cdw_mkisofs_create_multi_string(const cdw_disc_t *disc, /* out */ char **multi_string)
{
	cdw_assert (disc->simple_type == CDW_DISC_SIMPLE_TYPE_CD
		    || disc->simple_type == CDW_DISC_SIMPLE_TYPE_DVD,

		    "ERROR: simple disc type is neither CD nor DVD: %d\n",
		    disc->simple_type);
	cdw_assert (*multi_string == (char *) NULL, "ERROR: *multi_string is not NULL\n");

	cdw_sdm ("INFO disc simple type is %s\n",
		 disc->simple_type == CDW_DISC_SIMPLE_TYPE_CD ? "CD" : "DVD");
	cdw_vdm ("INFO: last_sess_start=%ld, next_sess_start=%ld\n",
		 disc->cdrecord_info.last_sess_start, disc->cdrecord_info.next_sess_start);

	if (disc->cdrecord_info.last_sess_start == 0
	    && disc->cdrecord_info.last_sess_start == disc->cdrecord_info.next_sess_start) {

		return 0; /* no sessions on CD disc */

	} else if (0 <= disc->cdrecord_info.last_sess_start
		   && disc->cdrecord_info.last_sess_start < disc->cdrecord_info.next_sess_start) {

		/* " -C 26216,40778 -prev-session /dev/hdb "
		   "-prev-session" points to device with already existing
		   session, the device to which we want to merge/append new
		   data with cdrecord */

		/* config.cdrw_device is a fixed-size table, so it's guaranteed to be != NULL */
		const char *drive = cdw_drive_get_drive_fullpath();
		cdw_assert (drive != (char *) NULL && strlen(drive), "ERROR: invalid \"drive\" = \"%s\"\n", drive);

	        int n = asprintf(multi_string, " -C %ld,%ld -prev-session %s ",
				 disc->cdrecord_info.last_sess_start,
				 disc->cdrecord_info.next_sess_start,
				 drive);

		if (n == -1) {
			cdw_vdm ("ERROR: failed to allocate memory for multi string\n");
			return -1;
		} else {
			cdw_vdm ("INFO: successfully created multi string: \"%s\"\n",
				 *multi_string);
			return (ssize_t) strlen(*multi_string);
		}
	} else {
		cdw_vdm ("ERROR: session starts mismatch: last_sess_start=%ld, next_sess_start=%ld\n",
			 disc->cdrecord_info.last_sess_start, disc->cdrecord_info.next_sess_start);

		return -1;
	}
}
