/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2010 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#define _BSD_SOURCE /* lstat() */

#include <unistd.h>
#include <errno.h>
#include <string.h>
/* open() */
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>


#include "cdw_config.h"
#include "cdw_fs.h"
#include "cdw_iso9660.h"
#include "isosize.h"
#include "cdw_debug.h"
#include "cdw_widgets.h"
#include "gettext.h"

extern cdw_config_t global_config;      /* variable holding cdw configuration */



long cdw_iso_image_get_n_sectors(void)
{
	cdw_vdm ("INFO: inspecting file \"%s\"\n", global_config.iso_image_full_path);
	int f = open(global_config.iso_image_full_path, O_RDONLY);
	if (f == -1) {
		cdw_fs_errno_handler(errno);
		cdw_vdm ("ERROR: failed to open iso file image\n");
		return CDW_GEN_ERROR;
	}
	long long size = isosize(f);
	close(f);
	/* WARNING: sector size is set as constant (2048), but:
	   - 2048 is at the moment the only sector size supported by cdw
	   - mkisofs man page describes "-print-size" as: "Print
	   estimated filesystem size in multiples of the sector
	   size (2048 bytes) and exit". Why only 2048? Who should
	   re-calculate the size when another size should be used?
	   cdrecord or caller of cdrecord? */

	long sectors = size / 2048;
	long rem = size % 2048;
	if (rem != 0) {
		cdw_vdm ("ERROR: size of image file is not multiple of 2048 (size = %lld, size / 2048 = %lld)\n", size, size % 2048);
	}

	cdw_vdm ("INFO: returning %ld sectors -> %fGB\n", sectors, ((double) sectors) * 2.0 / (1024 * 1024));
	return sectors;
}





double cdw_iso_image_get_size_mb(void)
{
	struct stat stbuf;
	int rv = lstat(global_config.iso_image_full_path, &stbuf);
	if (rv != 0) {
		int e = errno;
		cdw_vdm ("ERROR: can't lstat() iso file (\"%s\")\n", strerror(e));
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Error"),
				   /* 2TRANS: this is message in dialog window */
				   _("There seems to be problem with iso file. Please check the file location."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		cdw_fs_errno_handler(e);

		return CDW_GEN_ERROR;
	}

	long long iso_file_size_b = (long long) stbuf.st_size;
	double iso_file_size_mb = (double) (iso_file_size_b / (1024 * 1024));
	cdw_sdm ("iso_file_size_b = %lld, iso_file_size_mb = %f\n", iso_file_size_b, iso_file_size_mb);

	return iso_file_size_mb;
}



