/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2010 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <string.h>
#include <stdlib.h>
#include <errno.h>

#include "cdw_debug.h"
#include "cdw_form.h"
#include "cdw_widgets.h"
#include "cdw_string.h"

static void cdw_form_driver_focus_off(cdw_form_t *cdw_form, int fi);
static void cdw_form_driver_focus_on(cdw_form_t *cdw_form, int fi);
static void cdw_form_driver_toggle_focus(cdw_form_t *cdw_form, int fi, bool off);
static void cdw_form_driver_toggle_checkbox(cdw_form_t *cdw_form, int fi);
static CDW_DROPDOWN *cdw_form_driver_get_dropdown_from_form(cdw_form_t *cdw_form, int fi);
static CDW_BUTTON   *cdw_form_driver_get_button_from_form(cdw_form_t *cdw_form, int fi);
static bool cdw_form_is_return_key(cdw_form_t *form, int key);

static bool is_checkbox_field_index(cdw_form_t *cdw_form, int fi);
static bool is_dropdown_field_index(cdw_form_t *cdw_form, int fi);
static bool is_button_field_index(cdw_form_t *cdw_form, int fi);
static bool is_input_field_index(cdw_form_t *cdw_form, int fi);

static void cdw_form_driver_go_to_field(FORM *form, int fi, int n_fields);


/* used by cdw_form_get_field_string() */
static char empty_string[] = "";


/**
   \brief Constructor

   \param n_fields - expected maximal number of fields in a form
*/
cdw_form_t *cdw_form_new(int n_fields)
{
	cdw_form_t *cdw_form = (cdw_form_t *) malloc(sizeof (cdw_form_t));
	if (cdw_form == (cdw_form_t *) NULL) {
		cdw_vdm ("ERROR: failed to allocate memory for new cdw form\n");
		return (cdw_form_t *) NULL;
	}
	cdw_form->n_fields = n_fields;

	cdw_form->field_widget_types = malloc((size_t) cdw_form->n_fields * sizeof (int));
	if (cdw_form->field_widget_types ==  NULL) {
		cdw_vdm ("ERROR: failed to allocate fields widget ids table\n");
		free(cdw_form);
		cdw_form = (cdw_form_t *) NULL;
		return (cdw_form_t *) NULL;
	}

	cdw_form->window = (WINDOW *) NULL;
	cdw_form->subwindow = (WINDOW *) NULL;
	cdw_form->form = (FORM *) NULL;
	cdw_form->fields = (FIELD **) NULL;

	cdw_form->form_id = 0;
	cdw_form->n_return_keys = 0;
	for (int i = 0; i < N_RETURN_KEYS_MAX; i++) {
		cdw_form->return_keys[i] = 0;
	}

	cdw_form->handle_enter = (void (*)(cdw_form_t *, int, void *)) NULL;

	cdw_form->get_dropdown_from_form = cdw_form_driver_get_dropdown_from_form;
	cdw_form->get_button_from_form = cdw_form_driver_get_button_from_form;
	cdw_form->focus_off = cdw_form_driver_focus_off;
        cdw_form->focus_on = cdw_form_driver_focus_on;
	cdw_form->toggle_focus = cdw_form_driver_toggle_focus;
	cdw_form->toggle_checkbox = cdw_form_driver_toggle_checkbox;
	cdw_form->is_checkbox_field_index = is_checkbox_field_index;
	cdw_form->is_dropdown_field_index = is_dropdown_field_index;
	cdw_form->is_button_field_index = is_button_field_index;
	cdw_form->is_input_field_index = is_input_field_index;

	return cdw_form;
}





/**
   \brief cdw form destructor - stage 2
*/
void cdw_form_delete(cdw_form_t **cdw_form)
{
	cdw_assert (cdw_form != (cdw_form_t **) NULL, "ERROR: incorrect argument\n");
	if (*cdw_form == (cdw_form_t *) NULL) {
		cdw_vdm ("WARNING: passing NULL form to function\n");
		return;
	}
	cdw_assert ((*cdw_form)->window == (WINDOW *) NULL, "ERROR: cdw_form->window is not deallocated\n");
	cdw_assert ((*cdw_form)->subwindow == (WINDOW *) NULL, "ERROR: cdw_form->window is not deallocated\n");

	if ((*cdw_form)->field_widget_types != (int *) NULL) {
		free((*cdw_form)->field_widget_types);
		(*cdw_form)->field_widget_types = (int *) NULL;
	}

	free(*cdw_form);
	*cdw_form = (cdw_form_t *) NULL;

	return;
}





/**
   \brief cdw form destructor - stage 1

   Variable of type cdw_form_t needs to be deallocated
   in stages, this is first of them;
   second stage is cdw_form_delete()
*/
void cdw_form_delete_form_objects(cdw_form_t *cdw_form)
{
	if (cdw_form == (cdw_form_t *) NULL) {
		cdw_vdm ("WARNING: passing NULL form to function\n");
		return;
	}
	/* from ncurses docs: "The functions free_field() and
	   free_form are available to free field and form objects.
	   It is an error to attempt to free a field connected to
	   a form, but not vice-versa; thus, you will generally
	   free your form objects first. */
	if (cdw_form->form != (FORM *) NULL) {
		unpost_form(cdw_form->form);
		free_form(cdw_form->form);
		cdw_form->form = (FORM *) NULL;
	}

	for (int i = 0; i < cdw_form->n_fields; i++) {
		if (cdw_form->fields[i] != (FIELD *) NULL) {
			free_field(cdw_form->fields[i]);
			cdw_form->fields[i] = (FIELD *) NULL;
		}
	}

	return;
}





int cdw_form_driver(cdw_form_t *cdw_form, int initial_fi, void *data)
{
	cdw_assert (cdw_form->handle_enter != (void (*)(cdw_form_t *, int, void *)) NULL,
		    "ERROR: this form has no \"handle enter\" set up\n");

	cdw_form_driver_go_to_field(cdw_form->form, initial_fi, cdw_form->n_fields);
	initial_fi = field_index(current_field(cdw_form->form));
	cdw_form->focus_on(cdw_form, initial_fi);

	form_driver(cdw_form->form, REQ_END_LINE);
	wrefresh(cdw_form->subwindow);

	int key = KEY_END; /* safe initial value */
	do {
		wrefresh(cdw_form->subwindow);
		key = wgetch(cdw_form->subwindow);

		if (cdw_form_is_movement_key(key)) {
			int fi = field_index(current_field(cdw_form->form));
			cdw_form->focus_off(cdw_form, fi);
		}

		if (key == KEY_HOME) {
			form_driver(cdw_form->form, REQ_BEG_LINE);
		} else if (key == KEY_END) {
			form_driver(cdw_form->form, REQ_END_LINE);
		} else if (key == KEY_LEFT) {
			form_driver(cdw_form->form, REQ_PREV_CHAR);
		} else if (key == KEY_RIGHT) {
			form_driver(cdw_form->form, REQ_NEXT_CHAR);
		} else if (key == KEY_DOWN || key == CDW_KEY_TAB) {
			form_driver(cdw_form->form, REQ_NEXT_FIELD);
			form_driver(cdw_form->form, REQ_END_LINE);
		} else if (key == KEY_UP || key == KEY_BTAB) {
			form_driver(cdw_form->form, REQ_PREV_FIELD);
			form_driver(cdw_form->form, REQ_END_LINE);
		} else if (key == KEY_BACKSPACE) {
			form_driver(cdw_form->form, REQ_DEL_PREV);
		} else if (key == KEY_DC) {
			form_driver(cdw_form->form, REQ_DEL_CHAR);
		} else if (key == CDW_KEY_ENTER) {
			int fi = field_index(current_field(cdw_form->form));
			cdw_form->handle_enter(cdw_form, fi, data);
		} else {
			int fi = field_index(current_field(cdw_form->form));
			if (cdw_form->is_checkbox_field_index(cdw_form, fi)) {
				if (key == ' ' || key == 'x' || key == 'X') {
					cdw_form->toggle_checkbox(cdw_form, fi);
				}
			} else if (cdw_form->is_dropdown_field_index(cdw_form, fi)) {
				/* dropdown in its initial state reacts only
				   to enter, and this has already been handled
				   elsewhere by cdw_form->handle_enter() */
				;
			} else {
				/* normal text/input field, pass char from keyboard to field */
				form_driver(cdw_form->form, key);
			}
		}
		if (cdw_form_is_movement_key(key)) {
			int fi = field_index(current_field(cdw_form->form));
			cdw_form->focus_on(cdw_form, fi);
		}

	} while (key != CDW_KEY_ESCAPE && !cdw_form_is_return_key(cdw_form, key));

	form_driver(cdw_form->form, REQ_VALIDATION);
	curs_set(0); /* turn cursor off */
	return key;
}





bool cdw_form_is_return_key(cdw_form_t *form, int key)
{
	for (int i = 0; i < N_RETURN_KEYS_MAX; i++) {
		if (form->return_keys[i] == 0) {
			break;
		} else if (form->return_keys[i] == key) {
			return true;
		} else {
			;
		}
	}
	return false;
}





void cdw_form_add_return_char(cdw_form_t *form, int key)
{
	cdw_assert (form != (cdw_form_t *) NULL, "ERROR: \"form\" argument is NULL\n");
	cdw_assert (form->n_return_keys < N_RETURN_KEYS_MAX,
		    "ERROR: there are already %d / %d return keys in the form, can't add another one\n",
		    form->n_return_keys, N_RETURN_KEYS_MAX);
	cdw_assert (key != 0, "ERROR: trying to add key '0', but '0' is an initializer value\n");

#ifndef NDEBUG
	if (cdw_form_is_return_key(form, key)) {
		cdw_vdm ("WARNING: adding key %d / \"%s\", but it is already on the list of return keys\n",
			 key, cdw_ncurses_key_label(key));
	}
#endif

	form->return_keys[form->n_return_keys++] = key;

	return;
}





cdw_rv_t cdw_form_description_to_fields(cdw_form_descr_t descr[], void *cdw_form)
{
	cdw_form_t *form = (cdw_form_t *) cdw_form;
	FIELD **fields = form->fields;
	/* "<=" - to set proper value of guard in fields table */
	for (int i = 0; i <= form->n_fields; i++) {
		fields[i] = (FIELD *) NULL;
	}

	cdw_assert (form->field_widget_types != (int *) NULL, "ERROR: table of widget ids is NULL\n");

	for (int i = 0; i < form->n_fields; i++) {
		int widget_type = descr[i].widget_type;
		cdw_assert (i == descr[i].field_enum, "ERROR: order of items! field iterator (%d) != field enum (%d)\n", i, descr[i].field_enum);
		if (widget_type == -1) { /* guard in form description table */
			cdw_assert (0, "ERROR: reached descr guard, so loop condition is incorrect; i = %d, n_fields = %d\n", i, form->n_fields);
			break;
		} else if (widget_type == CDW_WIDGET_LABEL) {
			cdw_vdm ("INFO: creating field #%d: label with string = \"%s\"\n", i, (char *) descr[i].data1);
			fields[i] = cdw_label_field_new(descr[i].n_cols,
							descr[i].begin_y, descr[i].begin_x,
							(char *) descr[i].data1);
		} else if (widget_type == CDW_WIDGET_CHECKBOX) {
			cdw_vdm ("INFO: creating field #%d: checkbox\n", i);
			fields[i] = cdw_checkbox_field_new(descr[i].begin_y,
							   descr[i].begin_x,
							   (const char *) ((int) descr[i].data2 == 1 ? "X" : " "));

		} else if (widget_type == CDW_WIDGET_INPUT) {
			cdw_vdm ("INFO: creating field #%d: input with initial string = \"%s\"\n", i, (char *) descr[i].data1);
			fields[i] = cdw_ncurses_new_input_field((size_t) descr[i].n_lines,
								descr[i].n_cols,
								descr[i].begin_y,
								descr[i].begin_x,
								(char *) descr[i].data1,
								descr[i].data2,
								CDW_NCURSES_INPUT_NONE,
								CDW_COLORS_INPUT);
		} else if (widget_type == CDW_WIDGET_BUTTON) {
			cdw_vdm ("INFO: creating field #%d: button\n", i);
			fields[i] = cdw_ncurses_new_input_field((size_t) descr[i].n_lines,
								descr[i].n_cols,
								descr[i].begin_y,
								descr[i].begin_x,
								(char *) NULL,
								descr[i].data2,
								CDW_NCURSES_INPUT_NONE,
								CDW_COLORS_INPUT);
			/* to avoid situation when user presses some
			   alphanumeric key and the key is entered
			   into field under a button */
			field_opts_off(fields[i], O_EDIT);

			int rv = set_field_userptr(fields[i], (void *) descr[i].data1);
			cdw_assert (rv == E_OK, "ERROR: can't set userptr for field #%d\n", i);
		} else if (widget_type == CDW_WIDGET_DROPDOWN) {
			cdw_vdm ("INFO: creating field #%d: dropdown\n", i);
			fields[i] = cdw_ncurses_new_input_field((size_t) descr[i].n_lines,
								descr[i].n_cols,
								descr[i].begin_y,
								descr[i].begin_x,
								(char *) NULL,
								descr[i].data2,
								CDW_NCURSES_INPUT_NONE,
								CDW_COLORS_INPUT);
			int rv = set_field_userptr(fields[i], (void *) descr[i].data1);
			cdw_assert (rv == E_OK, "ERROR: can't set userptr, i = %d\n", i);
		} else {
			cdw_vdm ("ERROR: unknown widget type: %d (field #%d)\n", widget_type, i);
		}

		form->field_widget_types[i] = widget_type;

		if (fields[i] == (FIELD *) NULL) {
			cdw_vdm ("ERROR: failed to initialize field #%d\n", i);
			return CDW_GEN_ERROR;
		}
	}

	return CDW_OK;
}





bool cdw_form_is_movement_key(int key)
{
	if (key == KEY_DOWN
	    || key == CDW_KEY_TAB
	    || key == KEY_UP
	    || key  == KEY_BTAB
	    || key == KEY_BACKSPACE) { /* in some configurations backspace
					  may be a movement key, i.e. when
					  cursor is at the beginning of input field */
		return true;
	} else {
		return false;
	}
}





/**
   \brief Create new form

   Create new form, do this with proper assignment of its window, subwindow
   and form. Function does full clean up and returns NULL in case of failure.
   Otherwise pointer to properly created form is returned.

   Since "Every form has an associated pair of curses windows" (quote from man
   pages for set_form_win() and set_form_sub()), you have to create two
   windows (main window and subwindow) before calling this function.

   \param window - window in which form will be placed
   \param subwindow - subwindow of the form
   \param field - set of fields to be put in the form

   \return form on success
   \return NULL on failure
*/
FORM *cdw_form_new_form(WINDOW *window, WINDOW *subwindow, FIELD *field[])
{
	cdw_assert (window != (WINDOW *) NULL, "ERROR: function called with window == NULL\n");
	cdw_assert (subwindow != (WINDOW *) NULL, "ERROR: function called with subwindow == NULL\n");
	cdw_assert (field != (FIELD **) NULL, "ERROR: function called with fields array == NULL\n");

	FORM *form = new_form(field);
	if (form == (FORM *) NULL) {
#ifndef NDEBUG
		int e = errno;
		cdw_vdm ("ERROR: failed to create form with new_form(), error code is \"%s\"\n",
			 cdw_ncurses_error_string(e));
#endif
		return (FORM *) NULL;
	}
	int r;
	r = set_form_win(form, window);
	if (r != E_OK) {
		cdw_vdm ("ERROR: failed to set form window with set_form_win(), error code is \"%s\"\n", cdw_ncurses_error_string(r));
		free_form(form);
		form = (FORM *) NULL;
		return (FORM *) NULL;
	}
	r = set_form_sub(form, subwindow);
	if (r != E_OK) {
		cdw_vdm ("ERROR: failed to set form subwindow with set_form_sub(), error code is \"%s\"\n", cdw_ncurses_error_string(r));
		free_form(form);
		form = (FORM *) NULL;
		return (FORM *) NULL;
	}
	r = post_form(form);
	if (r != E_OK) {
		cdw_vdm ("ERROR: failed to post form with post_form(), error code is \"%s\"\n", cdw_ncurses_error_string(r));
		free_form(form);
		form = (FORM *) NULL;
		return (FORM *) NULL;
	}

	return form;
}





/**
   \brief Get 0/1 state associated with given field that is a checkboxes

   Check content of field's buffer (if it is "X" or not) and return 0 or 1.

   The field must be configured and used as a checkbox (field of sizes 1x1),
   because only first character of field's buffer is checked.

   \param field - field that you want to inspect

   \return true if given field's buffer contains "X" string
   \return false if given field's buffer does not contain "X" string
*/
bool cdw_form_get_field_bit(FIELD *field)
{
	cdw_assert (field != (FIELD *) NULL, "ERROR: field argument passed for examination is NULL\n");
	return !strncmp(field_buffer(field, 0), "X", 1) ? true : false;
}





/**
   \brief Get string from given field's buffer

   Function removes trailing white chars from te string before returning it.

   Function returns empty string ("") if given field's buffer is NULL.

   \param field - field that you want to inspect

   \return string from field's buffer, may be empty string ("")
*/
char *cdw_form_get_field_string(FIELD *field)
{
	assert(field != (FIELD *) NULL);

	char *s = field_buffer(field, 0);
	if (s == (char *) NULL) {
		return empty_string;
	} else {
		cdw_vdm ("INFO: returning field string \"%s\"\n", s);
		return cdw_string_rtrim(s);
	}

}





/* this function is called when a movement key has been detected;
   this function checks if current focus is on a widget that
   implements highlighted/non-highlighted states, and if it is,
   then highlight of current widget is turned off */
void cdw_form_driver_focus_off(cdw_form_t *cdw_form, int fi)
{
	cdw_form_t *form = (cdw_form_t *) cdw_form;
	form->toggle_focus(cdw_form, fi, true);
	return;
}





/* this function is called when a movement key has been detected;
   this function checks if current focus is on a widget that
   implements highlighted/non-highlighted states, and if it is,
   then highlight of current widget is turned off */
void cdw_form_driver_focus_on(cdw_form_t *cdw_form, int fi)
{
	cdw_form_t *form = (cdw_form_t *) cdw_form;
	form->toggle_focus(cdw_form, fi, false);
	return;
}





/* this function is called when a movement key has been detected;
   this function checks if current focus is on a widget that
   implements highlighted/non-highlighted states, and if it is,
   then highlight of current widget is turned off */
void cdw_form_driver_toggle_focus(cdw_form_t *cdw_form, int fi, bool off)
{
	cdw_assert (fi < cdw_form->n_fields,
		    "ERROR: fi is larger than number of fields: %d / %d\n",
		    fi, cdw_form->n_fields);

	if (cdw_form->is_button_field_index(cdw_form, fi)) {
		CDW_BUTTON *button = cdw_form->get_button_from_form(cdw_form, fi);
		if (button != (CDW_BUTTON *) NULL) {
			void (*function)(CDW_BUTTON *) = (void (*)(CDW_BUTTON *)) NULL;
			if (off) {
				function = cdw_button_unfocus;
				/* moving away from button, display blinking cursor */
				curs_set(1);
			} else {
				function = cdw_button_focus;
				/* moving into button, don't display blinking cursor */
				curs_set(0);
			}
			function(button);
		}
	} else if (cdw_form->is_dropdown_field_index(cdw_form, fi)) {
		CDW_DROPDOWN *dropdown = cdw_form_driver_get_dropdown_from_form(cdw_form, fi);
		if (dropdown != (CDW_DROPDOWN *) NULL) {
			void (*function)(CDW_DROPDOWN *) = (void (*)(CDW_DROPDOWN *)) NULL;
			if (off) {
				function = cdw_dropdown_unfocus;
				/* moving away from dropdown, display blinking cursor */
				curs_set(1);
			} else {
				function = cdw_dropdown_focus;
				/* moving into dropdown, don't display blinking cursor */
				curs_set(0);
			}
			function(dropdown);
		}
	} else if (cdw_form->is_checkbox_field_index(cdw_form, fi)
		   || cdw_form->is_input_field_index(cdw_form, fi)) {
		if (off) {
			/* moving away from checkbox/input, don't display blinking cursor */
			curs_set(0);
		} else {
			/* moving away from checkbox/input, display blinking cursor */
			curs_set(1);
		}
	} else {
		;
	}

	//form_driver(cdw_form->form, REQ_INS_MODE);
	wrefresh(cdw_form->subwindow);

	return;
}





CDW_DROPDOWN *cdw_form_driver_get_dropdown_from_form(cdw_form_t *cdw_form, int fi)
{
	cdw_assert (fi < cdw_form->n_fields,
		    "ERROR: fi is larger than number of fields: %d / %d\n",
		    fi, cdw_form->n_fields);
	FIELD *field = cdw_form->fields[fi];
	cdw_assert (field != (FIELD *) NULL, "ERROR field %d is NULL\n", fi);

	CDW_DROPDOWN *dropdown = (CDW_DROPDOWN *) NULL;
	if (cdw_form->field_widget_types[fi] == CDW_WIDGET_DROPDOWN) {
		dropdown = (CDW_DROPDOWN *) field_userptr(field);
		cdw_assert (dropdown != (CDW_DROPDOWN *) NULL, "ERROR: field user pointer is NULL dropdown (fi = %d)\n", fi);
	}

	return dropdown;
}





CDW_BUTTON *cdw_form_driver_get_button_from_form(cdw_form_t *cdw_form, int fi)
{
	cdw_assert (fi < cdw_form->n_fields,
		    "ERROR: fi is larger than number of fields: %d / %d\n",
		    fi, cdw_form->n_fields);
	FIELD *field = cdw_form->fields[fi];
	cdw_assert (field != (FIELD *) NULL, "ERROR field %d is NULL\n", fi);

	CDW_BUTTON *button = (CDW_BUTTON *) NULL;
	if (cdw_form->field_widget_types[fi] == CDW_WIDGET_BUTTON) {
		button = (CDW_BUTTON *) field_userptr(field);
		cdw_assert (button != (CDW_BUTTON *) NULL, "ERROR: field user pointer is NULL button (fi = %d)\n", fi);
	}

	return button;
}





bool is_checkbox_field_index(cdw_form_t *cdw_form, int fi)
{
	cdw_assert (fi < cdw_form->n_fields,
		    "ERROR: fi is larger than number of fields: %d / %d\n",
		    fi, cdw_form->n_fields);
	return cdw_form->field_widget_types[fi] == CDW_WIDGET_CHECKBOX ? true : false;
}





bool is_dropdown_field_index(cdw_form_t *cdw_form, int fi)
{
	cdw_assert (fi < cdw_form->n_fields,
		    "ERROR: fi is larger than number of fields: %d / %d\n",
		    fi, cdw_form->n_fields);
	return cdw_form->field_widget_types[fi] == CDW_WIDGET_DROPDOWN ? true : false;
}





bool is_button_field_index(cdw_form_t *cdw_form, int fi)
{
	cdw_assert (fi < cdw_form->n_fields,
		    "ERROR: fi is larger than number of fields: %d / %d\n",
		    fi, cdw_form->n_fields);
	return cdw_form->field_widget_types[fi] == CDW_WIDGET_BUTTON ? true : false;
}





bool is_input_field_index(cdw_form_t *cdw_form, int fi)
{
	cdw_assert (fi < cdw_form->n_fields,
		    "ERROR: fi is larger than number of fields: %d / %d\n",
		    fi, cdw_form->n_fields);
	return cdw_form->field_widget_types[fi] == CDW_WIDGET_INPUT ? true : false;
}





/**
   \brief Toggle state of checkbox

   Toggle state of checkbox that is placed in field with index\p fi.
   Make sure that this is really a checkbox before calling the function.

   Additionally the function shows/hides widgets in "tools" page as
   state of related checkbox changes.

   \param cdw_form - form in which a checkbox is placed
   \param fi - field index, index of field of form, that is related to the checkbox
*/
void cdw_form_driver_toggle_checkbox(cdw_form_t *cdw_form, int fi)
{
	cdw_assert (cdw_form->field_widget_types[fi] == CDW_WIDGET_CHECKBOX, "ERROR: called the function for non-checkbox\n");
	cdw_assert (fi < cdw_form->n_fields,
		    "ERROR: fi is larger than number of fields: %d / %d\n",
		    fi, cdw_form->n_fields);
	if (cdw_form_get_field_bit(cdw_form->fields[fi])) {
		set_field_buffer(cdw_form->fields[fi], 0, " ");
	} else {
		set_field_buffer(cdw_form->fields[fi], 0, "X");
	}
	return;
}





void cdw_form_driver_go_to_field(FORM *form, int fi, int n_fields)
{
	if (fi < 0 || fi >= n_fields) {
		cdw_vdm ("WARNING: fi = %d out of bounds (n_fields = %d)\n", fi, n_fields);
		fi = 0;
	}

	int rv = form_driver(form, REQ_FIRST_FIELD);
	if (rv != E_OK) {
		cdw_vdm ("ERROR: form_driver() returns \"%s\"\n", cdw_ncurses_error_string(rv));
	}

	/* initial field may not be first when driver needs to switch
	   to field in which there is an error */
	/* TODO: check why using simple for() loop fails: iterating stops
	   before correct field */
	int i = field_index(current_field(form));
	if (i == fi) {
		return;
	} else {
		while (i < fi && i < n_fields - 1) {
			rv = form_driver(form, REQ_NEXT_FIELD);
			if (rv != E_OK) {
				cdw_vdm ("ERROR: form_driver() returns \"%s\"\n", cdw_ncurses_error_string(rv));
				break;
			} else {
				i = field_index(current_field(form));
			}
		}
	}

	return;
}

