/*
 *      Miscellaneous utilities.
 *
 *      Copyright (c) 2006 Martin Ellis <martin.ellis@kdemail.net>
 *      Copyright (c) 2006-2007 Naoaki Okazaki
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA, or visit
 * http://www.gnu.org/copyleft/gpl.html .
 *
 */

/* $Id: console_posix.c 328 2007-02-10 17:50:11Z nyaochi $ */

/** 
 * @file 
 * Utility functions for the easypmp command line program (mostly
 * display related).
 *
 * @addtogroup cui
 * @{
 */

#ifdef  HAVE_CONFIG_H
#include <config.h>
#endif/*HAVE_CONFIG_H*/

#include <os.h>
#include <stdio.h>
#include <stdlib.h>
#include <pmplib/ucs2char.h>

#ifdef  HAVE_STRING_H
#include <string.h>
#endif/*HAVE_STRING_H*/
#ifdef	HAVE_UNISTD_H
#include <unistd.h>
#endif/*HAVE_UNISTD_H*/

#include "console.h"

#define     MAX(a, b)       ((a) > (b) ? (a) : (b))
#define     MIN(a, b)       ((a) < (b) ? (a) : (b))

#if CAN_GET_WIN_SIZE
#include <sys/ioctl.h>
#include <signal.h>
#endif/*CAN_GET_WIN_SIZE*/

#include "util.h"

#if CAN_GET_WIN_SIZE
/** 
 * The number of characters that can be printed on a single line,
 * without causing a line wrap.  Since the right-most column is
 * required for the cursor, this is one less than the actual terminal
 * width.
 *
 * Defaults to 79 on systems where we can't tell the width of the
 * terminal.
 */
static volatile unsigned short int window_width;
#else
static const unsigned short int window_width = 79;
#endif

/** 
 * The minimum width of the terminal we're willing to entertain.  If
 * the terminal gets narrower than this width, we treat it as this
 * width.  Note that it must be at least 2 to allow for one character
 * and the cursor.
*/
static const int min_term_width = 6;


#define POSSIBLE_TTYS 2
/** 
 * Flags to indicate whether stdin, stdout, and stderr are attached to
 * a terminal.  These are used to determine whether we should check
 * the width of some progress lines before printing them.  Initialised
 * in display_init.
 */
static int fd_is_tty[POSSIBLE_TTYS+1];

#if CAN_GET_WIN_SIZE

/** 
 * Handler for the "terminal window changed size" signal.
 * 
 * @param unused 
 */
static void window_size_changed(int unused)
{
	static struct winsize wsize;
	if (ioctl(1, TIOCGWINSZ, &wsize) != -1) {
		if (wsize.ws_col > min_term_width) {
			window_width = wsize.ws_col - 1;
		} else {
			window_width = min_term_width;
		}
	}
}

#endif/*CAN_GET_WIN_SIZE*/

static int is_tty(FILE *fp)
{
	int fd = fileno(fp);
	if (0 < fd && fd <= POSSIBLE_TTYS && fd_is_tty[fd]) {
		return 1;
	} else {
		return 0;
	}
}

/** 
 * Determines whether stdin, stdout and stderr are associated with a
 * TTY, and update fd_is_tty accordingly.  Also sets up
 * window_size_changed as a signal handler.
 */
int console_init()
{
#ifdef	HAVE_ISATTY
	int i;
	for(i = 0; i <= POSSIBLE_TTYS; ++i)
		fd_is_tty[i] = isatty(i);

#if	CAN_GET_WIN_SIZE
	signal(SIGWINCH,window_size_changed);
	window_size_changed(0);
#endif/*CAN_GET_WIN_SIZE*/
#endif/*HAVE_ISATTY*/
	return 0;
}

/** 
 * Deletes all text on the current line by overwriting it with spaces.
 *
 * A 'blank line' is written to a given file pointer, @p fp.  That is,
 * a number of spaces equal to the current window width is written.
 * This is followed by a carriage return character, to return the
 * cursor to the start of the line.
*/
int console_clearln(FILE *fp)
{
	if (is_tty(fp)) {
		/* fmt needs 4 chars (%, -, s, \r) + 
		   room for window_width as string (max. 65535) +
		   null terminator */
		char fmt[10];
		sprintf(fmt, "%%-%us\r", window_width);
		fprintf(fp, fmt, "");
		return 0;
	} else {
		return 1;
	}
}

/** 
 * Displays a UCS-2 string truncated for the terminal width.
 *
 * Displays as much of a UCS-2 encoded string as will fit on a single
 * line in the terminal, and returning the cursor to the start of the
 * line.  If the terminal is less that the given minimum width, that
 * minimum number of characters is displayed anyway, even if it means
 * the text will wrap onto the next line.
 *
 * If @p fp isn't associated with a terminal, just print the whole line.
 *
 * @param fp FILE* to print on
 * @param line the UCS-2 encoded string to display
 * @param min_width minimum number of characters to print 
 */
int console_println(FILE *fp, const ucs2char_t* line, int min_width)
{
	const int margin = 5;

	if (is_tty(fp)) {
		int x = 0, width = 0;
		const ucs2char_t* p = line;

		while (*p) {
			wchar_t c = (wchar_t)*p;

			if (window_width <= x + margin) {
				int ndotts = MIN(3, window_width - x);
				while (ndotts-- > 0) fputc('.', fp);
				break;
			} else {
				/* I don't understand why fputwc(c, fp); doesn't work... */
				fputucs2c(*p, fp);
			}
			p++;
			x += wcwidth(c);
		}
		fputc('\r', fp);
		return (int)(p - line);
	} else {
		fprints(fp, "%s\n", line);
		return ucs2len(line);
	}
}


/** @} */
