/*
 *      Digital Media Player library.
 *
 *      Copyright (c) 2005-2007 Naoaki Okazaki
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 *
 */

/* $Id: pmp.h 328 2007-02-10 17:50:11Z nyaochi $ */

#ifndef	__PMP_H__
#define	__PMP_H__

#include <pmplib/os_types.h>

#ifdef	PMP_EXPORTS
#define	PMPAPI	__declspec(dllexport)
#else
#define	PMPAPI
#endif/*PMP_EXPORTS*/

#ifdef	__cplusplus
extern "C" {
#endif/*__cplusplus*/


/**
 * @mainpage PMPlib API Documentation
 *
 *	@section intro Introduction
 *	
 *	This is the documentation for the PMPlib API. 
 */

/** 
 * \addtogroup pmp PMPlib APIs
 * @{ 
 *
 *	The PMPlib APIs define the programing interface for both applications
 *	and driver modules.
 */

/*
 * Forward structure declarations.
 */
struct tag_pmp_t;
/** Synonym of struct tag_pmp_t. */
typedef struct tag_pmp_t pmp_t;

struct tag_pmp_music_t;
/** Synonym of struct tag_pmp_music_t. */
typedef struct tag_pmp_music_t pmp_music_t;

/**
 * Status codes.
 */
enum {
	/** Success. */
	PMP_SUCCESS = 0,
	/** Undetermined error. */
	PMPERR_ERROR = 0x80000000,
	/** Device is not found or unsupported. */
	PMPERR_DEVICENOTFOUND,
	/** Insufficient memory. */
	PMPERR_INSUFFICIENTMEMORY,
	/** Operation is not implemented. */
	PMPERR_NOTIMPLIMENTED,
	/** The entry is not found. */
	PMPERR_ENTRYNOTFOUND,
	/** Inconsistent operation/data. */
	PMPERR_INCONSISTENCY,
	/** Error while opening a file for reading. */
	PMPERR_OPENFORREAD,
	/** Error while opening a file for writing. */
	PMPERR_OPENFORWRITE,
	/** Error while reading a file. */
	PMPERR_READ,
	/** Error while writing a file. */
	PMPERR_WRITE,
};

/**
 * Open flags for portable media device.
 *	Specify one or more flags to pmp_t::open() member function.
 */
enum {
	/**
	 * Read access to the music database.
	 *	The member function pmp_t::open() will read the music database.
	 */
	PMPOF_MUSIC_DB_READ		= 0x0001,
	/**
	 * Write access to the music database.
	 *	The member function pmp_t::close() will write the music database.
	 */
	PMPOF_MUSIC_DB_WRITE	= 0x0002,
	/**
	 * Read access to the music playlists.
	 *	The member function pmp_t::open() will read the music playlists.
	 */
	PMPOF_MUSIC_PL_READ		= 0x0004,
	/**
	 * Write access to the music playlists.
	 *	The member function pmp_t::close() will write the music playlists.
	 */
	PMPOF_MUSIC_PL_WRITE	= 0x0008,
};

/**
 * Macro to define FourCC representations of codecs.
 */
#define	PMPFOURCC(a, b, c, d)	\
	((uint32_t)(a) << 24 | (uint32_t)(b) << 16 | (uint32_t)(c) << 8 | (uint32_t)(d))

/** Unknown codec. */
#define	PMPCODEC_NONE			PMPFOURCC(' ',' ',' ',' ')
/** MPEG Audio Layer III. */
#define	PMPCODEC_MPEGLAYER3		PMPFOURCC('M','P','1','3')
/** Windows Media Audio. */
#define	PMPCODEC_WMA			PMPFOURCC('W','M','A',' ')
/** Ogg Vorbis. */
#define	PMPCODEC_VORBIS			PMPFOURCC('O','V','1',' ')
/** Microsoft Riff WAVE. */
#define	PMPCODEC_WAV			PMPFOURCC('W','A','V','E')

/**
 * Flags for device functionality.
 */
enum {
	/** None. */
	PMPF_NONE =			0x0000,
	/** The device supports music database. */
	PMPMF_SUPPORT =		0x0001,
	/** The device supports subdirs under music directory. */
	PMPMF_RECURSIVE =	0x0002,	
	/** The device supports music playlist. */
	PMPPF_SUPPORT =		0x0010,
	/** The device supports subdirs under playlist directory. */
	PMPPF_RECURSIVE =	0x0020,
};


/**
 * @defgroup device PMPlib device API
 * @{ 
 *
 *	PMPlib Device API is the root interface to a driver.
 */

/** Maximum size, in chars, of a device description. */
#define	PMP_DECLSIZE	(128)
/** Unavailable field of a device description. */
#define	PMP_DECLUNAVAIL	("***")

/**
 * PMP device decription.
 *	This structure stores information about a PMP device. A PMP device driver
 *	should fill these fields properly so that an application can obtain the
 *	information about the device. An empty value represents the uncertainity
 *	of the field value at this stage. The value ::PMP_DECLUNAVAIL (\c "***")
 *	implies the unavailability of the field value for the device.
 */
typedef struct {
	/** Device identifier (e.g., "iriver_e10_ums_1.0-1.4"). */
	char	id[PMP_DECLSIZE];
	/** Manufacturer (e.g., "iriver"). */
	char	manufacturer[PMP_DECLSIZE];
	/** Model name (e.g., "E10"). */
	char	name[PMP_DECLSIZE];
	/** Firmware mode (e.g., "UM"). */
	char	mode[PMP_DECLSIZE];
	/** Firmware default language (e.g., "JP"). */
	char	language[PMP_DECLSIZE];
	/** Firmware version (e.g., "1.4"). */
	char	version[PMP_DECLSIZE];
	/** Minimum firmware version (e.g., "1.0"). */
	char	min_version[PMP_DECLSIZE];
	/** Maximum firmware version (e.g., "1.4"). */
	char	max_version[PMP_DECLSIZE];
} pmp_device_description_t;

/**
 * PMP device environment.
 *	This structure stores information about predefined paths, capability, 
 */
typedef struct {
	/** Path to the root directory of the player. */
	ucs2char_t		path_to_root[MAX_PATH];
	/** Relative path to the system directory from the root. */
	ucs2char_t		path_to_system[MAX_PATH];
	/** Relative path to the music directory from the root. */
	ucs2char_t		path_to_music[MAX_PATH];
	/** Relative path to the playlist directory from the root. */
	ucs2char_t		path_to_playlist[MAX_PATH];

	/** Music flags. */
	uint32_t		music_flag;
	/** Playlist flags. */
	uint32_t		playlist_flag;

	/** Number of elements in \a audio_codecs array. */
	uint32_t		num_audio_codecs;
	/** Array of PMPFOURCC values corresponding to the supported audio codecs. */
	uint32_t*		audio_codecs;

	/** Number of elements in \a audio_extensions array. */
	uint32_t		num_audio_extensions;
	/** Array of ucs2char_t string values for audio file extensions. */
	ucs2char_t**	audio_extensions;

	/** Description about the device. */
	const pmp_device_description_t	decl;
} pmp_device_information_t;

/**
 * The root interface for portable media device.
 *
 *	This structure represents the basic interface that is common to any
 *	portal media devices. It defines several member variables and pure
 *	virtual functions to access the implementation for the target device.
 *	Use pmplib_create() function to obtain the instance suitable for a
 *	specific device.
 *
 *	@see	pmplib_create
 */
struct tag_pmp_t {
	/**
	 * Pointer for the internal use.
	 *
	 *	This member variable is reserved for the internal use of the library.
	 *	Do not modify the value.
	 */
	void* instance;

	/**
	 * Reference counter.
	 *
	 *	This member variable is reserved for the library to manage the
	 *	reference counter. Do not modify the value directly. Call
	 *	member functions add_ref() and release() to increase and
	 *	decrease the reference counter.
	 */
	uint32_t ref_count;

	/**
	 * Open flags.
	 *
	 *	This member variable is reserved for the library to store the
	 *	flags specified in open() function. Do not modify the value.
	 */
	uint32_t flag;

	/**
	 * PMP device decription.
	 */
	pmp_device_information_t info;

	/**
	 * The pointer to pmp_music_t interface.
	 */
	pmp_music_t* music;

	/**
	 * Increment the reference counter.
	 *	@param	pmp			The pointer to the pmp_t instance.
	 */
	uint32_t (*add_ref)(pmp_t* pmp);

	/**
	 * Decrement the reference counter.
	 *	If the reference counter becomes zero after this decrement,
	 *	this function will destroy the pmp_t instance.
	 *	@param	pmp			The pointer to the pmp_t instance.
	 */
	uint32_t (*release)(pmp_t* pmp);

	/**
	 * Open the PMP device.
	 *	@param	pmp			The pointer to the pmp_t instance.
	 *	@param	flag		The open flags.
	 */
	result_t (*open)(pmp_t* pmp, uint32_t flag);

	/**
	 * Close the PMP device.
	 *	@param	pmp			The pointer to the pmp_t instance.
	 */
	result_t (*close)(pmp_t* pmp);
};

/** 
 * @} 
 */

/**
 * @defgroup music PMPlib music API
 * @{ 
 *
 *	PMPlib Music API provides the interface to manage music files in a portable
 *	media device.
 */

/**
 * Structure for a music record.
 */
struct tag_pmp_music_record_t {
	/**
	 * Relative path for the music file from the mount location of the device.
	 *	The path character must be '\\' on all platforms (including POSIX).
	 *	e.g., "\\Music\\Beatles\\AbbeyRoad\\01_AbbeyRoad.mp3"
 	 *	@assert @code filename != NULL @endcode
	 */
	ucs2char_t		*filename;

	/**
	 * Title name.
	 *	e.g., "Come Together"
 	 *	@assert @code title != NULL @endcode
	 */
	ucs2char_t		*title;

	/**
	 * Artist name.
	 *	e.g., "The Beatles"
 	 *	@assert @code artist != NULL @endcode
	 */
	ucs2char_t		*artist;

	/**
	 * Composer.
	 *	e.g., "John Lennon"
 	 *	@assert @code composer != NULL @endcode
	 */
	ucs2char_t		*composer;

	/**
	 * Album name.
	 *	e.g., "Abbey Road"
 	 *	@assert @code album != NULL @endcode
	 */
	ucs2char_t		*album;

	/**
	 * Genre name.
	 *	e.g., "Rock"
 	 *	@assert @code genre != NULL @endcode
	 */
	ucs2char_t		*genre;

	/**
	 * Release date.
	 *	e.g., "1969"
 	 *	@assert @code date != NULL @endcode
	 */
	ucs2char_t		*date;

	/**
	 * FourCC representation of the codec.
	 *	e.g., PMPCODEC_MPEGLAYER3
	 */
	uint32_t		codec;

	/**
	 * Track number.
	 *	e.g., 1
	 */
	uint32_t		track_number;

	/**
	 * Sample rate in [Hz].
	 *	e.g., 44100
	 */
	uint32_t		sample_rate;

	/**
	 * Bitrate in [bps].
	 *	e.g., 128000
	 */
	uint32_t		bitrate;

	/**
	 * Duration in seconds.
	 *	e.g., 260
	 */	
	uint32_t		duration;

	/**
	 * File size in bytes.
	 *	e.g., 4160000
	 */
	uint32_t		filesize;

	/**
	 * Timestamp of the last update of the music file.
	 *	The value is equivalent to \c st.st_mtime after \c stat(&st, filename);
	 */
	uint32_t		ts_update;

	/**
	 * Rating.
	 *	e.g., 5
	 */
	uint32_t		rating;

	/**
	 * Play count on the device.
	 *	e.g., 23
	 */
	uint32_t		play_count;

	/**
	 * Timestamp of the recent playback on the device.
	 */
	uint32_t		ts_playback;

	/**
	 * timestamp when the track was imported.
	 */
	uint32_t		ts_import;
};

/**
 * Synonym of struct tag_pmp_music_record_t.
 */
typedef struct tag_pmp_music_record_t pmp_music_record_t;

/**
 * Structure for a playlist.
 */
typedef struct {
	/**
	 * Name of the playlist.
	 */
	ucs2char_t *name;

	/**
	 * The number of entries in the playlist.
	 */
	int num_entries;

	/**
	 * Pointer to an array of entries referring to media files.
	 */
	ucs2char_t **entries;
} pmp_playlist_t;

/**
 * Interface for organizing music files.
 *
 *	This interface is responsible for several operations relevant to music
 *	files in a portable media device. The major operations are:
 *	- setting a list of music records for building a music database
 *	- getting a list of music records from the existing music database
 *	- dumping the existing music database in a text format
 *	- setting a list of music files for generating
 *	- getting a list of music playlists used by the device
 *
 *	You cannot create an instance of this structure by yourself. Access
 *	pmp_t::music member to obtain this interface.
 */
struct tag_pmp_music_t {
	/**
	 * Pointer for the internal use.
	 *
	 *	This member variable is reserved for the internal use of the library.
	 *	Do not modify the value.
	 */
	void* instance;

	/**
	 * The pointer to pmp_t interface.
	 *
	 *	@assert
	 *		@code this->pmp->music == this @endcode
	 */
	pmp_t* pmp;

	/**
	 * Set a list of music records.
	 *
	 *	This function sets a list of music records that provide a complete
	 *	information about all music files on a device. Based on this list,
	 *	PMPlib will build a music database on the device.
	 *
	 *	This function does not write out any content on the player immediately
	 *	for the following reason. Some portable devices manage a music database
	 *	and playlists separately, but some integrate playlist information into
	 *	a music database. In the latter case, PMPlib cannot build a music
	 *	database until a definitive list of playlists is obtained. In order to
	 *	support various devices in a unified interface, this function reserves
	 *	a future update of the music database; a database update is prolonged
	 *	until the device is closed by pmp->close() call.
	 *	
	 *	@param	music		The pointer to the pmp_music_t interface.
	 *	@param	records		An array of music records.
	 *	@param	num_records	The number of elements in \a records array.
	 *	@retval	result_t	The status code after this operation.
	 *
	 *	@assert
	 *		@code music != NULL @endcode
	 *		@code records != NULL @endcode
	 */
	result_t (*set_records)(pmp_music_t* music, const pmp_music_record_t* records, uint32_t num_records);

	/**
	 * Get a list of music records.
	 *
	 *	This function gets the list of music records in the existing music
	 *	database or set by \a set_records call.
	 *
	 *	If the argument \a records is NULL, this function returns the number of
	 *	music records to the variable whose address is specified by the
	 *	argument \a num_records. You can use this for determining the size of
	 *	array \a records necessary to obtain all records from this function.
	 *
	 *	If the argument \a records is not NULL, this function copies the
	 *	current music records to the array whose address is specified by the
	 *	argument \a records and whose size is specified by the variable
	 *	pointed by the argument \a num_records.
	 *
	 *	@param	music		The pointer to the pmp_music_t interface.
	 *	@param	records		An array of music records.
	 *	@param	num_records	The pointer to the variable presenting the number
	 *						of elements in \a records array.
	 *	@retval	result_t	The status code after this operation.
	 *
	 *	@assert
	 *		@code music != NULL @endcode
	 *		@code num_records != NULL @endcode
	 */
	result_t (*get_records)(pmp_music_t* music, pmp_music_record_t* records, uint32_t* num_records);

	/**
	 * Set a list of music playlists.
	 *
	 *	This function sets a list of music playlists. PMPlib will generate the
	 *	playlists on the device based on this list.
	 *
	 *	This function does not write out any content on the player immediately;
	 *	playlist generation is prolonged until the device is closed by
	 *	pmp->close() call.
	 *	
	 *	@param	music		The pointer to the pmp_music_t interface.
	 *	@param	playlists	An array of music playlists.
	 *	@param	num_playlists	The number of elements in \a playlists array.
	 *	@retval	result_t	The status code after this operation.
	 *
	 *	@assert
	 *		@code music != NULL @endcode
	 *		@code playlists != NULL @endcode
	 */
	result_t (*set_playlists)(pmp_music_t* music, const pmp_playlist_t* playlists, uint32_t num_playlists);

	/**
	 * Get a list of music playlists.
	 *
	 *	This function gets the list of music playlists in the device or set by
	 *	\a set_playlists call.
	 *
	 *	If the argument \a playlists is NULL, this function returns the number
	 *	of music playlists to the variable whose address is specified by the
	 *	argument \a num_playlists. You can use this for determining the size of
	 *	array \a playlists necessary to obtain all playlists.
	 *
	 *	If the argument \a playlists is not NULL, this function copies the
	 *	current music playlists to the array whose address is specified by the
	 *	argument \a playlists and whose size is specified by the variable
	 *	pointed by the argument \a num_playlists.
	 *
	 *	@param	music		The pointer to the pmp_music_t interface.
	 *	@param	playlists	An array of music playlists.
	 *	@param	num_playlists	The pointer to the variable presenting the
	 *							number of elements in \a playlists array.
	 *	@retval	result_t	The status code after this operation.
	 *
	 *	@assert
	 *		@code music != NULL @endcode
	 *		@code num_playlists != NULL @endcode
	 */
	result_t (*get_playlists)(pmp_music_t* music, pmp_playlist_t* playlists, uint32_t* num_playlists);

	/**
	 * Dump the music database.
	 *
	 *	@param	music		The pointer to the pmp_music_t interface.
	 *	@param	fp			The pointer to an output stream.
	 *	@param	flag		The flags.
	 *	@retval	result_t	The status code after this operation.
	 *
	 *	@assert
	 *		@code music != NULL @endcode
	 */
	result_t (*dump)(pmp_music_t* music, FILE *fp, int flag);
};

/**
 * Initialize pmp_music_record_t structure.
 *
 *	@param	record			The pointer to pmp_music_record_t instance.
 */
PMPAPI void pmplib_record_init(pmp_music_record_t* record);

/**
 * Uninitialize pmp_music_record_t structure.
 *	
 *	This function frees the memory blocks used by the \a record.
 *
 *	@param	record			The pointer to pmp_music_record_t instance.
 */
PMPAPI void pmplib_record_finish(pmp_music_record_t* record);

/**
 * Clone a pmp_music_record_t record.
 *
 *	This function copies the values in the \a src record to \a dst.
 *
 *	@param	dst				The pointer to the destination.
 *	@param	src				The pointer to the source.
 *	@retval	result_t		The status code after this operation.
 */
PMPAPI result_t pmplib_record_clone(pmp_music_record_t* dst, const pmp_music_record_t* src);

/**
 * Uninitialize an array of pmp_music_record_t records.
 *	
 *	This function frees the memory blocks used by each record in the array and
 *	the array.
 *
 *	@param	records			The pointer to pmp_music_record_t array.
 *	@param	num_records		The number of elements in the array.
 */
PMPAPI void pmplib_records_finish(pmp_music_record_t* records, int num_records);

/**
 * Clone a pmp_music_record_t array.
 *
 *	This function copies all records in the \a src array to \a dst.
 *
 *	@param	dst				The destination array.
 *	@param	src				The source array.
 *	@param	num_records		The number of elements in the source array.
 *	@retval	result_t		The status code after this operation.
 */
PMPAPI result_t pmplib_records_clone(pmp_music_record_t* dst, const pmp_music_record_t* src, int num_records);

/**
 * Initialize pmp_playlist_t structure.
 *
 *	@param	playlist		The pointer to pmp_playlist_t instance.
 */
PMPAPI void pmplib_playlist_init(pmp_playlist_t* playlist);

/**
 * Uninitialize pmp_playlist_t structure.
 *	
 *	This function frees the memory blocks used by the \a playlist.
 *
 *	@param	playlist		The pointer to pmp_playlist_t instance.
 */
PMPAPI void pmplib_playlist_finish(pmp_playlist_t* playlist);

/**
 * Clone a playlist.
 *
 *	This function copies the values in the \a src playlist to \a dst.
 *
 *	@param	dst				The pointer to the destination.
 *	@param	src				The pointer to the source.
 */
PMPAPI void pmplib_playlist_clone(pmp_playlist_t* dst, const pmp_playlist_t* src);

/**
 * Uninitialize an array of pmp_playlist_t playlists.
 *	
 *	This function frees the memory blocks used by each playlist in the array and
 *	the array itself.
 *
 *	@param	playlists		The pointer to pmp_playlist_t array.
 *	@param	num_playlists	The number of elements in the array.
 */
PMPAPI void pmplib_playlists_finish(pmp_playlist_t* playlists, int num_playlists);

/**
 * Clone a pmp_playlist_t array.
 *
 *	This function copies all playlists in the \a src array to \a dst.
 *
 *	@param	dst				The destination array.
 *	@param	src				The source array.
 *	@param	num_playlists	The number of elements in the source array.
 */
PMPAPI void pmplib_playlists_clone(pmp_playlist_t* dst, const pmp_playlist_t* src, int num_playlists);

/** 
 * @} 
 */

/** 
 * \addtogroup pmp_export PMPlib export functions for a driver module
 * @{
 *
 *	This API defines the interface that a driver module must implement.
 */

/**
 * Prototype of a callback function receiving device IDs from pmplib_create().
 *
 *	@param	instance		The user-defined instance value.
 *	@param	devid			The device identifier.
 */
typedef void (*pmplib_enumerate_devid_callback_t)(void *instance, const char *devid);

/**
 * Query a driver and construct a ::pmp_t instance.
 *
 *	This function queries the driver suitable for the portable media device
 *	specified by the mount location and/or identifier. A driver module must
 *	return the interface to the driver (::pmp_t instance) if it supports
 *	the device. Parameter \a path_to_device is expected to be specified. If
 *	parameter \a id is not \c NULL, the driver module should return the
 *	interface to the driver with the device identifier specified in \a id
 *	parameter. If the parameter \a id is \c NULL, the driver module should
 *	recognize the model of the device based on the content of the files located
 *	under \a path_to_device.
 *
 *	@param	path_to_device	A UCS-2 string representing the location of the
 *							target device. This parameter cannot be \c NULL.
 *	@param	id				A C-string representing the device identifier of
 *							the target device. If this argument is \c NULL,
 *							this function tries to recognize the model of the
 *							device based on the location (\a path_to_device).
 *	@retval	ptr_pmp			The pointer to the buffer to receive the pointer
 *							to the driver interface (::pmp_t instance).
 *	@retval	result_t		The status code.
 *
 *	@assert
 *		@code ptr_pmp != NULL @endcode
 *		@code path_to_device != NULL @endcode
 */
typedef result_t (*pmplib_create_t)(pmp_t** ptr_pmp, const ucs2char_t* path_to_device, const char *id);

/**
 * Enumerate device identifiers supported by the driver module.
 *	
 *	The driver module should enumerate the device identifiers supported
 *	by the module and invoke the \a callback function for each identifier.
 *
 *	@param	callback		The pointer to the callback function to receive
 *							device identifiers.
 *	@param	instance		A user-defined instance value. The callback
 *							function will receive the same value.
 *	@retval	result_t		The status code.
 *
 *	@assert
 *		@code callback != NULL @endcode
 */
typedef result_t (*pmplib_enumerate_devid_t)(pmplib_enumerate_devid_callback_t callback, void *instance);

/** 
 * @} 
 */

/** 
 * \addtogroup pmp_driver PMPlib miscellaneous routines
 * @{
 *
 *	PMPlib miscellaneous routines provide the utility for a driver module.
 */

/**
 * Increment a variable value and check the resultant value.
 *
 *	This function prevents more than one thread from using
 *	pmplib_interlocked_increment() or pmplib_interlocked_decrement() function
 *	to access the same variable simultaneously.
 *
 *	@param	count			The pointer to the variable whose value is to be
 *							increased.
 *	@retval	uint32_t		The value of the variable after this increment.
 */
PMPAPI uint32_t pmplib_interlocked_increment(uint32_t* count);

/**
 * Decrement a variable value and check the resultant value.
 *
 *	This function prevents more than one thread from using
 *	pmplib_interlocked_increment() or pmplib_interlocked_decrement() function
 *	to access the same variable simultaneously.
 *
 *	@param	count			The pointer to the variable whose value is to be
 *							decreased.
 *	@retval	uint32_t		The value of the variable after this decrement.
 */
PMPAPI uint32_t pmplib_interlocked_decrement(uint32_t* count);

/** 
 * @} 
 */





/**
 * @defgroup pmplib PMPlib helper API
 * @{ 
 *
 *	The PMPlib Helper API provides the utility for choosing a suitable PMPlib
 *	driver from a number of installed drivers. Because the PMPlib project
 *	implements a number of drivers for different portable media devices, an
 *	application must choose a suitable driver that matches to the target
 *	device. By using this API, an application can query a driver suitable for
 *	the portable media device specified by the mount location (root directory
 *	of the device) and/or the string identifier of a driver. An application can
 *	also enumerate the string identifiers of the drivers available in the
 *	system.
 *
 *	An application must call pmplib_init() function to initialize the API
 *	and obtain the pointer to a ::pmplib_t instance that is used for subsequent
 *	calls of the API.
 *	The application can query an interface to a PMPlib driver by using
 *	pmplib_create() function. A list of installed device identifiers is
 *	obtained by receiving callback notifications invoked by
 *	pmplib_enumerate_devid() function. The application must terminate this
 *	API by calling pmplib_finish() function.
 *
 *	This is an example of querying an interface to the driver that matches to
 *	the device connected to the mount point (Win32 path "D:\"):
 *
 *	@code
 * #include <pmplib/ucs2char.h>
 * #include <pmplib/pmp.h>
 * 
 * int main(int argc, char *argv[])
 * {
 *     result_t ret = 0;
 *     pmp_t* pmp = NULL;
 *     pmplib_t* pmplib = NULL;
 *     static const ucs2char_t path[] = {'D',':','\\',0};
 *
 *     // Initialize the PMPlib Helper API.
 *     if ((ret = pmplib_init(&pmplib)) != 0) {
 *         // Failed to initialize the API.
 *         goto error_exit;
 *     }
 *
 *     // Query an interface to the PMPlib driver suitable for the path.
 *     if ((ret = pmplib_create(pmplib, &pmp, path, NULL)) != 0) {
 *         // Failed to find a suitable driver.
 *         goto error_exit;
 *     }
 *     
 *     // Processing with the driver.
 *     ...
 *
 *     // Release the driver.
 *     pmp->release(pmp);
 *
 *     // Terminate the helper API.
 *     pmplib_finish(pmplib);
 *     return 0;
 *
 * error_exit:
 *     // Error handling.
 *     ...
 *     return 1;
 * }
 *	@endcode
 *
 *	@{
 */

struct tag_pmplib_t;

/**
 * The structure for the PMPlib Helper API.
 *	An application should use a pointer to this structure as a handle value
 *	for this API. Call pmplib_init() function to obtain a pointer to this
 *	structure and pmplib_finish() function to terminate this API. Access to
 *	a member in this structure is prohibited since it is reserved for the
 *	internal use of the library only.
 */
typedef struct tag_pmplib_t pmplib_t;

/**
 * Initialize the PMPlib helper library.
 *
 *	This function loads the drivers installed in the system to prepare them.
 *	
 *	@retval	ptr_pmplib		The pointer to the buffer to receive the pointer
 *							to the ::pmplib_t instance initialized by this
 *							function.
 *	@retval	result_t		The status code.
 *
 *	@assert
 *		@code ptr_pmplib != NULL @endcode
 *
 *	@note
 *		Call this function before using any other functions in this API. 
 */
PMPAPI result_t pmplib_init(pmplib_t** ptr_pmplib);

/**
 * Uninitialize the PMPlib helper library.
 *
 *	This function detatch the drivers loaded by pmplib_init() function.
 *	
 *	@param	pmplib			The pointer to the ::pmplib_t instance.
 *	@retval	result_t		The status code.
 *
 *	@assert
 *		@code ptr_pmplib != NULL @endcode
 *
 *	@note
 *		Call this function to terminate this API.
 */
PMPAPI result_t pmplib_finish(pmplib_t* pmplib);

/**
 * Query a driver and create a ::pmp_t instance.
 *
 *	This function queries the driver suitable for the portable media device
 *	specified by the mount location and/or identifier, and returns the
 *	interface to the driver (::pmp_t instance). An application must specify
 *	parameter \a path_to_device. If parameter \a id is not \c NULL, this
 *	function obtains the interface to the driver with the device identifier
 *	specified in \a id parameter. If the parameter \a id is \c NULL, this
 *	function tries to recognize the model of the device based on the content
 *	of the files located under \a path_to_device. If successful, the interface
 *	to the driver will be returned in the \a pmp argument.
 *
 *	@param	pmplib			The pointer to the ::pmplib_t instance.
 *	@param	ptr_pmp			The pointer to the buffer to receive the pointer
 *							to the driver interface (::pmp_t instance).
 *	@param	path_to_device	A UCS-2 string representing the location of the
 *							target device. This parameter cannot be \c NULL.
 *	@param	id				A C-string representing the device identifier of
 *							the target device. If this argument is \c NULL,
 *							this function tries to recognize the model of the
 *							device based on the location (\a path_to_device).
 *	@retval	result_t		The status code.
 *
 *	@assert
 *		@code pmplib != NULL @endcode
 *		@code ptr_pmp != NULL @endcode
 *		@code path_to_device != NULL @endcode
 */
PMPAPI result_t pmplib_create(pmplib_t* pmplib, pmp_t** ptr_pmp, const ucs2char_t* path_to_device, const char *id);

/**
 * Enumerate device identifiers of the installed drivers.
 *	
 *	This function enumerates the device identifiers of the installed drivers,
 *	and invokes the \a callback function for each identifier.
 *
 *	@param	pmplib			The pointer to the ::pmplib_t instance.
 *	@param	callback		The pointer to the callback function to receive
 *							device identifiers.
 *	@param	instance		A user-defined instance value. The callback
 *							function will receive the same value.
 *	@retval	result_t		The status code.
 *
 *	@assert
 *		@code pmplib != NULL @endcode
 *		@code callback != NULL @endcode
 */
PMPAPI result_t pmplib_enumerate_devid(pmplib_t* pmplib, pmplib_enumerate_devid_callback_t callback, void *instance);

/** 
 * @} 
 */

/** 
 * @} 
 */


#ifdef	__cplusplus
}
#endif/*__cplusplus*/

#endif/*__PMP_H__*/

