/*
 * Copyright (c) 1997 Shigio Yamaguchi. All rights reserved.
 * Copyright (c) 1999 Tama Communications Corporation. All rights reserved.
 * Copyright (c) 2006 Martin Ellis <martin.ellis@kdemail.net>.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * @file
 *
 * @addtogroup filepath
 * @{
 */

#ifdef	HAVE_CONFIG_H
#include <config.h>
#endif/*HAVE_CONFIG_H_*/

#include <os.h>
#include <errno.h>
#include <stdlib.h>
#include <pmplib/ucs2char.h>

#include "rel2abs.h"

/** 
 * Convert an relative path name into absolute path name.
 *
 * If the given @p path is already an absolute path, and the @p result
 * buffer is large enough, then the absolute path is used as the
 * @p result.
 *
 * On error, returns nil and sets errno to one of the following
 * values:
 * - EINVAL - the base directory was not an absolute path (did not
 *            begin with a '/'), or @p size was 0;
 * - ERANGE - the @p size of the @p result buffer was too small.
 *
 * @param path relative path to convert.
 * @param base base directory (must be absolute path).
 * @param result result buffer.
 * @param size size of result buffer.
 * 
 * @return non-nil on success, nil if there was an error.
 */
ucs2char_t* rel2abs(const ucs2char_t *path, const ucs2char_t *base,
                    ucs2char_t *result, const size_t size)
{
	static const ucs2char_t ucs2cs_dot[] = {'.',0};
	static const ucs2char_t ucs2cs_dot_dot[] = {'.','.',0};
	static const ucs2char_t ucs2cs_dot_slash[] = {'.','/',0};
	static const ucs2char_t ucs2cs_dot_dot_slash[] = {'.','.','/',0};
	const ucs2char_t *pp, *bp;
	/*
	 * endp points the last position which is safe in the result buffer.
	 */
	const ucs2char_t *endp = result + size - 1;
	ucs2char_t *rp;
	int length;

	if (*path == '/') {
		/* if the given path is absolute, just use that as the result */
		if (ucs2len(path) >= size)
			goto erange;
		ucs2cpy(result, path);
		goto finish;
	} else if (*base != '/' || !size) {
		errno = EINVAL;
		return (NULL);
	} else if (size == 1)
		goto erange;

	length = ucs2len(base);

	if (!ucs2cmp(path, ucs2cs_dot) || !ucs2cmp(path, ucs2cs_dot_slash)) {
		if (length >= size)
			goto erange;
		ucs2cpy(result, base);
		/*
		 * rp points the last char.
		 */
		rp = result + length - 1;
		/*
		 * remove the last '/'.
		 */
		if (*rp == '/') {
			if (length > 1)
				*rp = 0;
		} else
			rp++;
		/* rp point NULL char */
		if (*++path == '/') {
			/*
			 * Append '/' to the tail of path name.
			 */
			*rp++ = '/';
			if (rp > endp)
				goto erange;
			*rp = 0;
		}
		goto finish;
	}
	bp = base + length;
	if (*(bp - 1) == '/')
		--bp;
	/*
	 * up to root.
	 */
	for (pp = path; *pp && *pp == '.'; ) {
		if (!ucs2ncmp(pp, ucs2cs_dot_dot_slash, 3)) {
			pp += 3;
			while (bp > base && *--bp != '/')
				;
		} else if (!ucs2ncmp(pp, ucs2cs_dot_slash, 2)) {
			pp += 2;
		} else if (!ucs2ncmp(pp, ucs2cs_dot_dot, 3)) {
			pp += 2;
			while (bp > base && *--bp != '/')
				;
		} else
			break;
	}
	/*
	 * down to leaf.
	 */
	length = bp - base;
	if (length >= size)
		goto erange;
	ucs2ncpy(result, base, length);
	rp = result + length;
	if (*pp || *(pp - 1) == '/' || length == 0)
		*rp++ = '/';
	if (rp + ucs2len(pp) > endp)
		goto erange;
	ucs2cpy(rp, pp);
finish:
	return result;
erange:
	errno = ERANGE;
	return (NULL);
}
/** @} */
