/*
 *      Low level interface for idx (e.g., U10.idx).
 *
 *      Copyright (c) 2005-2007 Naoaki Okazaki
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 *
 */

/* $Id: idx.c 328 2007-02-10 17:50:11Z nyaochi $ */

#ifdef	HAVE_CONFIG_H
#include <config.h>
#endif/*HAVE_CONFIG_H*/
#ifdef	HAVE_STRING_H
#include <string.h>
#endif/*HAVE_STRING_H*/

#include <os.h>
#include <stdio.h>
#include <stdlib.h>
#include <pmplib/ucs2char.h>
#include <pmplib/pmp.h>

#include "util.h"
#include "serialize.h"
#include "ip2db.h"


void idx_repr(uint8_t* buffer, idxpage_t* pages, FILE *fpo)
{
	uint32_t i;
	uint32_t num_pages;

	/* Read the number of pages. */
	idxheader_setget_num_pages(buffer, &num_pages, 0);

	for (i = 0;i < num_pages;++i) {
		uint8_t *p = &buffer[pages[i].offset];
		g_idx_exports[pages[i].type].dump(p, pages[i].offset, fpo);
	}
}

static int locate_node(uint8_t* block, idx_key_t* idxkey, int* ptr_i, uint32_t* ptr_child)
{
	int ret = -1;
	int i;
	uint16_t n;
	uint32_t child = 0;
	idx_export_t* idxexp = &g_idx_exports[idxkey->type];

	idxexp->setget_num_childlen(block, &n, 0);

	for (i = 0;i < n;++i) {
		idx_key_t x;

		memset(&x, 0, sizeof(x));
		x.type = idxkey->type;
		idxexp->setget_child(block, i, &x, &child, 0);

		ret = idxexp->comp_idxkey(&x, idxkey);
		if (ret >= 0) {
			break;
		}
	}
	if (ptr_i) {
		*ptr_i = i;
	}
	if (ptr_child) {
		*ptr_child = i < n ? child : 0;
	}
	return ret;
}

int idx_first(uint8_t* buffer, uint32_t node, idx_key_t* idxkey, uint32_t* leafid)
{
	idx_export_t* idxexp = &g_idx_exports[idxkey->type];

	for (;;) {
		uint8_t *block = &buffer[0x400 * (node-1)];
		uint8_t height = idxexp->get_height(block);
		uint16_t n;
		idxexp->setget_num_childlen(block, &n, 0);

		/* Fail if a node has no childrens. */
		if (n == 0) {
			return -1;	/* No first element. */
		}

		/* Obtain the first child in this node. */
		idxexp->setget_child(block, 0, idxkey, &node, 0);
		if (height == 0) {
			*leafid = node;
			break;
		}
	}
	return 0;	/* Exit with success. */
}

int idx_next(uint8_t* buffer, uint32_t node, idx_key_t* idxkey, uint32_t* leafid)
{
	int i = 0;
	idx_export_t* idxexp = g_idx_exports + idxkey->type;
	uint8_t *block = &buffer[0x400 * (node-1)];
	uint8_t height = idxexp->get_height(block);
	uint16_t n;
	uint32_t child;
	int match = locate_node(block, idxkey, &i, &child);
	idxexp->setget_num_childlen(block, &n, 0);

	if (height == 0) {
		if (i+1 < n) {
			idxexp->setget_child(block, i+1, idxkey, leafid, 0);
			return 0;
		} else {
			return -1;
		}
	} else {
		int ret = idx_next(buffer, child, idxkey, leafid);
		if (ret < 0) {
			if (i+1 < n) {
				idxexp->setget_child(block, i+1, idxkey, &child, 0);
				idx_first(buffer, child, idxkey, leafid);
				return 0;
			} else {
				return -1;
			}
		} else {
			return 0;
		}
	}
}

result_t idx_get(uint8_t* buffer, idx_key_t* idxkey, uint32_t* leafid, int flag)
{
	int ret;
	uint8_t* p = NULL;
	uint8_t height = 0;
	uint32_t node = 0;
	uint32_t child = 0;

	/* Check key type. */
	if (idxkey->type < IP2DBIDX_PAGE_NODE_ENTRYNUMBER || IP2DBIDX_PAGE_NODE_FORMAT < idxkey->type) {
		return PMPERR_INCONSISTENCY;
	}

	*leafid = 0;

	/* */
	node = idxkey->type;
	for (;;) {
		int i, match;
		p = &buffer[0x400 * (node-1)];
		match = locate_node(p, idxkey, &i, &child);
		if (0 <= match) {
			idxnode_setget_height(p, &height, 0);
			if (height == 0) {
				if (match == 0) {
					*leafid = child;
					ret = 0;
				} else {
					*leafid = 0;
					ret = PMPERR_ENTRYNOTFOUND;
				}
				break;
			} else {
				node = child;
			}
		} else {
			ret = PMPERR_ENTRYNOTFOUND;
			break;	/* Not found. */
		}
	}

	return ret;
}
