/*
 *      B-tree leaf (idx).
 *
 *      Copyright (c) 2005-2007 Naoaki Okazaki
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 *
 */

/* $Id: idx_leaf.c 328 2007-02-10 17:50:11Z nyaochi $ */

#ifdef	HAVE_CONFIG_H
#include <config.h>
#endif/*HAVE_CONFIG_H*/
#ifdef	HAVE_STRING_H
#include <string.h>
#endif/*HAVE_STRING_H*/

#include <os.h>
#include <stdio.h>
#include <stdlib.h>
#include <pmplib/ucs2char.h>

#include "util.h"
#include "serialize.h"
#include "ip2db.h"

void idxleaf_setget_unknown1(uint8_t* block, uint32_t* value, int is_storing)
{
	serialize_uint32be(&block[0x000], value, is_storing);
}

void idxleaf_setget_leaf_prev(uint8_t* block, uint32_t* value, int is_storing)
{
	serialize_uint32be(&block[0x004], value, is_storing);
}

void idxleaf_setget_leaf_next(uint8_t* block, uint32_t* value, int is_storing)
{
	serialize_uint32be(&block[0x008], value, is_storing);
}

void idxleaf_setget_size_avail(uint8_t* block, uint16_t* value, int is_storing)
{
	serialize_uint16be(&block[0x00C], value, is_storing);
}

void idxleaf_setget_size_used(uint8_t* block, uint16_t* value, int is_storing)
{
	serialize_uint16be(&block[0x00E], value, is_storing);
}

void idxleaf_setget_unknown2(uint8_t* block, uint16_t* value, int is_storing)
{
	serialize_uint16be(&block[0x010], value, is_storing);
}

void idxleaf_setget_unknown3(uint8_t* block, uint16_t* value, int is_storing)
{
	serialize_uint16be(&block[0x012], value, is_storing);
}

void idxleaf_setget_num_data(uint8_t* block, uint16_t* value, int is_storing)
{
	serialize_uint16be(&block[0x014], value, is_storing);
}

void idxleaf_setget_leaf_line(uint8_t* block, uint8_t* value, int is_storing)
{
	serialize_uint8(&block[0x016], value, is_storing);
}

void idxleaf_setget_unknown5(uint8_t* block, uint8_t* value, int is_storing)
{
	serialize_uint8(&block[0x3FD], value, is_storing);
}

void idxleaf_setget_unknown6(uint8_t* block, uint16_t* value, int is_storing)
{
	serialize_uint16be(&block[0x3FE], value, is_storing);
}

void idxleaf_setget_data_offset_size(uint8_t* block, int i, offset_size_t* value, int is_storing)
{
	uint8_t* p = &block[0x3FB - sizeof(offset_size_t) * i];
	p -= serialize_uint16be(p, &value->offset, is_storing);
	p -= serialize_uint16be(p, &value->size, is_storing);
}

void idxleaf_setget_data_direct(uint8_t* block, int i, ip2db_idxleaf_data_t* data, int is_storing)
{
	uint8_t* p = NULL;
	offset_size_t offset_size;

	idx_leafdata_free(data);

	idxleaf_setget_data_offset_size(block, i, &offset_size, 0);
	p = &block[offset_size.offset];

	p += serialize_ucs2be_string_var_alloc(p, &data->pathname) * sizeof(ucs2char_t);
	p += sizeof(ucs2char_t);
	p += serialize_ucs2be_string_var_alloc(p, &data->filename) * sizeof(ucs2char_t);
	p += sizeof(ucs2char_t);
	p += serialize_ucs2be_string_var_alloc(p, &data->title) * sizeof(ucs2char_t);
	p += sizeof(ucs2char_t);
	p += serialize_ucs2be_string_var_alloc(p, &data->artist) * sizeof(ucs2char_t);
	p += sizeof(ucs2char_t);
	p += serialize_ucs2be_string_var_alloc(p, &data->album) * sizeof(ucs2char_t);
	p += sizeof(ucs2char_t);
	p += serialize_ucs2be_string_var_alloc(p, &data->genre) * sizeof(ucs2char_t);
	p += sizeof(ucs2char_t);
}

void idxleaf_setget_data(
	uint8_t* block,
	int i,
	offset_size_t* field_access,
	ip2db_idxleaf_data_t* data,
	int is_storing
	)
{
	uint8_t* p = NULL;
	offset_size_t offset_size, *fa = NULL;

	idxleaf_setget_data_offset_size(block, i, &offset_size, 0);

	if (is_storing) {
		fa = &field_access[IP2DBDAT_IDX_PATHNAME];
		p = &block[offset_size.offset + fa->offset];
		serialize_ucs2be_string_fixed(p, data->pathname, fa->size / sizeof(ucs2char_t), is_storing);
		fa = &field_access[IP2DBDAT_IDX_FILENAME];
		p = &block[offset_size.offset + fa->offset];
		serialize_ucs2be_string_fixed(p, data->filename, fa->size / sizeof(ucs2char_t), is_storing);
		fa = &field_access[IP2DBDAT_IDX_TITLE];
		p = &block[offset_size.offset + fa->offset];
		serialize_ucs2be_string_fixed(p, data->title, fa->size / sizeof(ucs2char_t), is_storing);
		fa = &field_access[IP2DBDAT_IDX_ARTIST];
		p = &block[offset_size.offset + fa->offset];
		serialize_ucs2be_string_fixed(p, data->artist, fa->size / sizeof(ucs2char_t), is_storing);
		fa = &field_access[IP2DBDAT_IDX_ALBUM];
		p = &block[offset_size.offset + fa->offset];
		serialize_ucs2be_string_fixed(p, data->album, fa->size / sizeof(ucs2char_t), is_storing);
		fa = &field_access[IP2DBDAT_IDX_GENRE];
		p = &block[offset_size.offset + fa->offset];
		serialize_ucs2be_string_fixed(p, data->genre, fa->size / sizeof(ucs2char_t), is_storing);
	} else {
		idx_leafdata_free(data);

		fa = &field_access[IP2DBDAT_IDX_PATHNAME];
		p = &block[offset_size.offset + fa->offset];
		data->pathname = ucs2calloc(fa->size + sizeof(ucs2char_t));
		if (!data->pathname) {
			goto error_exit;
		}
		serialize_ucs2be_string_fixed(p, data->pathname, fa->size / sizeof(ucs2char_t), is_storing);
		fa = &field_access[IP2DBDAT_IDX_FILENAME];
		p = &block[offset_size.offset + fa->offset];
		data->filename = ucs2calloc(fa->size + sizeof(ucs2char_t));
		if (!data->filename) {
			goto error_exit;
		}
		serialize_ucs2be_string_fixed(p, data->filename, fa->size / sizeof(ucs2char_t), is_storing);
		fa = &field_access[IP2DBDAT_IDX_TITLE];
		p = &block[offset_size.offset + fa->offset];
		data->title = ucs2calloc(fa->size + sizeof(ucs2char_t));
		if (!data->title) {
			goto error_exit;
		}
		serialize_ucs2be_string_fixed(p, data->title, fa->size / sizeof(ucs2char_t), is_storing);
		fa = &field_access[IP2DBDAT_IDX_ARTIST];
		p = &block[offset_size.offset + fa->offset];
		data->artist = ucs2calloc(fa->size + sizeof(ucs2char_t));
		if (!data->artist) {
			goto error_exit;
		}
		serialize_ucs2be_string_fixed(p, data->artist, fa->size / sizeof(ucs2char_t), is_storing);
		fa = &field_access[IP2DBDAT_IDX_ALBUM];
		p = &block[offset_size.offset + fa->offset];
		data->album = ucs2calloc(fa->size + sizeof(ucs2char_t));
		if (!data->album) {
			goto error_exit;
		}
		serialize_ucs2be_string_fixed(p, data->album, fa->size / sizeof(ucs2char_t), is_storing);
		fa = &field_access[IP2DBDAT_IDX_GENRE];
		p = &block[offset_size.offset + fa->offset];
		data->genre = ucs2calloc(fa->size + sizeof(ucs2char_t));
		if (!data->genre) {
			goto error_exit;
		}
		serialize_ucs2be_string_fixed(p, data->genre, fa->size / sizeof(ucs2char_t), is_storing);
	}

	return ;

error_exit:
	idx_leafdata_free(data);
}

void idx_leafdata_init(ip2db_idxleaf_data_t* data)
{
	memset(data, 0, sizeof(*data));
}

void idx_leafdata_free(ip2db_idxleaf_data_t* data)
{
	ucs2free(data->pathname);
	ucs2free(data->filename);
	ucs2free(data->title);
	ucs2free(data->artist);
	ucs2free(data->album);
	ucs2free(data->genre);
	idx_leafdata_init(data);
}

void idxleaf_init(uint8_t* block)
{
	uint32_t unknown1 = 0x00007E26;
	uint32_t leaf_prev = 0;
	uint32_t leaf_next = 0;
	uint16_t size_avail = 1024-3;
    uint16_t size_used = 0;
	uint16_t unknown2 = 0;
	uint16_t unknown3 = 0;
	uint16_t num_data = 0;
	uint8_t leaf_line = 0;
	uint8_t  unknown5 = 0x7C;
	uint16_t unknown6 = 0;

	idxleaf_setget_unknown1(block, &unknown1, 1);
	idxleaf_setget_leaf_prev(block, &leaf_prev, 1);
	idxleaf_setget_leaf_next(block, &leaf_next, 1);
	idxleaf_setget_size_avail(block, &size_avail, 1);
	idxleaf_setget_size_used(block, &size_used, 1);
	idxleaf_setget_unknown2(block, &unknown2, 1);
	idxleaf_setget_unknown3(block, &unknown3, 1);
	idxleaf_setget_num_data(block, &num_data, 1);
	idxleaf_setget_leaf_line(block, &leaf_line, 1);
	idxleaf_setget_unknown5(block, &unknown5, 1);
	idxleaf_setget_unknown6(block, &unknown6, 1);
}

void idxleaf_set_field_access(ip2db_idxleaf_data_t* data, offset_size_t* field_access)
{
	field_access[0].offset = 0;
	field_access[0].size = (uint16_t)(ucs2len(data->pathname) * sizeof(ucs2char_t));
	field_access[1].offset = field_access[0].offset + field_access[0].size + sizeof(ucs2char_t);
	field_access[1].size = (uint16_t)(ucs2len(data->filename) * sizeof(ucs2char_t));
	field_access[2].offset = field_access[1].offset + field_access[1].size + sizeof(ucs2char_t);
	field_access[2].size = (uint16_t)(ucs2len(data->title) * sizeof(ucs2char_t));
	field_access[3].offset = field_access[2].offset + field_access[2].size + sizeof(ucs2char_t);
	field_access[3].size = (uint16_t)(ucs2len(data->artist) * sizeof(ucs2char_t));
	field_access[4].offset = field_access[3].offset + field_access[3].size + sizeof(ucs2char_t);
	field_access[4].size = (uint16_t)(ucs2len(data->album) * sizeof(ucs2char_t));
	field_access[5].offset = field_access[4].offset + field_access[4].size + sizeof(ucs2char_t);
	field_access[5].size = (uint16_t)(ucs2len(data->genre) * sizeof(ucs2char_t));
}

void idxleaf_repr(uint8_t* block, uint32_t offset, FILE *fp)
{
	int i;
	uint32_t unknown1, leaf_prev, leaf_next;
	uint16_t size_avail, size_used, unknown2, unknown3, num_data;
	uint8_t  leaf_line;
	uint8_t  unknown5;
	uint16_t unknown6;

	idxleaf_setget_unknown1(block, &unknown1, 0);
	idxleaf_setget_leaf_prev(block, &leaf_prev, 0);
	idxleaf_setget_leaf_next(block, &leaf_next, 0);
	idxleaf_setget_size_avail(block, &size_avail, 0);
	idxleaf_setget_size_used(block, &size_used, 0);
	idxleaf_setget_unknown2(block, &unknown2, 0);
	idxleaf_setget_unknown3(block, &unknown3, 0);
	idxleaf_setget_num_data(block, &num_data, 0);
	idxleaf_setget_leaf_line(block, &leaf_line, 0);
	idxleaf_setget_unknown5(block, &unknown5, 0);
	idxleaf_setget_unknown6(block, &unknown6, 0);

	fprintf(fp, "PAGE %d (0x%08X) DATA = [\n", offset / 0x400 + 1, offset);

	fprintf(fp, "  unknown1 (00007E26): %08X\n", unknown1);
	fprintf(fp, "  prev_leaf: %d\n", leaf_prev);
	fprintf(fp, "  next_leaf: %d\n", leaf_next);
	fprintf(fp, "  size_avail: %d\n", size_avail);		/* (1024 - size_used - 3 - 4 * num_data) */
	fprintf(fp, "  size_used: %d\n", size_used);
	fprintf(fp, "  unknown2 (0): %04X\n", unknown2);
	fprintf(fp, "  unknown3 (0): %04X\n", unknown3);
	fprintf(fp, "  num_data: %d\n", num_data);
	fprintf(fp, "  leaf_line: %d\n", leaf_line);

	for (i = 0;i < num_data;++i) {
		offset_size_t offset_size;
		ip2db_idxleaf_data_t data;

		idx_leafdata_init(&data);

		idxleaf_setget_data_offset_size(block, i, &offset_size, 0);
		idxleaf_setget_data_direct(block, i, &data, 0);

		fprintf(fp, "  data %d {offset: 0x%X, size: 0x%X} [\n", i, offset_size.offset, offset_size.size);
		fprints(fp, "    pathname: %s\n", data.pathname);
		fprints(fp, "    filename: %s\n", data.filename);
		fprints(fp, "    title: %s\n", data.title);
		fprints(fp, "    artist: %s\n", data.artist);
		fprints(fp, "    album: %s\n", data.album);
		fprints(fp, "    genre: %s\n", data.genre);
		fprintf(fp, "  ]\n");
	}
	fprintf(fp, "  unknown5: 0x%X\n", unknown5);
	fprintf(fp, "  unknown6: 0x%X\n", unknown6);

	fprintf(fp, "]\n");
}
