/*
 *      B-tree node page in idx.
 *
 *      Copyright (c) 2005-2007 Naoaki Okazaki
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 *
 */

/* $Id: idx_node.c 328 2007-02-10 17:50:11Z nyaochi $ */

#ifdef	HAVE_CONFIG_H
#include <config.h>
#endif/*HAVE_CONFIG_H*/
#ifdef	HAVE_STRING_H
#include <string.h>
#endif/*HAVE_STRING_H*/

#include <os.h>
#include <stdio.h>
#include <stdlib.h>
#include <pmplib/ucs2char.h>

#include "util.h"
#include "serialize.h"
#include "ip2db.h"

void idxnode_setget_size(uint8_t* block, uint16_t* value, int is_storing)
{
	serialize_uint16be(&block[0x000], value, is_storing);
}

void idxnode_setget_num_children(uint8_t* block, uint16_t* value, size_t keysize, int is_storing)
{
	uint16_t size;

	if (is_storing) {
		size = (uint16_t)(keysize * (*value) + sizeof(uint16_t));
		idxnode_setget_size(block, &size, is_storing);
	} else {
		idxnode_setget_size(block, &size, is_storing);
		*value = (uint16_t)((size - sizeof(uint16_t)) / keysize);
	}
}


void idxnode_setget_height(uint8_t* block, uint8_t* value, int is_storing)
{
	serialize_uint8(&block[0x3FE], value, is_storing);
}

void idxnode_setget_dword(uint8_t* block, uint16_t i, uint32_t* key, uint32_t* dup, uint32_t* next, int is_storing)
{
	uint8_t* p = dup ? 
		&block[sizeof(uint16_t) + i * sizeof(uint32_t) * 3] :
		&block[sizeof(uint16_t) + i * sizeof(uint32_t) * 2] ;

	p += serialize_uint32be(p, key, is_storing);
	if (dup) {
		p += serialize_uint32be(p, dup, is_storing);
	}
	p += serialize_uint32be(p, next, is_storing);
}

void idxnode_setget_str(uint8_t* block, uint16_t i, ucs2char_t* key, size_t keylen, uint32_t* dup, uint32_t* next, int is_storing)
{
	uint8_t* p = &block[sizeof(uint16_t) + i * (keylen * sizeof(ucs2char_t) + sizeof(uint32_t) + sizeof(uint32_t))];
	p += serialize_ucs2be_string_fixed(p, key, keylen, is_storing) * sizeof(ucs2char_t);
	p += serialize_uint32be(p, dup, is_storing);
	p += serialize_uint32be(p, next, is_storing);
}

void idxnode_init(uint8_t* block)
{
	uint16_t size = 2;

	memset(block, 0, PAGESIZE);
	idxnode_setget_size(block, &size, 1);
}


void idxnode_repr_dword(uint8_t* block, int has_dup, uint32_t offset, FILE *fp)
{
	int i;
	uint8_t height;
	uint16_t size, num_nodes;

	idxnode_setget_size(block, &size, 0);
	idxnode_setget_height(block, &height, 0);
	if (has_dup) {
		num_nodes = (size - sizeof(size)) / (sizeof(uint32_t) * 3);
	} else {
		num_nodes = (size - sizeof(size)) / (sizeof(uint32_t) * 2);
	}

	fprintf(fp, "PAGE %d (0x%08X) NODE_DWORD%s {size: 0x%X, height: %d} = [\n",
		offset / 0x400 + 1, offset, (has_dup ? "" : "_UNIQUE"), size, height);

	for (i = 0;i < num_nodes;++i) {
		uint32_t key, dup, next;

		if (has_dup) {
			idxnode_setget_dword(block, i, &key, &dup, &next, 0);
		} else {
			idxnode_setget_dword(block, i, &key, NULL, &next, 0);
		}

		fprintf(fp, "  ");
		if (has_dup) {
			fprintf(fp, "(%d, %d)", key, dup);
		} else {
			fprintf(fp, "%d", key);
		}
		fprintf(fp, ": ");
		print_next(fp, next, height);
		fprintf(fp, "\n");
	}

	fprintf(fp, "]\n");
}

void idxnode_repr_str(uint8_t* block, size_t keylen, uint32_t offset, FILE *fp)
{
	int i;
	uint8_t height;
	uint16_t size, num_children;

	idxnode_setget_size(block, &size, 0);
	idxnode_setget_height(block, &height, 0);
	idxnode_setget_num_children(block, &num_children, sizeof(uint32_t) * 2 + sizeof(ucs2char_t) * keylen, 0);

	fprintf(fp, "PAGE %d (0x%08X) NODE_STR%d {size: 0x%X, height: %d} = [\n",
		offset / 0x400 + 1, offset, keylen, size, height);

	for (i = 0;i < num_children;++i) {
		ucs2char_t key[24];
		uint32_t dup, next;

		idxnode_setget_str(block, i, key, keylen, &dup, &next, 0);

		fprintf(fp, "  (");
		print_keystr(fp, key, keylen);
		fprintf(fp, ", %d): ", dup);
		print_next(fp, next, height);
		fprintf(fp, "\n");
	}

	fprintf(fp, "]\n");
}
