/*
 *      Low-level library for db.dic.
 *
 *      Copyright (c) 2005-2007 Naoaki Okazaki
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 *
 */

/* $Id: dic.c 328 2007-02-10 17:50:11Z nyaochi $ */

/*
Some important findings from db.dic:
- This file defines field names/types in a database.
- This file stores offset addresses of root nodes in db.idx.
*/

#ifdef	HAVE_CONFIG_H
#include <config.h>
#endif/*HAVE_CONFIG_H*/

#include <os.h>
#include <stdio.h>
#include <stdlib.h>
#include <memory.h>
#include <pmplib/ucs2char.h>

#include "serialize.h"
#include "util.h"
#include "ip3db.h"
#include "dic.h"

static dic_index_t music_indices[] = {
	{0, 0x0086, {IP3DBF_MUSIC_TITLE, IP3DBF_MUSIC_NONE, IP3DBF_MUSIC_NONE}},
	{0, 0x00F8, {IP3DBF_MUSIC_RATING, IP3DBF_MUSIC_NONE, IP3DBF_MUSIC_NONE}},
	{0, 0x0152, {IP3DBF_MUSIC_TRACKNUMBER, IP3DBF_MUSIC_NONE, IP3DBF_MUSIC_NONE}},
	{0, 0x022C, {IP3DBF_MUSIC_CHANGEDFLAG, IP3DBF_MUSIC_NONE, IP3DBF_MUSIC_NONE}},
	{0, 0x027A, {IP3DBF_MUSIC_CLUSM, IP3DBF_MUSIC_NONE, IP3DBF_MUSIC_NONE}},
	{0, 0x031C, {IP3DBF_MUSIC_UID, IP3DBF_MUSIC_NONE, IP3DBF_MUSIC_NONE}},
	{0, 0x0352, {IP3DBF_MUSIC_GENRE, IP3DBF_MUSIC_ARTIST, IP3DBF_MUSIC_TITLE}},
	{0, 0x0384, {IP3DBF_MUSIC_ARTIST, IP3DBF_MUSIC_ALBUM, IP3DBF_MUSIC_TRACKNUMBER}},
	{0, 0x03C2, {IP3DBF_MUSIC_ARTIST, IP3DBF_MUSIC_TRACKNUMBER, IP3DBF_MUSIC_NONE}},
	{0, 0x03F4, {IP3DBF_MUSIC_ARTIST, IP3DBF_MUSIC_TITLE, IP3DBF_MUSIC_NONE}},
	{0, 0x041A, {IP3DBF_MUSIC_GENRE, IP3DBF_MUSIC_TITLE, IP3DBF_MUSIC_NONE}},
	{0, 0x043E, {IP3DBF_MUSIC_ALBUM, IP3DBF_MUSIC_TRACKNUMBER, IP3DBF_MUSIC_NONE}},
};

static dic_index_t references_indices[] = {
	{0, 0x0764, {IP3DBF_REFERENCES_PARENTCLUSTER, IP3DBF_REFERENCES_NONE, IP3DBF_REFERENCES_NONE}},
	{0, 0x078C, {IP3DBF_REFERENCES_CHILDCLUSTER, IP3DBF_REFERENCES_NONE, IP3DBF_REFERENCES_NONE}},
};

static dic_index_t objects_indices[] = {
	{0, 0x099A, {IP3DBF_OBJECTS_UID, IP3DBF_MUSIC_NONE, IP3DBF_MUSIC_NONE}},
	{0, 0x09F0, {IP3DBF_OBJECTS_FILETYPE, IP3DBF_MUSIC_NONE, IP3DBF_MUSIC_NONE}},
	{0, 0x0A0E, {IP3DBF_OBJECTS_OBJECTNAME, IP3DBF_MUSIC_NONE, IP3DBF_MUSIC_NONE}},
	{0, 0x0ACE, {IP3DBF_OBJECTS_FILETYPE, IP3DBF_OBJECTS_PARENTUID, IP3DBF_OBJECTS_PROPERTIES}},
};

static int dic_serialize(dic_t* dic, uint8_t* buffer, int is_storing);

static void dic_field_init(dic_field_t* entry)
{
	memset(entry, 0, sizeof(*entry));
}

static void dic_field_finish(dic_field_t* entry)
{
	ucs2free(entry->name);
	dic_field_init(entry);
}

static size_t dic_field_serialize(uint8_t* block, dic_field_t* entry, int is_storing)
{
	uint8_t *p = block;
	p += serialize_uint32be(p, &entry->next, 0);
	p += serialize_uint32be(p, &entry->type, 0);
	p += serialize_uint32be(p, &entry->idx_root, 0);
	if (is_storing) {
		p += (serialize_ucs2be_string_var(p, entry->name, is_storing) + 1) * sizeof(ucs2char_t);
	} else {
		p += (serialize_ucs2be_string_var_alloc(p, &entry->name) + 1) * sizeof(ucs2char_t);
	}
	return (size_t)(p - block);
}

static void dic_field_dump(const dic_field_t* entry, FILE *fp)
{
	fprintf(fp, "    next: 0x%08X\n", entry->next);
	fprintf(fp, "    type: 0x%08X\n", entry->type);
	fprintf(fp, "    idx_root: 0x%08X\n", entry->idx_root);
	fprints(fp, "    name: %s\n", entry->name);
}



static void dic_table_init(dic_table_t* list)
{
	memset(list, 0, sizeof(*list));
}

static void dic_table_finish(dic_table_t* list)
{
	int i;
	for (i = 0;i < list->num_fields;++i) {
		dic_field_finish(&list->fields[i]);
	}
	free(list->name);
	free(list->fields);
}

static void dic_table_dump(dic_table_t* list, FILE *fp)
{
	int i;

	fprints(fp, "%s = {\n", list->name);
	fprintf(fp, "  unknown0: 0x%08X\n", list->unknown0);
	fprintf(fp, "  next: 0x%08X\n", list->next);
	fprintf(fp, "  unknown1: 0x%08X\n", list->unknown1);
	fprintf(fp, "  dat_page: 0x%08X\n", list->dat_page);
	fprintf(fp, "  offset_fields: 0x%08X\n", list->offset_fields);
	fprintf(fp, "  offset_indices: 0x%08X\n", list->offset_indices);

	for (i = 0;i < list->num_fields;++i) {
		const dic_field_t* entry = &list->fields[i];
		fprintf(fp, "  FIELD %d = {\n", i);
		dic_field_dump(entry, fp);
		fprintf(fp, "  }\n");
	}

	fprintf(fp, "}\n");
}

static size_t dic_table_serialize(uint8_t* buffer, uint32_t start, dic_table_t* list, int is_storing)
{
	uint32_t i, next;
	uint8_t *p = buffer + start;

	p += serialize_uint32be(p, &list->unknown0, is_storing);
	p += serialize_uint32be(p, &list->next, is_storing);
	p += serialize_uint32be(p, &list->unknown1, is_storing);
	p += serialize_uint32be(p, &list->dat_page, is_storing);
	p += serialize_uint32be(p, &list->offset_fields, is_storing);
	p += serialize_uint32be(p, &list->offset_indices, is_storing);
	if (is_storing) {
		p += (serialize_ucs2be_string_var(p, list->name, is_storing) + 1) * sizeof(ucs2char_t);
	} else {
		p += (serialize_ucs2be_string_var_alloc(p, &list->name) + 1) * sizeof(ucs2char_t);
	}

	next = list->offset_fields;
	for (i = 0;i < list->num_fields;++i) {
		dic_field_serialize(buffer + next, &list->fields[i], is_storing);
		next = list->fields[i].next;
	}

	for (i = 0;i < list->num_indices;++i) {
		uint32_t offset = list->indices[i].offset + sizeof(uint32_t) * 2;
		serialize_uint32be(buffer + offset, &list->indices[i].idx_root, is_storing);
	}

	return (size_t)(p - (buffer + start));
}



dic_t* dic_new()
{
	uint8_t* dic_template = NULL;
	dic_t* dic = (dic_t*)malloc(sizeof(dic_t));
	if (dic) {
		dic_table_init(&dic->music);
		dic->music.num_fields = IP3DBF_MUSIC_LAST;
		dic->music.fields = (dic_field_t*)calloc(dic->music.num_fields, sizeof(dic_field_t));
		dic->music.num_indices = sizeof(music_indices) / sizeof(music_indices[0]);
		dic->music.indices = music_indices;
		dic_table_init(&dic->references);
		dic->references.num_fields = IP3DBF_REFERENCES_LAST;
		dic->references.fields = (dic_field_t*)calloc(dic->references.num_fields, sizeof(dic_field_t));
		dic->references.num_indices = sizeof(references_indices) / sizeof(references_indices[0]);
		dic->references.indices = references_indices;
		dic_table_init(&dic->objects);
		dic->objects.num_fields = IP3DBF_OBJECTS_LAST;
		dic->objects.fields = (dic_field_t*)calloc(dic->objects.num_fields, sizeof(dic_field_t));
		dic->objects.num_indices = sizeof(objects_indices) / sizeof(objects_indices[0]);
		dic->objects.indices = objects_indices;

		/* Read the database description, which rules everything in a database. */
		dic_template = dic_get_template(&dic->size);
		dic->buffer = (uint8_t*)malloc(dic->size);
		if (dic->buffer) {
			memcpy(dic->buffer, dic_template, dic->size);
			dic_serialize(dic, dic->buffer, 0);
		}
	}
	return dic;
}

void dic_finish(dic_t* dic)
{
	if (dic) {
		dic_table_finish(&dic->music);
		dic_table_finish(&dic->references);
		dic_table_finish(&dic->objects);
		free(dic->buffer);
		free(dic);
	}
}

int dic_read(dic_t* dic, FILE *fpi)
{
	free(dic->buffer);
	dic->size = 0;

	fread_all(fpi, &dic->buffer, &dic->size);

	/* Parse db.dic */
	if (dic_serialize(dic, dic->buffer, 0) != 0) {
		return 1;
	}
	return 0;
}

int dic_write(dic_t* dic, FILE *fpo)
{
	if (dic_serialize(dic, dic->buffer, 1) != 0) {
		return 1;
	}

	fwrite(dic->buffer, 1, dic->size, fpo);

	return 0;
}

static int dic_serialize(dic_t* dic, uint8_t* buffer, int is_storing)
{
	uint32_t next = 0;
	uint8_t* p = buffer;

	p += serialize_uint32be(p, &dic->header.size, is_storing);
	p += serialize_uint32be(p, &dic->header.num_dat_pages, is_storing);
	p += serialize_uint32be(p, &dic->header.num_idx_pages, is_storing);
	p += serialize_uint32be(p, &dic->header.unknown1, is_storing);

	dic_table_serialize(buffer, 0x0018, &dic->music, is_storing);
	dic_table_serialize(buffer, 0x0736, &dic->references, is_storing);
	dic_table_serialize(buffer, 0x0972, &dic->objects, is_storing);

	return 0;
}

void dic_dump(dic_t* dic, FILE *fp)
{
	fprintf(fp, "===== db.dic =====\n");
	dic_table_dump(&dic->music, fp);
	dic_table_dump(&dic->references, fp);
	dic_table_dump(&dic->objects, fp);
	fprintf(fp, "\n");
}

uint32_t dic_get_idxroot(dic_t* dic, int table, int index)
{
	const dic_index_t* indices = NULL;
	switch (table) {
	case IP3DBIDX_MUSIC:
		indices = dic->music.indices;
		break;
	case IP3DBIDX_REFERENCES:
		indices = dic->references.indices;
		break;
	case IP3DBIDX_OBJECTS:
		indices = dic->objects.indices;
		break;
	}
	if (indices) {
		return indices[index].idx_root;
	} else {
		return 0;
	}
}

void dic_set_idxroot(dic_t* dic, int table, int index, uint32_t root)
{
	dic_index_t* indices = NULL;
	switch (table) {
	case IP3DBIDX_MUSIC:
		indices = dic->music.indices;
		break;
	case IP3DBIDX_REFERENCES:
		indices = dic->references.indices;
		break;
	case IP3DBIDX_OBJECTS:
		indices = dic->objects.indices;
		break;
	}
	if (indices) {
		indices[index].idx_root = root;
	}
}

void dic_repr_index(dic_t* dic, int table, int index, FILE *fp)
{
	int i;
	const dic_table_t* tbl = NULL;
	const dic_index_t* idx = NULL;

	switch (table) {
	case IP3DBIDX_MUSIC:
		tbl = &dic->music;
		break;
	case IP3DBIDX_REFERENCES:
		tbl = &dic->references;
		break;
	case IP3DBIDX_OBJECTS:
		tbl = &dic->objects;
		break;
	}

	idx = &tbl->indices[index];
	for (i = 0;i < IP3DBIDX_MAX_KEYLEVEL;++i) {
		int field = idx->fields[i];
		if (field == IP3DBF_MUSIC_NONE) {
			break;
		}
		if (i != 0) {
			fputc('-', fp);
		}
		fprints(fp, "%s", tbl->fields[field].name);
	}
}

uint8_t *dic_get_template(long* size)
{
	static uint8_t dic_template[] = {
		0x00, 0x00, 0x0B, 0x16, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0xEA, 0x00, 0x00, 0x04, 0x6E, 0x00, 0x00, 0x09, 0x72, 
		0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x3C, 0x00, 0x00, 0x03, 0x52, 
		0x00, 0x4D, 0x00, 0x75, 0x00, 0x73, 0x00, 0x69, 0x00, 0x63, 0x00, 0x00, 0x00, 0x00, 0x00, 0x56, 
		0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x41, 0x00, 0x72, 0x00, 0x74, 0x00, 0x69, 
		0x00, 0x73, 0x00, 0x74, 0x00, 0x00, 0x00, 0x00, 0x00, 0x6E, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 
		0x00, 0x00, 0x00, 0x41, 0x00, 0x6C, 0x00, 0x62, 0x00, 0x75, 0x00, 0x6D, 0x00, 0x00, 0x00, 0x00, 
		0x00, 0x86, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x47, 0x00, 0x65, 0x00, 0x6E, 
		0x00, 0x72, 0x00, 0x65, 0x00, 0x00, 0x00, 0x00, 0x00, 0x9E, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 
		0x00, 0x00, 0x00, 0x54, 0x00, 0x69, 0x00, 0x74, 0x00, 0x6C, 0x00, 0x65, 0x00, 0x00, 0x00, 0x00, 
		0x00, 0xBC, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x46, 0x00, 0x69, 0x00, 0x6C, 
		0x00, 0x65, 0x00, 0x50, 0x00, 0x61, 0x00, 0x74, 0x00, 0x68, 0x00, 0x00, 0x00, 0x00, 0x00, 0xDA, 
		0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x46, 0x00, 0x69, 0x00, 0x6C, 0x00, 0x65, 
		0x00, 0x4E, 0x00, 0x61, 0x00, 0x6D, 0x00, 0x65, 0x00, 0x00, 0x00, 0x00, 0x00, 0xF8, 0x00, 0x00, 
		0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x00, 0x75, 0x00, 0x72, 0x00, 0x61, 0x00, 0x74, 
		0x00, 0x69, 0x00, 0x6F, 0x00, 0x6E, 0x00, 0x00, 0x00, 0x00, 0x01, 0x12, 0x00, 0x00, 0x00, 0x03, 
		0x00, 0x00, 0x00, 0x00, 0x00, 0x52, 0x00, 0x61, 0x00, 0x74, 0x00, 0x69, 0x00, 0x6E, 0x00, 0x67, 
		0x00, 0x00, 0x00, 0x00, 0x01, 0x30, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x55, 
		0x00, 0x73, 0x00, 0x65, 0x00, 0x43, 0x00, 0x6F, 0x00, 0x75, 0x00, 0x6E, 0x00, 0x74, 0x00, 0x00, 
		0x00, 0x00, 0x01, 0x52, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x46, 0x00, 0x69, 
		0x00, 0x6C, 0x00, 0x65, 0x00, 0x46, 0x00, 0x6F, 0x00, 0x72, 0x00, 0x6D, 0x00, 0x61, 0x00, 0x74, 
		0x00, 0x00, 0x00, 0x00, 0x01, 0x76, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x54, 
		0x00, 0x72, 0x00, 0x61, 0x00, 0x63, 0x00, 0x6B, 0x00, 0x4E, 0x00, 0x75, 0x00, 0x6D, 0x00, 0x62, 
		0x00, 0x65, 0x00, 0x72, 0x00, 0x00, 0x00, 0x00, 0x01, 0x8A, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 
		0x00, 0x00, 0x00, 0x44, 0x00, 0x72, 0x00, 0x6D, 0x00, 0x00, 0x00, 0x00, 0x01, 0xA2, 0x00, 0x00, 
		0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x4C, 0x00, 0x79, 0x00, 0x72, 0x00, 0x69, 0x00, 0x63, 
		0x00, 0x00, 0x00, 0x00, 0x01, 0xC0, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x50, 
		0x00, 0x75, 0x00, 0x72, 0x00, 0x63, 0x00, 0x68, 0x00, 0x61, 0x00, 0x73, 0x00, 0x65, 0x00, 0x00, 
		0x00, 0x00, 0x01, 0xEE, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x50, 0x00, 0x72, 
		0x00, 0x6F, 0x00, 0x74, 0x00, 0x65, 0x00, 0x63, 0x00, 0x74, 0x00, 0x69, 0x00, 0x6F, 0x00, 0x6E, 
		0x00, 0x53, 0x00, 0x74, 0x00, 0x61, 0x00, 0x74, 0x00, 0x75, 0x00, 0x73, 0x00, 0x00, 0x00, 0x00, 
		0x02, 0x10, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x00, 0x61, 0x00, 0x6D, 
		0x00, 0x70, 0x00, 0x6C, 0x00, 0x65, 0x00, 0x52, 0x00, 0x61, 0x00, 0x74, 0x00, 0x65, 0x00, 0x00, 
		0x00, 0x00, 0x02, 0x2C, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x42, 0x00, 0x69, 
		0x00, 0x74, 0x00, 0x52, 0x00, 0x61, 0x00, 0x74, 0x00, 0x65, 0x00, 0x00, 0x00, 0x00, 0x02, 0x50, 
		0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x43, 0x00, 0x68, 0x00, 0x61, 0x00, 0x6E, 
		0x00, 0x67, 0x00, 0x65, 0x00, 0x64, 0x00, 0x46, 0x00, 0x6C, 0x00, 0x61, 0x00, 0x67, 0x00, 0x00, 
		0x00, 0x00, 0x02, 0x7A, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x41, 0x00, 0x75, 
		0x00, 0x64, 0x00, 0x69, 0x00, 0x6F, 0x00, 0x57, 0x00, 0x41, 0x00, 0x56, 0x00, 0x45, 0x00, 0x43, 
		0x00, 0x6F, 0x00, 0x64, 0x00, 0x65, 0x00, 0x63, 0x00, 0x00, 0x00, 0x00, 0x02, 0x92, 0x00, 0x00, 
		0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x43, 0x00, 0x6C, 0x00, 0x75, 0x00, 0x73, 0x00, 0x4D, 
		0x00, 0x00, 0x00, 0x00, 0x02, 0xAA, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x43, 
		0x00, 0x6C, 0x00, 0x75, 0x00, 0x73, 0x00, 0x41, 0x00, 0x00, 0x00, 0x00, 0x02, 0xCE, 0x00, 0x00, 
		0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x41, 0x00, 0x6C, 0x00, 0x62, 0x00, 0x75, 0x00, 0x6D, 
		0x00, 0x41, 0x00, 0x72, 0x00, 0x74, 0x00, 0x50, 0x00, 0x6F, 0x00, 0x73, 0x00, 0x00, 0x00, 0x00, 
		0x02, 0xF8, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x4F, 0x00, 0x72, 0x00, 0x67, 
		0x00, 0x52, 0x00, 0x65, 0x00, 0x6C, 0x00, 0x65, 0x00, 0x61, 0x00, 0x73, 0x00, 0x65, 0x00, 0x44, 
		0x00, 0x61, 0x00, 0x74, 0x00, 0x65, 0x00, 0x00, 0x00, 0x00, 0x03, 0x1C, 0x00, 0x00, 0x00, 0x01, 
		0x00, 0x00, 0x00, 0x00, 0x00, 0x41, 0x00, 0x6C, 0x00, 0x62, 0x00, 0x75, 0x00, 0x6D, 0x00, 0x41, 
		0x00, 0x72, 0x00, 0x74, 0x00, 0x69, 0x00, 0x73, 0x00, 0x74, 0x00, 0x00, 0x00, 0x00, 0x03, 0x30, 
		0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x55, 0x00, 0x49, 0x00, 0x44, 0x00, 0x00, 
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x52, 0x00, 0x61, 
		0x00, 0x74, 0x00, 0x69, 0x00, 0x6E, 0x00, 0x67, 0x00, 0x74, 0x00, 0x69, 0x00, 0x6D, 0x00, 0x65, 
		0x00, 0x00, 0x00, 0x00, 0x03, 0x84, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x47, 
		0x00, 0x65, 0x00, 0x6E, 0x00, 0x72, 0x00, 0x65, 0x00, 0x00, 0x00, 0x41, 0x00, 0x72, 0x00, 0x74, 
		0x00, 0x69, 0x00, 0x73, 0x00, 0x74, 0x00, 0x00, 0x00, 0x54, 0x00, 0x69, 0x00, 0x74, 0x00, 0x6C, 
		0x00, 0x65, 0x00, 0x00, 0x00, 0x00, 0x03, 0xC2, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 
		0x00, 0x41, 0x00, 0x72, 0x00, 0x74, 0x00, 0x69, 0x00, 0x73, 0x00, 0x74, 0x00, 0x00, 0x00, 0x41, 
		0x00, 0x6C, 0x00, 0x62, 0x00, 0x75, 0x00, 0x6D, 0x00, 0x00, 0x00, 0x54, 0x00, 0x72, 0x00, 0x61, 
		0x00, 0x63, 0x00, 0x6B, 0x00, 0x4E, 0x00, 0x75, 0x00, 0x6D, 0x00, 0x62, 0x00, 0x65, 0x00, 0x72, 
		0x00, 0x00, 0x00, 0x00, 0x03, 0xF4, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x41, 
		0x00, 0x72, 0x00, 0x74, 0x00, 0x69, 0x00, 0x73, 0x00, 0x74, 0x00, 0x00, 0x00, 0x54, 0x00, 0x72, 
		0x00, 0x61, 0x00, 0x63, 0x00, 0x6B, 0x00, 0x4E, 0x00, 0x75, 0x00, 0x6D, 0x00, 0x62, 0x00, 0x65, 
		0x00, 0x72, 0x00, 0x00, 0x00, 0x00, 0x04, 0x1A, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 
		0x00, 0x41, 0x00, 0x72, 0x00, 0x74, 0x00, 0x69, 0x00, 0x73, 0x00, 0x74, 0x00, 0x00, 0x00, 0x54, 
		0x00, 0x69, 0x00, 0x74, 0x00, 0x6C, 0x00, 0x65, 0x00, 0x00, 0x00, 0x00, 0x04, 0x3E, 0x00, 0x00, 
		0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x47, 0x00, 0x65, 0x00, 0x6E, 0x00, 0x72, 0x00, 0x65, 
		0x00, 0x00, 0x00, 0x54, 0x00, 0x69, 0x00, 0x74, 0x00, 0x6C, 0x00, 0x65, 0x00, 0x00, 0x00, 0x00, 
		0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x41, 0x00, 0x6C, 0x00, 0x62, 
		0x00, 0x75, 0x00, 0x6D, 0x00, 0x00, 0x00, 0x54, 0x00, 0x72, 0x00, 0x61, 0x00, 0x63, 0x00, 0x6B, 
		0x00, 0x4E, 0x00, 0x75, 0x00, 0x6D, 0x00, 0x62, 0x00, 0x65, 0x00, 0x72, 0x00, 0x00, 0x00, 0x00, 
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
		0x04, 0x92, 0x00, 0x00, 0x00, 0x00, 0x00, 0x41, 0x00, 0x6C, 0x00, 0x62, 0x00, 0x75, 0x00, 0x6D, 
		0x00, 0x00, 0x00, 0x00, 0x04, 0xB0, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x46, 
		0x00, 0x69, 0x00, 0x6C, 0x00, 0x65, 0x00, 0x50, 0x00, 0x61, 0x00, 0x74, 0x00, 0x68, 0x00, 0x00, 
		0x00, 0x00, 0x04, 0xCE, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x46, 0x00, 0x69, 
		0x00, 0x6C, 0x00, 0x65, 0x00, 0x4E, 0x00, 0x61, 0x00, 0x6D, 0x00, 0x65, 0x00, 0x00, 0x00, 0x00, 
		0x04, 0xE8, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x41, 0x00, 0x72, 0x00, 0x74, 
		0x00, 0x69, 0x00, 0x73, 0x00, 0x74, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x01, 
		0x00, 0x00, 0x00, 0x00, 0x00, 0x41, 0x00, 0x6C, 0x00, 0x62, 0x00, 0x75, 0x00, 0x6D, 0x00, 0x00, 
		0x00, 0x00, 0x05, 0x18, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x47, 0x00, 0x65, 
		0x00, 0x6E, 0x00, 0x72, 0x00, 0x65, 0x00, 0x00, 0x00, 0x00, 0x05, 0x32, 0x00, 0x00, 0x00, 0x03, 
		0x00, 0x00, 0x00, 0x00, 0x00, 0x46, 0x00, 0x6F, 0x00, 0x72, 0x00, 0x6D, 0x00, 0x61, 0x00, 0x74, 
		0x00, 0x00, 0x00, 0x00, 0x05, 0x56, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 
		0x00, 0x61, 0x00, 0x6D, 0x00, 0x70, 0x00, 0x6C, 0x00, 0x65, 0x00, 0x57, 0x00, 0x69, 0x00, 0x64, 
		0x00, 0x74, 0x00, 0x68, 0x00, 0x00, 0x00, 0x00, 0x05, 0x7C, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 
		0x00, 0x00, 0x00, 0x53, 0x00, 0x61, 0x00, 0x6D, 0x00, 0x70, 0x00, 0x6C, 0x00, 0x65, 0x00, 0x48, 
		0x00, 0x65, 0x00, 0x69, 0x00, 0x67, 0x00, 0x68, 0x00, 0x74, 0x00, 0x00, 0x00, 0x00, 0x05, 0x9C, 
		0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x53, 0x00, 0x61, 0x00, 0x6D, 0x00, 0x70, 
		0x00, 0x6C, 0x00, 0x65, 0x00, 0x50, 0x00, 0x6F, 0x00, 0x73, 0x00, 0x00, 0x00, 0x00, 0x05, 0xBE, 
		0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x00, 0x61, 0x00, 0x6D, 0x00, 0x70, 
		0x00, 0x6C, 0x00, 0x65, 0x00, 0x53, 0x00, 0x69, 0x00, 0x7A, 0x00, 0x65, 0x00, 0x00, 0x00, 0x00, 
		0x05, 0xD6, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x43, 0x00, 0x6C, 0x00, 0x75, 
		0x00, 0x73, 0x00, 0x41, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 
		0x00, 0x01, 0x00, 0x55, 0x00, 0x49, 0x00, 0x44, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 
		0x07, 0x36, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x0E, 0x00, 0x00, 
		0x00, 0x00, 0x00, 0x50, 0x00, 0x68, 0x00, 0x6F, 0x00, 0x74, 0x00, 0x6F, 0x00, 0x00, 0x00, 0x00, 
		0x06, 0x2C, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x46, 0x00, 0x69, 0x00, 0x6C, 
		0x00, 0x65, 0x00, 0x50, 0x00, 0x61, 0x00, 0x74, 0x00, 0x68, 0x00, 0x00, 0x00, 0x00, 0x06, 0x4A, 
		0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x46, 0x00, 0x69, 0x00, 0x6C, 0x00, 0x65, 
		0x00, 0x4E, 0x00, 0x61, 0x00, 0x6D, 0x00, 0x65, 0x00, 0x00, 0x00, 0x00, 0x06, 0x62, 0x00, 0x00, 
		0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x57, 0x00, 0x69, 0x00, 0x64, 0x00, 0x74, 0x00, 0x68, 
		0x00, 0x00, 0x00, 0x00, 0x06, 0x7C, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 
		0x00, 0x65, 0x00, 0x69, 0x00, 0x67, 0x00, 0x68, 0x00, 0x74, 0x00, 0x00, 0x00, 0x00, 0x06, 0x96, 
		0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x46, 0x00, 0x6F, 0x00, 0x72, 0x00, 0x6D, 
		0x00, 0x61, 0x00, 0x74, 0x00, 0x00, 0x00, 0x00, 0x06, 0xBA, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 
		0x00, 0x00, 0x00, 0x53, 0x00, 0x61, 0x00, 0x6D, 0x00, 0x70, 0x00, 0x6C, 0x00, 0x65, 0x00, 0x57, 
		0x00, 0x69, 0x00, 0x64, 0x00, 0x74, 0x00, 0x68, 0x00, 0x00, 0x00, 0x00, 0x06, 0xE0, 0x00, 0x00, 
		0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x00, 0x61, 0x00, 0x6D, 0x00, 0x70, 0x00, 0x6C, 
		0x00, 0x65, 0x00, 0x48, 0x00, 0x65, 0x00, 0x69, 0x00, 0x67, 0x00, 0x68, 0x00, 0x74, 0x00, 0x00, 
		0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x53, 0x00, 0x61, 
		0x00, 0x6D, 0x00, 0x70, 0x00, 0x6C, 0x00, 0x65, 0x00, 0x50, 0x00, 0x6F, 0x00, 0x73, 0x00, 0x00, 
		0x00, 0x00, 0x07, 0x22, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x00, 0x61, 
		0x00, 0x6D, 0x00, 0x70, 0x00, 0x6C, 0x00, 0x65, 0x00, 0x53, 0x00, 0x69, 0x00, 0x7A, 0x00, 0x65, 
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x55, 
		0x00, 0x49, 0x00, 0x44, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0xD4, 0x00, 0x00, 
		0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0x64, 0x00, 0x00, 0x00, 0x00, 0x00, 0x52, 
		0x00, 0x65, 0x00, 0x66, 0x00, 0x65, 0x00, 0x72, 0x00, 0x65, 0x00, 0x6E, 0x00, 0x63, 0x00, 0x65, 
		0x00, 0x73, 0x00, 0x00, 0x00, 0x00, 0x07, 0x8C, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 
		0x00, 0x50, 0x00, 0x61, 0x00, 0x72, 0x00, 0x65, 0x00, 0x6E, 0x00, 0x74, 0x00, 0x43, 0x00, 0x6C, 
		0x00, 0x75, 0x00, 0x73, 0x00, 0x74, 0x00, 0x65, 0x00, 0x72, 0x00, 0x00, 0x00, 0x00, 0x07, 0xB2, 
		0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x43, 0x00, 0x68, 0x00, 0x69, 0x00, 0x6C, 
		0x00, 0x64, 0x00, 0x43, 0x00, 0x6C, 0x00, 0x75, 0x00, 0x73, 0x00, 0x74, 0x00, 0x65, 0x00, 0x72, 
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x46, 
		0x00, 0x69, 0x00, 0x6C, 0x00, 0x65, 0x00, 0x46, 0x00, 0x6F, 0x00, 0x72, 0x00, 0x6D, 0x00, 0x61, 
		0x00, 0x74, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0xF8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x56, 0x00, 0x69, 
		0x00, 0x64, 0x00, 0x65, 0x00, 0x6F, 0x00, 0x00, 0x00, 0x00, 0x08, 0x24, 0x00, 0x00, 0x00, 0x04, 
		0x00, 0x00, 0x00, 0x00, 0x00, 0x41, 0x00, 0x75, 0x00, 0x64, 0x00, 0x69, 0x00, 0x6F, 0x00, 0x53, 
		0x00, 0x61, 0x00, 0x6D, 0x00, 0x70, 0x00, 0x6C, 0x00, 0x65, 0x00, 0x52, 0x00, 0x61, 0x00, 0x74, 
		0x00, 0x65, 0x00, 0x00, 0x00, 0x00, 0x08, 0x4A, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 
		0x00, 0x41, 0x00, 0x75, 0x00, 0x64, 0x00, 0x69, 0x00, 0x6F, 0x00, 0x42, 0x00, 0x69, 0x00, 0x74, 
		0x00, 0x52, 0x00, 0x61, 0x00, 0x74, 0x00, 0x65, 0x00, 0x00, 0x00, 0x00, 0x08, 0x62, 0x00, 0x00, 
		0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x57, 0x00, 0x69, 0x00, 0x64, 0x00, 0x74, 0x00, 0x68, 
		0x00, 0x00, 0x00, 0x00, 0x08, 0x7C, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 
		0x00, 0x65, 0x00, 0x69, 0x00, 0x67, 0x00, 0x68, 0x00, 0x74, 0x00, 0x00, 0x00, 0x00, 0x08, 0xA2, 
		0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x54, 0x00, 0x6F, 0x00, 0x74, 0x00, 0x61, 
		0x00, 0x6C, 0x00, 0x42, 0x00, 0x69, 0x00, 0x74, 0x00, 0x52, 0x00, 0x61, 0x00, 0x74, 0x00, 0x65, 
		0x00, 0x00, 0x00, 0x00, 0x08, 0xC8, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x56, 
		0x00, 0x69, 0x00, 0x64, 0x00, 0x65, 0x00, 0x6F, 0x00, 0x42, 0x00, 0x69, 0x00, 0x74, 0x00, 0x52, 
		0x00, 0x61, 0x00, 0x74, 0x00, 0x65, 0x00, 0x00, 0x00, 0x00, 0x09, 0x06, 0x00, 0x00, 0x00, 0x04, 
		0x00, 0x00, 0x00, 0x00, 0x00, 0x46, 0x00, 0x72, 0x00, 0x61, 0x00, 0x6D, 0x00, 0x65, 0x00, 0x73, 
		0x00, 0x50, 0x00, 0x65, 0x00, 0x72, 0x00, 0x54, 0x00, 0x68, 0x00, 0x6F, 0x00, 0x75, 0x00, 0x73, 
		0x00, 0x61, 0x00, 0x6E, 0x00, 0x64, 0x00, 0x53, 0x00, 0x65, 0x00, 0x63, 0x00, 0x6F, 0x00, 0x6E, 
		0x00, 0x64, 0x00, 0x73, 0x00, 0x00, 0x00, 0x00, 0x09, 0x34, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 
		0x00, 0x00, 0x00, 0x56, 0x00, 0x69, 0x00, 0x64, 0x00, 0x65, 0x00, 0x6F, 0x00, 0x46, 0x00, 0x6F, 
		0x00, 0x75, 0x00, 0x72, 0x00, 0x43, 0x00, 0x43, 0x00, 0x43, 0x00, 0x6F, 0x00, 0x64, 0x00, 0x65, 
		0x00, 0x63, 0x00, 0x00, 0x00, 0x00, 0x09, 0x5E, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 
		0x00, 0x41, 0x00, 0x75, 0x00, 0x64, 0x00, 0x69, 0x00, 0x6F, 0x00, 0x57, 0x00, 0x41, 0x00, 0x56, 
		0x00, 0x45, 0x00, 0x43, 0x00, 0x6F, 0x00, 0x64, 0x00, 0x65, 0x00, 0x63, 0x00, 0x00, 0x00, 0x00, 
		0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x55, 0x00, 0x49, 0x00, 0x44, 
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 
		0x00, 0x01, 0x00, 0x00, 0x09, 0x9A, 0x00, 0x00, 0x0A, 0xCE, 0x00, 0x4F, 0x00, 0x62, 0x00, 0x6A, 
		0x00, 0x65, 0x00, 0x63, 0x00, 0x74, 0x00, 0x73, 0x00, 0x00, 0x00, 0x00, 0x09, 0xAE, 0x00, 0x00, 
		0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x55, 0x00, 0x49, 0x00, 0x44, 0x00, 0x00, 0x00, 0x00, 
		0x09, 0xCE, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x50, 0x00, 0x61, 0x00, 0x72, 
		0x00, 0x65, 0x00, 0x6E, 0x00, 0x74, 0x00, 0x55, 0x00, 0x69, 0x00, 0x64, 0x00, 0x00, 0x00, 0x00, 
		0x09, 0xF0, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x50, 0x00, 0x72, 0x00, 0x6F, 
		0x00, 0x70, 0x00, 0x65, 0x00, 0x72, 0x00, 0x74, 0x00, 0x69, 0x00, 0x65, 0x00, 0x73, 0x00, 0x00, 
		0x00, 0x00, 0x0A, 0x0E, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x46, 0x00, 0x69, 
		0x00, 0x6C, 0x00, 0x65, 0x00, 0x54, 0x00, 0x79, 0x00, 0x70, 0x00, 0x65, 0x00, 0x00, 0x00, 0x00, 
		0x0A, 0x30, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x4F, 0x00, 0x62, 0x00, 0x6A, 
		0x00, 0x65, 0x00, 0x63, 0x00, 0x74, 0x00, 0x4E, 0x00, 0x61, 0x00, 0x6D, 0x00, 0x65, 0x00, 0x00, 
		0x00, 0x00, 0x0A, 0x46, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x4E, 0x00, 0x61, 
		0x00, 0x6D, 0x00, 0x65, 0x00, 0x00, 0x00, 0x00, 0x0A, 0x64, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 
		0x00, 0x00, 0x00, 0x46, 0x00, 0x69, 0x00, 0x6C, 0x00, 0x65, 0x00, 0x53, 0x00, 0x69, 0x00, 0x7A, 
		0x00, 0x65, 0x00, 0x00, 0x00, 0x00, 0x0A, 0x82, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 
		0x00, 0x44, 0x00, 0x61, 0x00, 0x74, 0x00, 0x65, 0x00, 0x43, 0x00, 0x72, 0x00, 0x65, 0x00, 0x61, 
		0x00, 0x00, 0x00, 0x00, 0x0A, 0x9A, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x52, 
		0x00, 0x6F, 0x00, 0x77, 0x00, 0x49, 0x00, 0x44, 0x00, 0x00, 0x00, 0x00, 0x0A, 0xB4, 0x00, 0x00, 
		0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x50, 0x00, 0x55, 0x00, 0x4F, 0x00, 0x49, 0x00, 0x44, 
		0x00, 0x31, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 
		0x00, 0x50, 0x00, 0x55, 0x00, 0x4F, 0x00, 0x49, 0x00, 0x44, 0x00, 0x32, 0x00, 0x00, 0x00, 0x00, 
		0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x46, 0x00, 0x69, 0x00, 0x6C, 
		0x00, 0x65, 0x00, 0x54, 0x00, 0x79, 0x00, 0x70, 0x00, 0x65, 0x00, 0x00, 0x00, 0x50, 0x00, 0x61, 
		0x00, 0x72, 0x00, 0x65, 0x00, 0x6E, 0x00, 0x74, 0x00, 0x55, 0x00, 0x69, 0x00, 0x64, 0x00, 0x00, 
		0x00, 0x50, 0x00, 0x72, 0x00, 0x6F, 0x00, 0x70, 0x00, 0x65, 0x00, 0x72, 0x00, 0x74, 0x00, 0x69, 
		0x00, 0x65, 0x00, 0x73, 0x00, 0x00, 
	};
	if (size) {
		*size = sizeof(dic_template) / sizeof(dic_template[0]);
	}
	return dic_template;
}
