/*
 *      Media database reader/writer for iriver E10.
 *
 *      Copyright (c) 2005-2007 Naoaki Okazaki
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 *
 */

/* $Id: ip3db.h 331 2007-02-10 18:39:35Z nyaochi $ */

#ifndef	__IP3DB_IP3DB_H__
#define	__IP3DB_IP3DB_H__

enum {
	IP3DBIDX_MAX_KEYLEVEL = 3,	/**< The maximum number of multiple AVL trees. */
};

/**
 * Field type IDs used in db.dic (Do not change the associated values)
 */
enum {
	IP3DBVT_NONE		= 0,	/**< Attached to no specific type. */
	IP3DBVT_STRING		= 1,	/**< UCS-2 string. */
	IP3DBVT_BYTE		= 2,	/**< uint8_t */
	IP3DBVT_WORD		= 3,	/**< uint16_t */
	IP3DBVT_DWORD		= 4,	/**< uint32_t */
};

/**
 * Variant type for field values.
 */
typedef struct {
	uint32_t	type;
	union {
		uint8_t		byte;
		uint16_t	word;
		uint32_t	dword;
		ucs2char_t*	str;
	} value;
} ip3db_variant_t;

typedef struct {
	ucs2char_t* filepath;
	ucs2char_t* filename;
	int num_entries;
	ucs2char_t **entries;
	uint32_t	uid;
} ip3db_playlist_t;

/**
 * Fields in a music record.
 */
enum {
	IP3DBF_MUSIC_NONE = -1,
	IP3DBF_MUSIC_BEGIN = 0,
	IP3DBF_MUSIC_ARTIST = 0,
	IP3DBF_MUSIC_ALBUM,
	IP3DBF_MUSIC_GENRE,
	IP3DBF_MUSIC_TITLE,
	IP3DBF_MUSIC_FILEPATH,
	IP3DBF_MUSIC_FILENAME,
	IP3DBF_MUSIC_DURATION,
	IP3DBF_MUSIC_RATING,
	IP3DBF_MUSIC_USECOUNT,
	IP3DBF_MUSIC_FILEFORMAT,
	IP3DBF_MUSIC_TRACKNUMBER,
	IP3DBF_MUSIC_DRM,
	IP3DBF_MUSIC_LYRIC,
	IP3DBF_MUSIC_PURCHASE,
	IP3DBF_MUSIC_PROTECTIONSTATUS,
	IP3DBF_MUSIC_SAMPLERATE,
	IP3DBF_MUSIC_BITRATE,
	IP3DBF_MUSIC_CHANGEDFLAG,
	IP3DBF_MUSIC_AUDIOWAVECODEC,
	IP3DBF_MUSIC_CLUSM,
	IP3DBF_MUSIC_CLUSA,
	IP3DBF_MUSIC_ALBUMARTPOS,
	IP3DBF_MUSIC_ORGRELEASEDATE,
	IP3DBF_MUSIC_ALBUMARTIST,
	IP3DBF_MUSIC_UID,
	IP3DBF_MUSIC_RATINGTIME,
	IP3DBF_MUSIC_LAST,
};

enum {
	IP3DBF_REFERENCES_NONE = -1,
	IP3DBF_REFERENCES_BEGIN = 0,
	IP3DBF_REFERENCES_PARENTCLUSTER = 0,
	IP3DBF_REFERENCES_CHILDCLUSTER,
	IP3DBF_REFERENCES_FILEFORMAT,
	IP3DBF_REFERENCES_LAST,
};

/** 
 * A music record.
 */
typedef ip3db_variant_t ip3db_music_record_t[IP3DBF_MUSIC_LAST];

/**
 * Fields in a object record.
 */
enum {
	IP3DBF_OBJECTS_NONE = -1,
	IP3DBF_OBJECTS_BEGIN = 0,
	IP3DBF_OBJECTS_UID = 0,
	IP3DBF_OBJECTS_PARENTUID,
	IP3DBF_OBJECTS_PROPERTIES,
	IP3DBF_OBJECTS_FILETYPE,
	IP3DBF_OBJECTS_OBJECTNAME,
	IP3DBF_OBJECTS_NAME,
	IP3DBF_OBJECTS_FILESIZE,
	IP3DBF_OBJECTS_DATECREA,
	IP3DBF_OBJECTS_RAWID,
	IP3DBF_OBJECTS_PUOID1,
	IP3DBF_OBJECTS_PUOID2,
	IP3DBF_OBJECTS_LAST,
};

/**
 * Fields chunks in db.dat.
 */
enum {
	IP3DBIDX_NONE = -1,
	IP3DBIDX_MUSIC = 0,
	IP3DBIDX_REFERENCES,
	IP3DBIDX_OBJECTS,
	IP3DBIDX_LAST,
};

/**
 * Indices for the music chunk.
 */
enum {
	IP3DBIDX_MUSIC_NONE = -1,
	IP3DBIDX_MUSIC_BEGIN = 0,
	IP3DBIDX_MUSIC_TITLE = 0,
	IP3DBIDX_MUSIC_RATING,
	IP3DBIDX_MUSIC_TRACKNUMBER,
	IP3DBIDX_MUSIC_CHANGEDFLAG,
	IP3DBIDX_MUSIC_CLUSM,
	IP3DBIDX_MUSIC_UID,
	IP3DBIDX_MUSIC_GENRE_ARTIST_TITLE,
	IP3DBIDX_MUSIC_ARTIST_ALBUM_TRACKNUMBER,
	IP3DBIDX_MUSIC_ARTIST_TRACKNUMBER,
	IP3DBIDX_MUSIC_ARTIST_TITLE,
	IP3DBIDX_MUSIC_GENRE_TITLE,
	IP3DBIDX_MUSIC_ALBUM_TRACKNUMBER,
	IP3DBIDX_MUSIC_LAST,
};

enum {
	IP3DBIDX_REFERENCES_NONE = -1,
	IP3DBIDX_REFERENCES_BEGIN = 0,
	IP3DBIDX_REFERENCES_PARENTCLUSTER = 0,
	IP3DBIDX_REFERENCES_CHILDCLUSTER,
	IP3DBIDX_REFERENCES_LAST,
};

/**
 * Indices for the object chunk.
 */
enum {
	IP3DBIDX_OBJECT_NONE = -1,
	IP3DBIDX_OBJECT_BEGIN = 0,
	IP3DBIDX_OBJECT_UID,
	IP3DBIDX_OBJECT_FILETYPE,
	IP3DBIDX_OBJECT_OBJECTNAME,
	IP3DBIDX_OBJECT_FILETYPE_PARENTUID_PROP,
	IP3DBIDX_OBJECT_LAST,
};



struct tag_dat_t; typedef struct tag_dat_t dat_t;
struct tag_dic_t; typedef struct tag_dic_t dic_t;
struct tag_idx_t; typedef struct tag_idx_t idx_t;

typedef struct {
	dat_t*		dat;
	dic_t*		dic;
	idx_t*		idx;
} ip3db_t;

void ip3db_variant_init(ip3db_variant_t* var, int type);
void ip3db_variant_finish(ip3db_variant_t* var);
void ip3db_variant_set_byte(ip3db_variant_t* var, uint8_t val);
void ip3db_variant_set_word(ip3db_variant_t* var, uint16_t val);
void ip3db_variant_set_dword(ip3db_variant_t* var, uint32_t val);
void ip3db_variant_set_str(ip3db_variant_t* var, const ucs2char_t* val);
void ip3db_variant_clone(ip3db_variant_t* dst, const ip3db_variant_t* src);

void ip3db_init(ip3db_t* db);
void ip3db_finish(ip3db_t* db);
result_t ip3db_read(ip3db_t* db, const ucs2char_t* datfn, const ucs2char_t* dicfn, const ucs2char_t* idxfn);
result_t ip3db_write(ip3db_t* db, const ucs2char_t* datfn, const ucs2char_t* dicfn, const ucs2char_t* idxfn);
result_t ip3db_set(ip3db_t* db, const ip3db_music_record_t* records, int num_records, ip3db_playlist_t* playlists, int num_playlists);
int ip3db_num_records(ip3db_t* db);
ip3db_music_record_t* ip3db_get_record(ip3db_t* db, int i);
result_t ip3db_dump(ip3db_t* db, FILE *fpo);

void ip3db_record_init(ip3db_t* db, ip3db_music_record_t* record);
void ip3db_record_finish(ip3db_t* db, ip3db_music_record_t* record);

void ip3db_playlist_init(ip3db_playlist_t* playlist);
void ip3db_playlist_finish(ip3db_playlist_t* playlist);
void ip3db_playlist_copy(ip3db_playlist_t* dst, const ip3db_playlist_t* src);

#endif	/*_IP3DB_IP3DB_H__*/
