/*
 *      Low level interface for hdr (e.g., H10DB.hdr, DB5000.HDR, ...).
 *
 *      Copyright (c) 2005-2007 Naoaki Okazaki
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 *
 */

/* $Id: hdr.c 328 2007-02-10 17:50:11Z nyaochi $ */

#ifdef	HAVE_CONFIG_H
#include <config.h>
#endif/*HAVE_CONFIG_H*/
#ifdef	HAVE_STRING_H
#include <string.h>
#endif/*HAVE_STRING_H*/

#include <os.h>
#include <stdio.h>
#include <stdlib.h>
#include <pmplib/ucs2char.h>

#include "serialize.h"
#include "util.h"
#include "pp1db.h"

void hdr_finish(hdr_t* hdr)
{
	free(hdr->dat_record_offset);
	free(hdr->dat_field_offset);
	free(hdr->max_dat_field_size);
	free(hdr->fd);
	memset(hdr, 0, sizeof(*hdr));
}

size_t hdr_serialize(uint8_t* block, hdr_t* hdr, int is_storing)
{
	uint8_t* p = block;
	uint32_t i, j;
	uint16_t *q = NULL;

	p += serialize_uint32le(p, &hdr->unknown1, is_storing);
	p += serialize_uint32le(p, &hdr->unknown2, is_storing);
	p += serialize_ucs2le_string_fixed(p, hdr->pathname_dat, PP1DB_PATHLENGTH, is_storing) * sizeof(ucs2char_t);
	p += serialize_uint32le(p, &hdr->unknown3, is_storing);
	p += serialize_ucs2le_string_fixed(p, hdr->pathname_hdr, PP1DB_PATHLENGTH, is_storing) * sizeof(ucs2char_t);
	p += serialize_uint32le(p, &hdr->unknown4, is_storing);
	p += serialize_uint32le(p, &hdr->num_dat_entries, is_storing);
	p += serialize_uint32le(p, &hdr->num_dat_inactive_entries, is_storing);
	p += serialize_uint32le(p, &hdr->num_dat_fields, is_storing);

	for (i = 0;i < hdr->param.max_fields;i++) {
		p += field_descriptor_serialize(p, &hdr->fd[i], is_storing);
	}
	for (i = 0;i < hdr->param.padding_size;i++) {
		uint8_t c = 0;
		p += serialize_uint8(p, &c, is_storing);
	}

	for (i = 0;i < hdr->param.max_fields;i++) {
		p += serialize_uint32le(p, &hdr->max_dat_field_size[i], is_storing);
	}

	p += serialize_uint32le(p, &hdr->dat_size, is_storing);

	if (hdr->param.has_unknown5) {
		p += serialize_uint32le(p, &hdr->unknown5, is_storing);
	}

	for (i = 0;i < hdr->param.max_entries;i++) {
		uint16_t *q = FIELDOFFSETS(hdr->dat_field_offset, i, hdr->param.max_fields);
		for (j = 0;j < hdr->param.max_fields;j++) {
			p += serialize_uint16le(p, &q[j], is_storing);
		}
	}
	for (i = 0;i < hdr->param.max_entries+1;i++) {
		p += serialize_uint32le(p, &hdr->dat_record_offset[i], is_storing);
	}
	if (!is_storing) {
		for (i = 0;i < hdr->num_dat_entries;i++) {
			uint16_t *q = FIELDOFFSETS(hdr->dat_field_offset, i, hdr->param.max_fields);
			q[hdr->num_dat_fields] = hdr->dat_record_offset[i+1] - hdr->dat_record_offset[i];
		}
	}

	return (size_t)(p - block);
}

void hdr_repr(const hdr_t* hdr, FILE *fp)
{
	uint32_t i, j;

	fprintf(fp, "// Header\n");
	fprintf(fp, "unknown1: %d\n", hdr->unknown1);
	fprintf(fp, "unknown2: %d\n", hdr->unknown2);
	fprints(fp, "pathname_dat: %s\n", hdr->pathname_dat);
	fprintf(fp, "unknown3: %d\n", hdr->unknown3);
	fprints(fp, "pathname_hdr: %s\n", hdr->pathname_hdr);
	fprintf(fp, "unknown4: 0x%08X\n", hdr->unknown4);
	fprintf(fp, "num_dat_entries: %d\n", hdr->num_dat_entries);
	fprintf(fp, "num_dat_inactive_entries: %d\n", hdr->num_dat_inactive_entries);
	fprintf(fp, "num_dat_fields: %d\n", hdr->num_dat_fields);

	for (i = 0;i < hdr->param.max_fields;i++) {
		fprintf(fp, "fd[%2d]: ", i);
		field_descriptor_repr(&hdr->fd[i], fp);
		fprintf(fp, "\n");
	}

	for (i = 0;i < hdr->param.max_fields;i++) {
		fprintf(fp, "max_dat_field_size[%d]: %d\n", i, hdr->max_dat_field_size[i]);
	}

	fprintf(fp, "dat_size: %d\n", hdr->dat_size);
	if (hdr->param.has_unknown5) {
		fprintf(fp, "unknown5: %d\n", hdr->unknown5);
	}

	for (i = 0;i < hdr->param.max_entries;i++) {
		uint16_t* p = FIELDOFFSETS(hdr->dat_field_offset, i, hdr->param.max_fields);
		fprintf(fp, "dat_field_offset[%d] = {", i);
		for (j = 0;j < hdr->param.max_fields;j++) {
			if (j != 0) {
				fprintf(fp, ",");
			}
			fprintf(fp, "%3d", p[j]);
		}
		fprintf(fp, "}\n");
	}

	for (i = 0;i < hdr->param.max_entries+1;i++) {
		fprintf(fp, "dat_record_offset[%d]: %d\n", i, hdr->dat_record_offset[i]);
	}
	fprintf(fp, "\n");
}
