/*
 *      Media database reader/writer for PortalPlayer platform.
 *
 *      Copyright (c) 2005-2007 Naoaki Okazaki
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 *
 */

/* $Id: pp1db.c 328 2007-02-10 17:50:11Z nyaochi $ */

#ifdef	HAVE_CONFIG_H
#include <config.h>
#endif/*HAVE_CONFIG_H*/
#ifdef	HAVE_STRING_H
#include <string.h>
#endif/*HAVE_STRING_H*/

#include <os.h>
#include <stdio.h>
#include <stdlib.h>
#include <pmplib/ucs2char.h>
#include <pmplib/filepath.h>
#include <pmplib/pmp.h>

#include "pp1db.h"
#include "util.h"
#include "crc.h"

#define	COMP(a, b)	((a)>(b))-((a)<(b))

#ifdef	PMP_IRIVERPLUS1_EXPORTS
#define	PMPPORTALPLAYER1API	__declspec(dllexport)
#else
#define	PMPPORTALPLAYER1API
#endif

void dat_h10_repr(const dat_t* record, FILE *fp);
int dat_h10_set(dat_t* dst, const pmp_music_record_t* src, const ucs2char_t* path_to_root);
int dat_h10_get(pmp_music_record_t* dst, const dat_t* src, const ucs2char_t* path_to_root);


void pp1db_init(pp1db_t* db, hdr_init_t hdrinit)
{
	memset(db, 0, sizeof(*db));

	// Initialize HDR.
	db->hdr = calloc(1, sizeof(hdr_t));
	if (db->hdr) {
		hdrinit(db->hdr);
	}

	// Initialize DAT.
	db->dat = NULL;
	
	// Initialize IDX.
	db->idx = (idx_t**)calloc(db->hdr->num_dat_fields, sizeof(idx_t*));;
}

void pp1db_finish(pp1db_t* db)
{
	uint32_t i;

	for (i = 0;i < db->hdr->num_dat_fields;++i) {
		free(db->idx[i]);
	}
	free(db->idx);

	free(db->dat);

	if (db->hdr) {
		hdr_finish(db->hdr);
		free(db->hdr);
	}

	memset(db, 0, sizeof(*db));
}

result_t pp1db_read(pp1db_t* db, const ucs2char_t* hdr_filename)
{
	result_t ret = 0;
	long size;
	uint8_t* buffer = NULL;
	FILE *fp = NULL;
	ucs2char_t filename[MAX_PATH];
	uint32_t i, j;

	if (!db->hdr) {
		return PMPERR_INCONSISTENCY;
	}

	// Read hdr.
	fp = ucs2fopen(hdr_filename, "rb");
	if (!fp) {
		return PMPERR_OPENFORREAD;
	}
	ret = fread_all(fp, &buffer, &size);
	if (ret != 0) {
		fclose(fp);
		return ret;
	}
	hdr_serialize(buffer, db->hdr, 0);
	free(buffer);
	fclose(fp);

#if 0	/* For analyzing a new (unknown) player. */
	hdr_repr(db->hdr, stdout);
	fflush(stdout);
	return 0;
#endif

	// Read dat.
	ucs2cpy(filename, hdr_filename);
	filepath_remove_filespec(filename);
	filepath_addslash(filename);
	ucs2cat(filename, filepath_skippath(db->hdr->pathname_dat));

	fp = ucs2fopen(filename, "rb");
	if (!fp) {
		return PMPERR_OPENFORREAD;
	}
	ret = fread_all(fp, &buffer, &size);
	if (ret != 0) {
		fclose(fp);
		return ret;
	}
	db->dat	= calloc(db->hdr->num_dat_entries, sizeof(dat_t));
	if (!db->dat) {
		fclose(fp);
		return PMPERR_INSUFFICIENTMEMORY;
	}
	for (j = 0;j < db->hdr->num_dat_entries;++j) {
		uint8_t *p = &buffer[db->hdr->dat_record_offset[j]];
		dat_init(&db->dat[j], db->hdr);
		if (dat_serialize(
			p,
			&db->dat[j],
			db->hdr->num_dat_fields,
			FIELDOFFSETS(db->hdr->dat_field_offset, j, db->hdr->param.max_fields),
			0
			) <= 0) {
			free(buffer);
			fclose(fp);
			return PMPERR_INSUFFICIENTMEMORY;
		}
	}
	free(buffer);
	fclose(fp);

	free(db->idx);
	db->idx	= (idx_t**)calloc(db->hdr->num_dat_fields, sizeof(idx_t*));
	if (!db->idx) {
		fclose(fp);
		return PMPERR_INSUFFICIENTMEMORY;
	}

	// Read indices.
	for (i = 0;i < db->hdr->num_dat_fields;++i) {
		if (db->hdr->fd[i].has_index) {
			uint8_t* p = NULL;
			// Read dat.
			ucs2cpy(filename, hdr_filename);
			filepath_remove_filespec(filename);
			filepath_addslash(filename);
			ucs2cat(filename, filepath_skippath(db->hdr->fd[i].index_pathname));

			fp = ucs2fopen(filename, "rb");
			if (!fp) {
				return PMPERR_OPENFORREAD;
			}
			ret = fread_all(fp, &buffer, &size);
			if (ret != 0) {
				fclose(fp);
				return ret;
			}
			p = buffer;
			db->idx[i] = calloc(db->hdr->num_dat_entries, sizeof(idx_t));
			if (!db->idx[i]) {
				fclose(fp);
				return PMPERR_INSUFFICIENTMEMORY;
			}
			for (j = 0;j < db->hdr->num_dat_entries;++j) {
				p += idx_serialize(p, &db->idx[i][j], 0);
			}
			free(buffer);
			fclose(fp);
		}
	}

	return 0;
}

result_t pp1db_write(pp1db_t* db, const ucs2char_t* hdr_filename)
{
	result_t ret = 0;
	size_t size;
	uint8_t *buffer = NULL, *p = NULL;
	FILE *fp = NULL;
	ucs2char_t filename[MAX_PATH];
	uint32_t i, j;
	uint32_t offset = 0;

	if (!db->hdr) {
		return PMPERR_INCONSISTENCY;
	}

	// Write indices.
	for (i = 0;i < db->hdr->num_dat_fields;++i) {
		if (db->hdr->fd[i].has_index) {
			// Generate a filename.
			ucs2cpy(filename, hdr_filename);
			filepath_remove_filespec(filename);
			filepath_addslash(filename);
			ucs2cat(filename, filepath_skippath(db->hdr->fd[i].index_pathname));

			fp = ucs2fopen(filename, "wb");
			if (!fp) {
				return PMPERR_OPENFORWRITE;
			}

			buffer = calloc(db->hdr->num_dat_entries, sizeof(idx_t));
			if (!buffer) {
				fclose(fp);
				return PMPERR_INSUFFICIENTMEMORY;
			}

			p = buffer;
			for (j = 0;j < db->hdr->num_dat_entries;++j) {
				p += idx_serialize(p, &db->idx[i][j], 1);
			}
			fwrite(buffer, 1, sizeof(idx_t) * db->hdr->num_dat_entries, fp);
			free(buffer);
			fclose(fp);
		}
	}

	// Generate a filename for dat.
	ucs2cpy(filename, hdr_filename);
	filepath_remove_filespec(filename);
	filepath_addslash(filename);
	ucs2cat(filename, filepath_skippath(db->hdr->pathname_dat));

	fp = ucs2fopen(filename, "wb");
	if (!fp) {
		return PMPERR_OPENFORWRITE;
	}
	p = buffer = calloc(db->hdr->num_dat_entries, field_descriptor_get_max_record_size(db->hdr));
	if (!buffer) {
		fclose(fp);
		return PMPERR_INSUFFICIENTMEMORY;
	}
	offset = 0;
	db->hdr->dat_record_offset[0] = offset;
	for (j = 0;j < db->hdr->num_dat_entries;++j) {
		size_t record_size = dat_serialize(
			p,
			&db->dat[j],
			db->hdr->num_dat_fields,
			FIELDOFFSETS(db->hdr->dat_field_offset, j, db->hdr->param.max_fields),
			1
			);
		p += record_size;
		offset += (uint32_t)record_size;
		db->hdr->dat_record_offset[j+1] = offset;
	}
	size = (size_t)(p - buffer);
	fwrite(buffer, 1, size, fp);
	free(buffer);
	fclose(fp);

	db->hdr->dat_size = (uint32_t)size;

	// Read hdr.
	fp = ucs2fopen(hdr_filename, "wb");
	if (!fp) {
		return PMPERR_OPENFORWRITE;
	}
	buffer = calloc(db->hdr->param.size, 1);
	if (!buffer) {
		fclose(fp);
		return PMPERR_INSUFFICIENTMEMORY;
	}
	hdr_serialize(buffer, db->hdr, 1);
	fwrite(buffer, 1, db->hdr->param.size, fp);
	free(buffer);
	fclose(fp);

	return 0;
}

result_t pp1db_repr(pp1db_t* db, FILE *fp, int level)
{
	uint32_t i, j;

	if (!db->hdr) {
		return PMPERR_INCONSISTENCY;
	}

	hdr_repr(db->hdr, fp);
	if (db->dat) {
		fprintf(fp, "// DAT\n");
		for (j = 0;j < db->hdr->num_dat_entries;++j) {
			fprintf(fp, "RECORD %d = {\n", j);
			db->hdr->param.proc_dat_repr(&db->dat[j], fp);
			fprintf(fp, "}\n");
		}
		fprintf(fp, "\n");
	}

	fprintf(fp, "// IDX\n");
	for (i = 0;i < db->hdr->num_dat_fields;++i) {
		if (db->idx[i]) {
			fprintf(fp, "IDX %d", i);
			if (db->hdr->fd[i].has_index) {
				fprints(fp, " (%s)", db->hdr->fd[i].index_pathname);
			}
			fprintf(fp, " = {\n");

			for (j = 0;j < db->hdr->num_dat_entries;++j) {
				const char *mbs = NULL;
				int index = db->idx[i][j].index;

				fprintf(fp, "  [%4d]: ", j);

				switch (db->hdr->fd[i].field_type) {
				case PP1DB_FIELDTYPE_STRING:
					fprints(fp, "%s", db->dat[index].fields[i].value.str);
					break;
				case PP1DB_FIELDTYPE_INTEGER:
					fprintf(fp, "%d", db->dat[index].fields[i].value.dword);
					break;
				}

				fprintf(fp, " {");
				idx_repr(&db->idx[i][j], fp);
				fprintf(fp, "}");
				fprintf(fp, "\n");
			}
			fprintf(fp, "}\n");
		} else {
			fprintf(fp, "IDX %d does not exist.\n", i);
		}
	}
	fprintf(fp, "\n");
	return 0;
}

int pp1db_is_supported_ext(const ucs2char_t* filename)
{
	static const ucs2char_t ucs2cs_mp3[] = {'.','m','p','3',0};
	static const ucs2char_t ucs2cs_wma[] = {'.','w','m','a',0};
	static const ucs2char_t ucs2cs_wav[] = {'.','w','a','v',0};

	return (
		filepath_hasext(filename, ucs2cs_mp3) ||
		filepath_hasext(filename, ucs2cs_wma) ||
		filepath_hasext(filename, ucs2cs_wav)
		) ? 1 : 0;
}

int pp1db_is_supported_codec(uint32_t codec)
{
	return (
		(codec == PMPCODEC_MPEGLAYER3) ||
		(codec == PMPCODEC_WMA) ||
		(codec == PMPCODEC_WAV)
		) ? 1 : 0;
}

// Record for sorting.
typedef struct {
	dat_field_t* field;
	uint32_t index;
} sort_item_t;

static int comp_ucs2(const void * __x, const void *__y)
{
	const sort_item_t *_x = (const sort_item_t *)__x;
	const sort_item_t *_y = (const sort_item_t *)__y;
	const ucs2char_t* x = _x->field->value.str;
	const ucs2char_t* y = _y->field->value.str;
	int ret = 0;
	if (!x || !*x) {
		ret = ((!y || !*y) ? 0 : 1);
	} else {
		ret = ((!y || !*y) ? -1 : ucs2icmp(x, y));
	}
	if (ret == 0) {
		ret = COMP(_x->index, _y->index);
	}
	return ret;
}

static int comp_dword(const void *_x, const void *_y)
{
	const sort_item_t *x = (const sort_item_t *)_x;
	const sort_item_t *y = (const sort_item_t *)_y;
	int ret = COMP(x->field->value.dword, y->field->value.dword);
	if (ret == 0) {
		ret = COMP(x->index, y->index);
	}
	return ret;
}

static int is_bigendian(void)
{
	ucs2char_t c = 0x1234;
	uint8_t* p = (uint8_t*)&c;
	return (*p == 0x12);
}

static void ucs2big2little(ucs2char_t* value)
{
	if (is_bigendian()) {
		for (;*value;value++) {
			ucs2char_t val = (*value << 8) | (*value >> 8);
			*value = val;
		}
	}
}



static uint32_t ucs2crc(const ucs2char_t* value)
{
	if (value) {
		uint32_t crc = 0;
		ucs2char_t* value_lowercase = ucs2dup(value);
		ucs2lwr(value_lowercase);
		ucs2big2little(value_lowercase);
		crc = (uint32_t)crcSlow(
			(const char *)value_lowercase,
			(int)ucs2len(value_lowercase) * sizeof(ucs2char_t)
			);
		ucs2free(value_lowercase);
		return crc;
	} else {
		return 0;
	}
}


result_t pp1db_set(pp1db_t* db, const pmp_music_record_t* records, uint32_t num_records, ucs2char_t* path_to_root)
{
	uint32_t i, j, n = 0;
	sort_item_t* si = NULL;

	if (!db->hdr) {
		return PMPERR_INCONSISTENCY;
	}

	dat_free_array(db->dat, db->hdr);

	// Count valid entries.
	for (i = 0;i < num_records;++i) {
		if (pp1db_is_supported_codec(records[i].codec)) {
			++n;
		}
	}

	// Allocate DAT records.
	db->dat = calloc(n, sizeof(dat_t));
	if (!db->dat) {
		pp1db_finish(db);
		return PMPERR_INSUFFICIENTMEMORY;
	}

	// Convert records from pmp_music_record_t to ip2db_record_t.
	for (i = 0, j = 0;i < num_records;++i) {
		const pmp_music_record_t* src = &records[i];
		dat_t* dst = &db->dat[j];

		// Skip invalid entries.
		if (!pp1db_is_supported_codec(records[i].codec)) {
			continue;
		}

		dat_init(dst, db->hdr);
		db->hdr->param.proc_dat_set(dst, src, path_to_root);
		dat_round(dst, db->hdr);

		++j;
	}

	// Allocate an array for sorting records.
	si = calloc(n, sizeof(sort_item_t));
	if (!si) {
		pp1db_finish(db);
		return PMPERR_INSUFFICIENTMEMORY;
	}

	// Generate indices.
	for (i = 0;i < db->hdr->num_dat_fields;++i) {
		if (db->hdr->fd[i].has_index) {
			// Allocate index array.
			free(db->idx[i]);
			db->idx[i] = calloc(n, sizeof(idx_t));
			if (!db->idx[i]) {
				pp1db_finish(db);
				return PMPERR_INSUFFICIENTMEMORY;
			}

			// Preparation for sorting records.
			for (j = 0;j < n;++j) {
				si[j].index = j;
				si[j].field = &db->dat[j].fields[i];
			}

			// Sort records according to the index type and fill the index array.
			switch (db->hdr->fd[i].field_type) {
			case PP1DB_FIELDTYPE_STRING:
				// Sort by string.
				qsort(si, n, sizeof(si[0]), comp_ucs2);
				for (j = 0;j < n;++j) {
					db->idx[i][j].status = 0;	// All elements are active.
					db->idx[i][j].index = si[j].index;	// Sorted index number to a record.
					db->idx[i][j].check_value = ucs2crc(si[j].field->value.str);	// CRC-32 of the value.
				}
				break;
			case PP1DB_FIELDTYPE_INTEGER:
				// Sort by uint32_t.
				qsort(si, n, sizeof(si[0]), comp_dword);
				for (j = 0;j < n;++j) {
					db->idx[i][j].status = 0;	// All elements are active.
					db->idx[i][j].index = si[j].index;	// Sorted index number to a record.
					db->idx[i][j].check_value = si[j].field->value.dword;	// Actual value
				}
				break;
			}
		}
	}

	free(si);

	db->hdr->num_dat_entries = n;
	db->hdr->num_dat_inactive_entries = 0;	// No inactive elements.
	return 0;
}

result_t pp1db_get(pp1db_t* db, pmp_music_record_t* records, uint32_t* num_records, ucs2char_t* path_to_root)
{
	uint32_t i = 0, j = 0, n = 0;

	// Count active dat elements.
	for (i = 0;i < db->hdr->num_dat_entries;++i) {
		if (db->dat[i].status == 0) {
			++n;
		}
	}

	// Check database consistency.
	if (db->hdr->num_dat_entries - db->hdr->num_dat_inactive_entries != n) {
		return PMPERR_INCONSISTENCY;
	}

	if (!records) {
		*num_records = n;
		return 0;
	}

	if (*num_records < n) {
		*num_records = n;
		return PMPERR_INSUFFICIENTMEMORY;
	}

	for (i = 0, j = 0;i < db->hdr->num_dat_entries;++i) {
		const dat_t* src = &db->dat[i];
		pmp_music_record_t* dst = &records[j];

		pmplib_record_init(dst);
		db->hdr->param.proc_dat_get(dst, src, path_to_root);

		++j;
	}

	*num_records = n;

	return 0;
}
