/* This file is part of Clementine.
   Copyright 2010, David Sansome <me@davidsansome.com>

   Clementine is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   Clementine is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Clementine.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "tagwidget.h"
#include "playlist/playlistitemmimedata.h"
#include "radio/lastfmservice.h"
#include "radio/radiomimedata.h"
#include "radio/radiomodel.h"
#include "smartplaylists/generator.h"
#include "smartplaylists/generatormimedata.h"
#include "smartplaylists/querygenerator.h"
#include "ui/flowlayout.h"
#include "ui/iconloader.h"

#include <QPainter>
#include <QPropertyAnimation>

const int TagWidgetTag::kIconSize = 16;
const int TagWidgetTag::kIconTextSpacing = 8;
const int TagWidgetTag::kHPadding = 6;
const int TagWidgetTag::kVPadding = 2;

TagWidgetTag::TagWidgetTag(const QIcon& icon, const QString& text, QWidget* parent)
  : QWidget(parent),
    text_(text),
    icon_(icon),
    opacity_(0.0),
    animation_(new QPropertyAnimation(this, "background_opacity", this))
{
  setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);
}

QSize TagWidgetTag::sizeHint() const {
  QSize text = fontMetrics().size(0, text_);
  QSize icon(kIconSize + kIconTextSpacing, kIconSize);

  return QSize(kHPadding + icon.width() + text.width() + kHPadding,
               kVPadding + qMax(text.height(), icon.height()) + kVPadding);
}

void TagWidgetTag::set_background_opacity(float opacity) {
  opacity_ = opacity;
  update();
}

void TagWidgetTag::enterEvent(QEvent*) {
  animation_->stop();
  animation_->setEndValue(1.0);
  animation_->setDuration(80);
  animation_->start();
}

void TagWidgetTag::leaveEvent(QEvent*) {
  animation_->stop();
  animation_->setEndValue(0.0);
  animation_->setDuration(160);
  animation_->start();
}

void TagWidgetTag::paintEvent(QPaintEvent*) {
  QPainter p(this);

  const QRect tag_rect(rect());
  const QRect icon_rect(tag_rect.topLeft() + QPoint(kHPadding, kVPadding),
                        QSize(kIconSize, kIconSize));
  const QRect text_rect(icon_rect.topRight() + QPoint(kIconTextSpacing, 0),
                        QSize(tag_rect.width() - icon_rect.right() - kIconTextSpacing - kHPadding,
                              icon_rect.height()));

  // Use the tag's opacity
  p.setOpacity(0.3 + opacity_ * 0.7);

  // Background
  QColor background_color = palette().color(QPalette::Highlight);
  background_color.setAlpha(128);

  p.setRenderHint(QPainter::Antialiasing);
  p.setPen(QPen(palette().dark(), 1.0));
  p.setBrush(background_color);
  p.drawRoundedRect(tag_rect, 5, 5);

  // Icon
  p.drawPixmap(icon_rect, icon_.pixmap(kIconSize));

  // Text
  p.setOpacity(1.0);
  p.setPen(palette().color(QPalette::Text));
  p.drawText(text_rect, text_);
}

void TagWidgetTag::mouseReleaseEvent(QMouseEvent*) {
  emit Clicked();
}

void TagWidgetTag::contextMenuEvent(QContextMenuEvent*) {
  emit Clicked();
}


TagWidget::TagWidget(Type type, QWidget* parent)
  : QWidget(parent),
    type_(type),
    menu_(NULL)
{
  setLayout(new FlowLayout(4, 6, 4));
}

void TagWidget::AddTag(const QString& tag) {
  if (tag.isEmpty())
    return;

  TagWidgetTag* widget = new TagWidgetTag(icon_, tag, this);
  connect(widget, SIGNAL(Clicked()), SLOT(TagClicked()));

  layout()->addWidget(widget);
  tags_ << widget;
}

void TagWidget::EnsureMenuCreated() {
  if (menu_)
    return;

  menu_ = new QMenu(this);
  switch (type_) {
    case Type_Tags:
      menu_->addAction(QIcon(":/last.fm/as.png"), tr("Play last.fm tag radio"),
                       this, SLOT(PlayLastFmTagRadio()));
      break;

    case Type_Artists:
      menu_->addAction(QIcon(":/last.fm/as.png"), tr("Play last.fm artist radio"),
                       this, SLOT(PlayLastFmArtistRadio()));
      menu_->addAction(IconLoader::Load("folder-sound"), tr("Play from my Library"),
                       this, SLOT(PlayFromLibrary()));
      break;
  }
}

void TagWidget::TagClicked() {
  TagWidgetTag* tag = qobject_cast<TagWidgetTag*>(sender());
  if (!tag)
    return;

  EnsureMenuCreated();

  context_item_ = tag->text();
  menu_->popup(tag->mapToGlobal(tag->rect().bottomLeft()));
}

void TagWidget::PlayLastFmArtistRadio() {
  PlayLastFm("lastfm://artist/%1/similarartists");
}

void TagWidget::PlayLastFmTagRadio() {
  PlayLastFm("lastfm://globaltags/%1");
}

void TagWidget::PlayFromLibrary() {
  using smart_playlists::GeneratorMimeData;
  using smart_playlists::GeneratorPtr;
  using smart_playlists::QueryGenerator;
  using smart_playlists::Search;
  using smart_playlists::SearchTerm;

  GeneratorPtr gen(new QueryGenerator(QString(), Search(
      Search::Type_And, Search::TermList() <<
        SearchTerm(SearchTerm::Field_Artist, SearchTerm::Op_Contains, context_item_),
      Search::Sort_FieldAsc, SearchTerm::Field_Album, 100)));
  emit AddToPlaylist(new GeneratorMimeData(gen));
}

void TagWidget::PlayLastFm(const QString& url_pattern) {
  LastFMService* last_fm = RadioModel::Service<LastFMService>();
  if (!last_fm->IsAuthenticated()) {
    last_fm->ShowConfig();
    return;
  }

  QUrl url(url_pattern.arg(context_item_));
  PlaylistItemPtr item(last_fm->PlaylistItemForUrl(url));
  if (!item)
    return;

  emit AddToPlaylist(new PlaylistItemMimeData(item));
}
