/*
 *  Copyright (C) 2005 Kouji TAKAO <kouji@netlab.jp>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "assert-macros.h"

#include "gpass/attribute-list.h"

/***********************************************************
 *
 * initialize/terminate,  setup/teardown
 *
 ***********************************************************/
static void
initialize(void)
{
    g_type_init();
}

static void
terminate(void)
{
}

static void
setup(void)
{
}

static void
teardown(void)
{
}

/***********************************************************
 *
 * test case
 *
 ***********************************************************/
START_TEST(test_prepend)
{
    GPassAttributeList *list = g_object_new(GPASS_TYPE_ATTRIBUTE_LIST, NULL);
    GPassAttribute *attr;
    GList *p;
    GError *error;
    
    attr = g_object_new(GPASS_TYPE_ATTRIBUTE,
                        "type", GPASS_ATTRIBUTE_TYPE_INTEGER,
                        "name", "integer", NULL);
    error = gpass_attribute_list_prepend(list, attr);
    ASSERT_NULL(error);

    attr = g_object_new(GPASS_TYPE_ATTRIBUTE,
                        "type", GPASS_ATTRIBUTE_TYPE_BOOLEAN,
                        "name", "boolean", NULL);
    error = gpass_attribute_list_prepend(list, attr);
    ASSERT_NULL(error);

    attr = g_object_new(GPASS_TYPE_ATTRIBUTE,
                        "type", GPASS_ATTRIBUTE_TYPE_STRING,
                        "name", "string", NULL);
    error = gpass_attribute_list_prepend(list, attr);
    ASSERT_NULL(error);

    p = list->list;
    attr = GPASS_ATTRIBUTE(p->data);
    ASSERT_EQUAL_ENUM(GPASS_ATTRIBUTE_TYPE_STRING, attr->type);
    ASSERT_EQUAL_STRING("string", attr->name);
    p = g_list_next(p);
    attr = GPASS_ATTRIBUTE(p->data);
    ASSERT_EQUAL_ENUM(GPASS_ATTRIBUTE_TYPE_BOOLEAN, attr->type);
    ASSERT_EQUAL_STRING("boolean", attr->name);
    p = g_list_next(p);
    attr = GPASS_ATTRIBUTE(p->data);
    ASSERT_EQUAL_ENUM(GPASS_ATTRIBUTE_TYPE_INTEGER, attr->type);
    ASSERT_EQUAL_STRING("integer", attr->name);
    p = g_list_next(p);
    
    g_object_unref(list);
}
END_TEST

START_TEST(test_append)
{
    GPassAttributeList *list = g_object_new(GPASS_TYPE_ATTRIBUTE_LIST, NULL);
    GPassAttribute *attr;
    GList *p;
    GError *error;
    
    attr = g_object_new(GPASS_TYPE_ATTRIBUTE,
                        "type", GPASS_ATTRIBUTE_TYPE_INTEGER,
                        "name", "integer", NULL);
    error = gpass_attribute_list_append(list, attr);
    ASSERT_NULL(error);
    ASSERT_EQUAL_INT(1, gpass_attribute_list_length(list));

    attr = g_object_new(GPASS_TYPE_ATTRIBUTE,
                        "type", GPASS_ATTRIBUTE_TYPE_BOOLEAN,
                        "name", "boolean", NULL);
    error = gpass_attribute_list_append(list, attr);
    ASSERT_NULL(error);
    ASSERT_EQUAL_INT(2, gpass_attribute_list_length(list));

    attr = g_object_new(GPASS_TYPE_ATTRIBUTE,
                        "type", GPASS_ATTRIBUTE_TYPE_STRING,
                        "name", "string", NULL);
    error = gpass_attribute_list_append(list, attr);
    ASSERT_NULL(error);
    ASSERT_EQUAL_INT(3, gpass_attribute_list_length(list));

    p = list->list;
    attr = GPASS_ATTRIBUTE(p->data);
    ASSERT_EQUAL_ENUM(GPASS_ATTRIBUTE_TYPE_INTEGER, attr->type);
    ASSERT_EQUAL_STRING("integer", attr->name);
    p = g_list_next(p);
    attr = GPASS_ATTRIBUTE(p->data);
    ASSERT_EQUAL_ENUM(GPASS_ATTRIBUTE_TYPE_BOOLEAN, attr->type);
    ASSERT_EQUAL_STRING("boolean", attr->name);
    p = g_list_next(p);
    attr = GPASS_ATTRIBUTE(p->data);
    ASSERT_EQUAL_ENUM(GPASS_ATTRIBUTE_TYPE_STRING, attr->type);
    ASSERT_EQUAL_STRING("string", attr->name);
    p = g_list_next(p);
    
    g_object_unref(list);
}
END_TEST

START_TEST(test_append__error)
{
    GPassAttributeList *list = g_object_new(GPASS_TYPE_ATTRIBUTE_LIST, NULL);
    GPassAttribute *attr;
    GError *error;
    
    attr = g_object_new(GPASS_TYPE_ATTRIBUTE,
                        "type", GPASS_ATTRIBUTE_TYPE_INTEGER, NULL);

    /* empty name */
    error = gpass_attribute_list_append(list, attr);
    ASSERT_NOT_NULL(error);
    g_error_free(error);

    g_object_set(attr, "name", "", NULL);
    error = gpass_attribute_list_append(list, attr);
    ASSERT_NOT_NULL(error);
    g_error_free(error);

    g_object_set(attr, "name", "integer", NULL);
    error = gpass_attribute_list_append(list, attr);
    ASSERT_NULL(error);
    
    /* already append */
    attr = g_object_new(GPASS_TYPE_ATTRIBUTE,
                        "type", GPASS_ATTRIBUTE_TYPE_INTEGER,
                        "name", "integer", NULL);
    error = gpass_attribute_list_append(list, attr);
    ASSERT_NOT_NULL(error);
    g_error_free(error);
    
    g_object_set(attr, "name", "integer2", NULL);
    error = gpass_attribute_list_append(list, attr);
    ASSERT_NULL(error);
    
    g_object_unref(list);
}
END_TEST

START_TEST(test_lookup)
{
    GPassAttributeList *list = g_object_new(GPASS_TYPE_ATTRIBUTE_LIST, NULL);
    GPassAttribute *attr;
    
    attr = g_object_new(GPASS_TYPE_ATTRIBUTE,
                        "type", GPASS_ATTRIBUTE_TYPE_INTEGER,
                        "name", "integer", NULL);
    gpass_attribute_list_append(list, attr);
    attr = g_object_new(GPASS_TYPE_ATTRIBUTE,
                        "type", GPASS_ATTRIBUTE_TYPE_BOOLEAN,
                        "name", "boolean", NULL);
    gpass_attribute_list_append(list, attr);
    attr = g_object_new(GPASS_TYPE_ATTRIBUTE,
                        "type", GPASS_ATTRIBUTE_TYPE_STRING,
                        "name", "string", NULL);
    gpass_attribute_list_append(list, attr);

    attr = gpass_attribute_list_lookup(list, "boolean");
    ASSERT_NOT_NULL(attr);
    ASSERT_EQUAL_ENUM(GPASS_ATTRIBUTE_TYPE_BOOLEAN, attr->type);
    ASSERT_EQUAL_STRING("boolean", attr->name);

    attr = gpass_attribute_list_lookup(list, "string");
    ASSERT_NOT_NULL(attr);
    ASSERT_EQUAL_ENUM(GPASS_ATTRIBUTE_TYPE_STRING, attr->type);
    ASSERT_EQUAL_STRING("string", attr->name);

    attr = gpass_attribute_list_lookup(list, "integer");
    ASSERT_NOT_NULL(attr);
    ASSERT_EQUAL_ENUM(GPASS_ATTRIBUTE_TYPE_INTEGER, attr->type);
    ASSERT_EQUAL_STRING("integer", attr->name);

    attr = gpass_attribute_list_lookup(list, "not found");
    ASSERT_NULL(attr);
    
    g_object_unref(list);
}
END_TEST

START_TEST(test_remove)
{
    GPassAttributeList *list = g_object_new(GPASS_TYPE_ATTRIBUTE_LIST, NULL);
    GPassAttribute *attr;
    
    attr = g_object_new(GPASS_TYPE_ATTRIBUTE,
                        "type", GPASS_ATTRIBUTE_TYPE_INTEGER,
                        "name", "integer", NULL);
    gpass_attribute_list_append(list, attr);
    attr = g_object_new(GPASS_TYPE_ATTRIBUTE,
                        "type", GPASS_ATTRIBUTE_TYPE_BOOLEAN,
                        "name", "boolean", NULL);
    gpass_attribute_list_append(list, attr);
    attr = g_object_new(GPASS_TYPE_ATTRIBUTE,
                        "type", GPASS_ATTRIBUTE_TYPE_STRING,
                        "name", "string", NULL);
    gpass_attribute_list_append(list, attr);

    gpass_attribute_list_remove(list, "boolean");
    attr = gpass_attribute_list_lookup(list, "boolean");
    ASSERT_NULL(attr);
    ASSERT_EQUAL_INT(2, gpass_attribute_list_length(list));

    gpass_attribute_list_remove(list, "string");
    attr = gpass_attribute_list_lookup(list, "string");
    ASSERT_NULL(attr);
    ASSERT_EQUAL_INT(1, gpass_attribute_list_length(list));

    gpass_attribute_list_remove(list, "integer");
    attr = gpass_attribute_list_lookup(list, "integer");
    ASSERT_NULL(attr);
    ASSERT_EQUAL_INT(0, gpass_attribute_list_length(list));

    attr = gpass_attribute_list_lookup(list, "not found");
    ASSERT_NULL(attr);
    
    g_object_unref(list);
}
END_TEST

START_TEST(test_create_cursor)
{
    GPassAttributeList *list = g_object_new(GPASS_TYPE_ATTRIBUTE_LIST, NULL);
    GPassAttribute *attr;
    GPassAttributeListCursor *cursor;
    GError *error;
    
    attr = g_object_new(GPASS_TYPE_ATTRIBUTE,
                        "type", GPASS_ATTRIBUTE_TYPE_INTEGER,
                        "name", "integer", NULL);
    gpass_attribute_list_append(list, attr);
    attr = g_object_new(GPASS_TYPE_ATTRIBUTE,
                        "type", GPASS_ATTRIBUTE_TYPE_BOOLEAN,
                        "name", "boolean", NULL);
    gpass_attribute_list_append(list, attr);
    attr = g_object_new(GPASS_TYPE_ATTRIBUTE,
                        "type", GPASS_ATTRIBUTE_TYPE_STRING,
                        "name", "string", NULL);
    gpass_attribute_list_append(list, attr);

    cursor = gpass_attribute_list_create_cursor(list);
    ASSERT_FALSE(gpass_attribute_list_cursor_is_done(cursor));
    g_object_get(cursor, "attribute", &attr, NULL);
    ASSERT_EQUAL_ENUM(GPASS_ATTRIBUTE_TYPE_INTEGER, attr->type);
    ASSERT_EQUAL_STRING("integer", attr->name);
    error = gpass_attribute_list_cursor_next(cursor);
    ASSERT_NULL(error);
    
    ASSERT_FALSE(gpass_attribute_list_cursor_is_done(cursor));
    g_object_get(cursor, "attribute", &attr, NULL);
    ASSERT_EQUAL_ENUM(GPASS_ATTRIBUTE_TYPE_BOOLEAN, attr->type);
    ASSERT_EQUAL_STRING("boolean", attr->name);
    error = gpass_attribute_list_cursor_next(cursor);
    ASSERT_NULL(error);
    
    ASSERT_FALSE(gpass_attribute_list_cursor_is_done(cursor));
    g_object_get(cursor, "attribute", &attr, NULL);
    ASSERT_EQUAL_ENUM(GPASS_ATTRIBUTE_TYPE_STRING, attr->type);
    ASSERT_EQUAL_STRING("string", attr->name);

    error = gpass_attribute_list_cursor_next(cursor);
    ASSERT_NULL(error);
    ASSERT_TRUE(gpass_attribute_list_cursor_is_done(cursor));
    error = gpass_attribute_list_cursor_next(cursor);
    ASSERT_NOT_NULL(error);
    g_error_free(error);
    
    g_object_unref(cursor);
    g_object_unref(list);
}
END_TEST

START_TEST(test_dump)
{
    GPassAttributeList *list = g_object_new(GPASS_TYPE_ATTRIBUTE_LIST, NULL);
    GString *buf = g_string_new("");
    GPassAttribute *attr;
    const guchar expected_data[] = {
        0x8c, 0x04, /* 524 */
        0x01, /* true */
        0x0a, 'T', 'h', 'e', ' ', 's', 't', 'r', 'i', 'n', 'g',
    };
    GError *error;
    
    attr = g_object_new(GPASS_TYPE_ATTRIBUTE,
                        "type", GPASS_ATTRIBUTE_TYPE_INTEGER,
                        "name", "integer", NULL);
    gpass_attribute_set(attr, 524);
    gpass_attribute_list_append(list, attr);
    attr = g_object_new(GPASS_TYPE_ATTRIBUTE,
                        "type", GPASS_ATTRIBUTE_TYPE_BOOLEAN,
                        "name", "boolean", NULL);
    gpass_attribute_set(attr, TRUE);
    gpass_attribute_list_append(list, attr);
    attr = g_object_new(GPASS_TYPE_ATTRIBUTE,
                        "type", GPASS_ATTRIBUTE_TYPE_STRING,
                        "name", "string", NULL);
    gpass_attribute_set(attr, "The string");
    gpass_attribute_list_append(list, attr);

    error = gpass_attribute_list_dump(list, &buf);
    ASSERT_NULL(error);
    ASSERT_EQUAL_MEMORY(expected_data, buf->str, G_N_ELEMENTS(expected_data));

    g_string_free(buf, TRUE);
    g_object_unref(list);
}
END_TEST

START_TEST(test_load)
{
    GPassAttributeList *list = g_object_new(GPASS_TYPE_ATTRIBUTE_LIST, NULL);
    GPassAttribute *attr;
    const guchar dump_data[] = {
        0x8c, 0x04, /* 524 */
        0x01, /* true */
        0x0a, 'T', 'h', 'e', ' ', 's', 't', 'r', 'i', 'n', 'g'
    };
    gint read_len;
    GPassAttributeListCursor *cursor;
    gint i;
    gboolean b;
    const gchar *s;
    GError *error;
    
    attr = g_object_new(GPASS_TYPE_ATTRIBUTE,
                        "type", GPASS_ATTRIBUTE_TYPE_INTEGER,
                        "name", "integer", NULL);
    gpass_attribute_list_append(list, attr);
    attr = g_object_new(GPASS_TYPE_ATTRIBUTE,
                        "type", GPASS_ATTRIBUTE_TYPE_BOOLEAN,
                        "name", "boolean", NULL);
    gpass_attribute_list_append(list, attr);
    attr = g_object_new(GPASS_TYPE_ATTRIBUTE,
                        "type", GPASS_ATTRIBUTE_TYPE_STRING,
                        "name", "string", NULL);
    gpass_attribute_list_append(list, attr);

    error = gpass_attribute_list_load(list, dump_data, G_N_ELEMENTS(dump_data),
                                      &read_len);
    ASSERT_NULL(error);
    ASSERT_EQUAL_INT(G_N_ELEMENTS(dump_data), read_len);
    
    cursor = gpass_attribute_list_create_cursor(list);
    
    g_object_get(cursor, "attribute", &attr, NULL);
    gpass_attribute_get(attr, &i);
    ASSERT_EQUAL_INT(524, i);
    gpass_attribute_list_cursor_next(cursor);
    
    g_object_get(cursor, "attribute", &attr, NULL);
    gpass_attribute_get(attr, &b);
    ASSERT_TRUE(b);
    gpass_attribute_list_cursor_next(cursor);

    g_object_get(cursor, "attribute", &attr, NULL);
    gpass_attribute_get(attr, &s);
    ASSERT_EQUAL_STRING("The string", s);

    g_object_unref(cursor);
    g_object_unref(list);
}
END_TEST

/***********************************************************
 *
 * suite / main
 *
 ***********************************************************/
static Suite *
test_suite(void)
{
    Suite *s = suite_create("GPassAttributeList");
    TCase *tc;
    
    tc = tcase_create("functions");
    suite_add_tcase(s, tc);
    tcase_add_checked_fixture(tc, setup, teardown);
    
    tcase_add_test(tc, test_prepend);
    tcase_add_test(tc, test_append);
    tcase_add_test(tc, test_append__error);
    tcase_add_test(tc, test_lookup);
    tcase_add_test(tc, test_remove);
    tcase_add_test(tc, test_create_cursor);
    tcase_add_test(tc, test_dump);
    tcase_add_test(tc, test_load);
    return s;
}

int
main(int argc, char *argv[])
{
    Suite *s;
    SRunner *sr;
    int nf;

    initialize();
    
    s = test_suite();
    sr = srunner_create(s);
    srunner_run_all(sr, CK_ENV);
    nf = srunner_ntests_failed(sr);
    srunner_free(sr);
    
    terminate();
    return (nf == 0) ? EXIT_SUCCESS : EXIT_FAILURE;
}
