/*
 *  Copyright (C) 2005 Kouji TAKAO <kouji@netlab.jp>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "assert-macros.h"
#include "test-util.h"

#include "gpass/configuration.h"
#include "gpass/root-entry.h"
#include "gpass/entry-factory.h"
#include "gpass/file.h"

/***********************************************************
 *
 * initialize/terminate,  setup/teardown
 *
 ***********************************************************/
static void
initialize(void)
{
    g_type_init();
}

static void
terminate(void)
{
}

static void
setup(void)
{
}

static void
teardown(void)
{
    gpass_configuration_finalize();
}

/***********************************************************
 *
 * test case
 *
 ***********************************************************/
#define HEADER {                                        \
    0x88, 0xf4, 0xde, 0x08, 0x41, 0x32, 0xe4, 0x49,     \
    0x0c, 0x50, 0x7b, 0xe2, 0x1d, 0xc9, 0x88, 0x17,     \
    0xc1, 0x48, 0xdc, 0xe9, 0x4e, 0xf6, 0xff, 0xd4      \
}
#define HEADER_LENGTH 24

START_TEST(test_s_create)
{
    FILE *file;
    gchar *path;
    guchar expected_data[] = HEADER;
    guchar data[HEADER_LENGTH];
    gsize s;
    GError *error;

    test_util_create_temporary_file("w", &file, &path);
    fclose(file);
    
    error = gpass_file_create(path, "qwerty");
    ASSERT_NULL(error);
    file = fopen(path, "r");
    s = fread(&data, sizeof(gchar), HEADER_LENGTH, file);
    fclose(file);
    ASSERT_EQUAL_INT(HEADER_LENGTH, s);
    ASSERT_EQUAL_MEMORY(expected_data, data, HEADER_LENGTH);
    
    unlink(path);
    g_free(path);
}
END_TEST

START_TEST(test_open)
{
    GPassFile *file;
    FILE *fp;
    gchar *path;
    guchar data[] = HEADER;
    GError *error;
    
    test_util_create_temporary_file("w", &fp, &path);
    fclose(fp);
    
    error = gpass_file_open(&file, path, "qwerty");
    ASSERT_NOT_NULL(error);
    g_error_free(error);
    
    fp = fopen(path, "w");
    fwrite(data, sizeof(guchar), HEADER_LENGTH, fp);
    fclose(fp);

    ASSERT_NULL(gpass_file_open(&file, path, "qwerty"));
    gpass_file_close(file);
    
    unlink(path);
    g_free(path);
}
END_TEST

START_TEST(test_read)
{
    GPassFile *file;
    FILE *fp;
    gchar *path;
    GPassEntryFactory *factory;
    GPassEntry *entries;
    GError *error;
    
    test_util_create_temporary_file("w", &fp, &path);
    fclose(fp);
    gpass_file_create(path, "qwerty");
    gpass_file_open(&file, path, "qwerty");

    factory = g_object_new(GPASS_TYPE_ENTRY_FACTORY, NULL);
    error = gpass_file_read(file, factory, &entries);
    g_object_unref(factory);
    gpass_file_close(file);
    
    ASSERT_NULL(error);
    ASSERT_TRUE(GPASS_IS_ROOT_ENTRY(entries));
    ASSERT_NULL(gpass_entry_next_sibling(entries));
    ASSERT_NULL(gpass_entry_prev_sibling(entries));
    ASSERT_NULL(gpass_entry_first_child(entries));
    g_object_unref(entries);

    unlink(path);
    g_free(path);
}
END_TEST

static void
check_children(GPassEntry *expected_entries, GPassEntry *entries)
{
    GPassEntry *p1, *p2;
    
    p1 = gpass_entry_first_child(expected_entries);
    p2 = gpass_entry_first_child(entries);
    while (p1 != NULL && p2 != NULL) {
        ASSERT_TRUE(gpass_entry_equal(p1, p2));
        if (gpass_entry_has_child(p1)) {
            check_children(p1, p2);
        }
        p1 = gpass_entry_next_sibling(p1);
        p2 = gpass_entry_next_sibling(p2);
    }
    ASSERT_NULL(p1);
    ASSERT_NULL(p2);
}

static void
check_wrote(GPassFile *file, GPassEntry *expected_entries,
            GPassEntryFactory *factory)
{
    GPassEntry *entries;
    GError *error;

    error = gpass_file_read(file, factory, &entries);
    ASSERT_NULL(error);

    ASSERT_TRUE(GPASS_IS_ROOT_ENTRY(expected_entries));
    ASSERT_NULL(gpass_entry_next_sibling(expected_entries));
    ASSERT_NULL(gpass_entry_prev_sibling(expected_entries));

    ASSERT_TRUE(GPASS_IS_ROOT_ENTRY(entries));
    ASSERT_NULL(gpass_entry_next_sibling(entries));
    ASSERT_NULL(gpass_entry_prev_sibling(entries));

    check_children(expected_entries, entries);

    g_object_unref(entries);
}

static void
check_backup(const gchar *path, GPassEntry *expected_entries,
             GPassEntryFactory *factory)
{
    GPassFile *backup_file;
    gchar *backup_path;
    GError *error;

    backup_path = g_strdup_printf("%s.bak", path);
    error = gpass_file_open(&backup_file, backup_path, "qwerty");
    ASSERT_NULL(error);

    check_wrote(backup_file, expected_entries, factory);
    gpass_file_close(backup_file);
    
    unlink(backup_path);
    g_free(backup_path);
}

START_TEST(test_write)
{
    GPassFile *file;
    FILE *fp;
    gchar *path;
    GPassEntryFactory *factory;
    GPassEntry *entry;
    GPassEntry *entries;
    GError *error;

    test_util_create_temporary_file("w", &fp, &path);
    fclose(fp);
    gpass_file_create(path, "qwerty");
    gpass_file_open(&file, path, "qwerty");

    factory = g_object_new(GPASS_TYPE_ENTRY_FACTORY, NULL);
    entries = g_object_new(GPASS_TYPE_ROOT_ENTRY, NULL);
    gpass_entry_factory_create_entry(factory, "general", &entry);
    g_object_set(entry,
                 "name", "entry-1",
                 "description", "The entry No.1",
                 "creation-time", 10000,
                 "modification-time", 20000,
                 "expiration", TRUE,
                 "expiration-time", 30000,
                 "hostname", "http://projects.netlabjp/gpass/",
                 "username", "gpass",
                 "password", "qwerty",
                 NULL);
    gpass_entry_append(entries, entry);

    error = gpass_file_write(file, entries);
    ASSERT_NULL(error);
    
    check_wrote(file, entries, factory);
    g_object_unref(entries);
    gpass_file_close(file);
    unlink(path);

    entries = g_object_new(GPASS_TYPE_ROOT_ENTRY, NULL);
    check_backup(path, entries, factory);
    g_object_unref(entries);

    g_object_unref(factory);
    g_free(path);
}
END_TEST

START_TEST(test_write__list)
{
    GPassFile *file;
    FILE *fp;
    gchar *path;
    GPassEntryFactory *factory;
    GPassEntry *entry;
    GPassEntry *entries;
    GError *error;

    test_util_create_temporary_file("w", &fp, &path);
    fclose(fp);
    gpass_file_create(path, "qwerty");
    gpass_file_open(&file, path, "qwerty");

    factory = g_object_new(GPASS_TYPE_ENTRY_FACTORY, NULL);
    entries = g_object_new(GPASS_TYPE_ROOT_ENTRY, NULL);
    gpass_entry_factory_create_entry(factory, "general", &entry);
    g_object_set(entry, "name", "entry-1", "description", "The entry No.1",
                 NULL);
    gpass_entry_append(entries, entry);
    gpass_entry_factory_create_entry(factory, "general", &entry);
    g_object_set(entry, "name", "entry-2", "description", "The entry No.2",
                 NULL);
    gpass_entry_append(entries, entry);
    gpass_entry_factory_create_entry(factory, "general", &entry);
    g_object_set(entry, "name", "entry-3", "description", "The entry No.3",
                 NULL);
    gpass_entry_append(entries, entry);
    gpass_entry_factory_create_entry(factory, "general", &entry);
    g_object_set(entry, "name", "entry-4", "description", "The entry No.4",
                 NULL);
    gpass_entry_append(entries, entry);

    error = gpass_file_write(file, entries);
    ASSERT_NULL(error);

    check_wrote(file, entries, factory);
    g_object_unref(entries);
    gpass_file_close(file);
    unlink(path);

    entries = g_object_new(GPASS_TYPE_ROOT_ENTRY, NULL);
    check_backup(path, entries, factory);
    g_object_unref(entries);

    g_object_unref(factory);
    g_free(path);
}
END_TEST

START_TEST(test_write__tree)
{
    GPassFile *file;
    FILE *fp;
    gchar *path;
    GPassEntryFactory *factory;
    GPassEntry *entries;
    GPassEntry *entry, *folder, *sub_folder;
    GError *error;

    test_util_create_temporary_file("w", &fp, &path);
    fclose(fp);
    gpass_file_create(path, "qwerty");
    gpass_file_open(&file, path, "qwerty");

    factory = g_object_new(GPASS_TYPE_ENTRY_FACTORY, NULL);
    
    /*
     * + root
     *   + entry-1
     *   + folder-2
     *     + entry-2-1
     *     + entry-2-2
     *   + folder-3
     *     + folder-3-1
     *       + entry-3-1-1
     *     + entry-3-2
     *     + folder-3-3
     *       + entry-3-3-1
     *       + entry-3-3-2
     *   + entry-4
     */
    entries = g_object_new(GPASS_TYPE_ROOT_ENTRY, NULL);
    gpass_entry_factory_create_entry(factory, "general", &entry);
    g_object_set(entry, "name", "entry-1", "description", "The entry No.1",
                 NULL);
    gpass_entry_append(entries, entry);
    
    gpass_entry_factory_create_entry(factory, "folder", &folder);
    g_object_set(folder, "name", "folder-2",
                 "description", "The folder No.2", NULL);
    gpass_entry_append(entries, folder);
    gpass_entry_factory_create_entry(factory, "general", &entry);
    g_object_set(entry, "name", "entry-2-1", "description", "The entry No.2-1",
                 NULL);
    gpass_entry_append(folder, entry);
    gpass_entry_factory_create_entry(factory, "general", &entry);
    g_object_set(entry, "name", "entry-2-2", "description", "The entry No.2-2",
                 NULL);
    gpass_entry_append(folder, entry);

    gpass_entry_factory_create_entry(factory, "folder", &folder);
    g_object_set(folder, "name", "folder-3",
                 "description", "The folder No.3", NULL);
    gpass_entry_append(entries, folder);
    gpass_entry_factory_create_entry(factory, "folder", &sub_folder);
    g_object_set(sub_folder, "name", "folder-3-1",
                 "description", "The folder No.3-1", NULL);
    gpass_entry_append(folder, sub_folder);
    gpass_entry_factory_create_entry(factory, "general", &entry);
    g_object_set(entry, "name", "entry-3-1-1",
                 "description", "The entry No.3-1-1", NULL);
    gpass_entry_append(sub_folder, entry);
    gpass_entry_factory_create_entry(factory, "general", &entry);
    g_object_set(entry, "name", "entry-3-2", "description", "The entry No.3-2",
                 NULL);
    gpass_entry_append(folder, entry);
    gpass_entry_factory_create_entry(factory, "folder", &sub_folder);
    g_object_set(sub_folder, "name", "folder-3-3",
                 "description", "The folder No.3-3", NULL);
    gpass_entry_append(folder, sub_folder);
    gpass_entry_factory_create_entry(factory, "general", &entry);
    g_object_set(entry, "name", "entry-3-3-1",
                 "description", "The entry No.3-3-1", NULL);
    gpass_entry_append(sub_folder, entry);
    gpass_entry_factory_create_entry(factory, "general", &entry);
    g_object_set(entry, "name", "entry-3-3-2",
                 "description", "The entry No.3-3-2", NULL);
    gpass_entry_append(sub_folder, entry);
    
    gpass_entry_factory_create_entry(factory, "general", &entry);
    g_object_set(entry, "name", "entry-4", "description", "The entry No.3",
                 NULL);
    gpass_entry_append(entries, entry);

    error = gpass_file_write(file, entries);
    ASSERT_NULL(error);

    check_wrote(file, entries, factory);
    g_object_unref(entries);
    gpass_file_close(file);
    unlink(path);

    entries = g_object_new(GPASS_TYPE_ROOT_ENTRY, NULL);
    check_backup(path, entries, factory);
    g_object_unref(entries);

    g_object_unref(factory);
    g_free(path);
}
END_TEST

START_TEST(test_write__backup)
{
    GPassFile *file;
    FILE *fp;
    gchar *path;
    GPassEntryFactory *factory;
    GPassEntry *entry;
    GPassEntry *entries;
    GPassEntry *backup_entries;
    GError *error;

    test_util_create_temporary_file("w", &fp, &path);
    fclose(fp);
    gpass_file_create(path, "qwerty");
    gpass_file_open(&file, path, "qwerty");

    factory = g_object_new(GPASS_TYPE_ENTRY_FACTORY, NULL);
    backup_entries = g_object_new(GPASS_TYPE_ROOT_ENTRY, NULL);
    gpass_entry_factory_create_entry(factory, "general", &entry);
    g_object_set(entry,
                 "name", "entry-1",
                 "description", "The entry No.1",
                 "creation-time", 10000,
                 "modification-time", 20000,
                 "expiration", TRUE,
                 "expiration-time", 30000,
                 "hostname", "http://projects.netlabjp/gpass/",
                 "username", "gpass",
                 "password", "qwerty",
                 NULL);
    gpass_entry_append(backup_entries, entry);
    gpass_entry_factory_create_entry(factory, "general", &entry);
    g_object_set(entry,
                 "name", "entry-2",
                 "description", "The entry No.2",
                 "creation-time", 10000,
                 "modification-time", 20000,
                 "expiration", TRUE,
                 "expiration-time", 30000,
                 "hostname", "http://projects.netlabjp/gpass/",
                 "username", "gpass",
                 "password", "qwerty",
                 NULL);
    gpass_entry_append(backup_entries, entry);
    error = gpass_file_write(file, backup_entries);
    ASSERT_NULL(error)

    entries = g_object_new(GPASS_TYPE_ROOT_ENTRY, NULL);
    gpass_entry_factory_create_entry(factory, "general", &entry);
    g_object_set(entry,
                 "name", "entry-1",
                 "description", "The entry No.1",
                 "creation-time", 10000,
                 "modification-time", 20000,
                 "expiration", TRUE,
                 "expiration-time", 30000,
                 "hostname", "http://projects.netlabjp/gpass/",
                 "username", "gpass",
                 "password", "qwerty",
                 NULL);
    gpass_entry_append(entries, entry);
    gpass_entry_factory_create_entry(factory, "general", &entry);
    g_object_set(entry,
                 "name", "entry-2",
                 "description", "The entry No.2",
                 "creation-time", 10000,
                 "modification-time", 20000,
                 "expiration", TRUE,
                 "expiration-time", 30000,
                 "hostname", "http://projects.netlabjp/gpass/",
                 "username", "gpass",
                 "password", "qwerty",
                 NULL);
    gpass_entry_append(entries, entry);
    error = gpass_file_write(file, entries);
    ASSERT_NULL(error)
    
    check_wrote(file, entries, factory);
    g_object_unref(entries);
    gpass_file_close(file);
    unlink(path);

    check_backup(path, backup_entries, factory);
    g_object_unref(backup_entries);

    g_object_unref(factory);
    g_free(path);
}
END_TEST

/***********************************************************
 *
 * suite / main
 *
 ***********************************************************/
static Suite *
test_suite(void)
{
    Suite *s = suite_create("GPassFile");
    TCase *tc;
    
    tc = tcase_create("functions");
    suite_add_tcase(s, tc);
    tcase_add_checked_fixture(tc, setup, teardown);

    tcase_add_test(tc, test_s_create);
    tcase_add_test(tc, test_open);
    tcase_add_test(tc, test_read);
    tcase_add_test(tc, test_write);
    tcase_add_test(tc, test_write__list);
    tcase_add_test(tc, test_write__tree);
    tcase_add_test(tc, test_write__backup);
    return s;
}

int
main(int argc, char *argv[])
{
    Suite *s;
    SRunner *sr;
    int nf;

    initialize();
    
    s = test_suite();
    sr = srunner_create(s);
    srunner_run_all(sr, CK_ENV);
    nf = srunner_ntests_failed(sr);
    srunner_free(sr);
    
    terminate();
    return (nf == 0) ? EXIT_SUCCESS : EXIT_FAILURE;
}
