/*
 *  Copyright (C) 2005 Kouji TAKAO <kouji@netlab.jp>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "assert-macros.h"

#include "gpass/pack.h"

/***********************************************************
 *
 * initialize/terminate,  setup/teardown
 *
 ***********************************************************/
static void
initialize(void)
{
}

static void
terminate(void)
{
}

static void
setup(void)
{
}

static void
teardown(void)
{
}

/***********************************************************
 *
 * test case
 *
 ***********************************************************/
START_TEST(test_pack_number)
{
    GString *buf = g_string_new(NULL);
    const guchar expected_data_10[] = { 0x0a };
    const guchar expected_data_524[] = { 0x8c, 0x04 };
    const guchar expected_data_50040[] = { 0xf8, 0x86, 0x03 };
    const guchar expected_data_1234567890[] = {
        0xd2, 0x85, 0xd8, 0xcc, 0x04
    };

    buf = g_string_assign(buf, "");
    gpass_pack_number(10, &buf);
    ASSERT_EQUAL_MEMORY(expected_data_10, buf->str,
                        G_N_ELEMENTS(expected_data_10));

    buf = g_string_assign(buf, "");
    gpass_pack_number(524, &buf);
    ASSERT_EQUAL_MEMORY(expected_data_524, buf->str, 
                        G_N_ELEMENTS(expected_data_524));

    buf = g_string_assign(buf, "");
    gpass_pack_number(50040, &buf);
    ASSERT_EQUAL_MEMORY(expected_data_50040, buf->str, 
                        G_N_ELEMENTS(expected_data_50040));

    buf = g_string_assign(buf, "");
    gpass_pack_number(1234567890, &buf);
    ASSERT_EQUAL_MEMORY(expected_data_1234567890, buf->str, 
                        G_N_ELEMENTS(expected_data_1234567890));

    g_string_free(buf, TRUE);
}
END_TEST

START_TEST(test_unpack_number)
{
    const guchar pack_data_0[] = { 0x00 };
    const guchar pack_data_10[] = { 0x0a };
    const guchar pack_data_524[] = { 0x8c, 0x04 };
    const guchar pack_data_50040[] = { 0xf8, 0x86, 0x03 };
    const guchar pack_data_error1[] = { 0x81, 0x81, 0x81, 0x81, 0x81 };
    const guchar pack_data_error2[] = { 0x81, 0x81 };
    gint val;
    gsize read_len;
    GError *error;

    error = gpass_unpack_number(pack_data_0, G_N_ELEMENTS(pack_data_0),
                                &val, &read_len);
    ASSERT_NULL(error);
    ASSERT_EQUAL_INT(0, val);
    ASSERT_EQUAL_INT(1, read_len);

    error = gpass_unpack_number(pack_data_10, G_N_ELEMENTS(pack_data_10),
                                &val, &read_len);
    ASSERT_NULL(error);
    ASSERT_EQUAL_INT(10, val);
    ASSERT_EQUAL_INT(1, read_len);

    error = gpass_unpack_number(pack_data_524, G_N_ELEMENTS(pack_data_524),
                                &val, &read_len);
    ASSERT_NULL(error);
    ASSERT_EQUAL_INT(524, val);
    ASSERT_EQUAL_INT(2, read_len);

    error = gpass_unpack_number(pack_data_50040, G_N_ELEMENTS(pack_data_50040),
                                &val, &read_len);
    ASSERT_NULL(error);
    ASSERT_EQUAL_INT(50040, val);
    ASSERT_EQUAL_INT(3, read_len);

    error = gpass_unpack_number(pack_data_error1,
                                G_N_ELEMENTS(pack_data_error1),
                                &val, &read_len);
    ASSERT_NOT_NULL(error);
    g_error_free(error);

    error = gpass_unpack_number(pack_data_error2,
                                G_N_ELEMENTS(pack_data_error2),
                                &val, &read_len);
    ASSERT_NOT_NULL(error);
    g_error_free(error);
}
END_TEST

START_TEST(test_pack_string)
{
    const guchar expected_data_1[] = {
        0x06, 's', 't', 'r', 'i', 'n', 'g', 
    };
    const guchar expected_data_2[] = {
        0x1a,
        'a', 'b', 'c', 'd', 'e', 'f', 'g',
        'h', 'i', 'j', 'k', 'l', 'm', 'n',
        'o', 'p', 'q', 'r', 's', 't', 'u',
        'v', 'w', 'x', 'y', 'z', 
    };
    const guchar expected_data_null[] = {
        0x00
    };
    GString *buf = g_string_new(NULL);
    
    buf = g_string_assign(buf, "");
    gpass_pack_string("string", &buf);
    ASSERT_EQUAL_MEMORY(expected_data_1, buf->str,
                        G_N_ELEMENTS(expected_data_1));

    buf = g_string_assign(buf, "");
    gpass_pack_string("abcdefghijklmnopqrstuvwxyz", &buf);
    ASSERT_EQUAL_MEMORY(expected_data_2, buf->str, 
                        G_N_ELEMENTS(expected_data_2));

    buf = g_string_assign(buf, "");
    gpass_pack_string(NULL, &buf);
    ASSERT_EQUAL_MEMORY(expected_data_null, buf->str, 
                        G_N_ELEMENTS(expected_data_null));

    buf = g_string_assign(buf, "");
    gpass_pack_string("", &buf);
    ASSERT_EQUAL_MEMORY(expected_data_null, buf->str, 
                        G_N_ELEMENTS(expected_data_null));
    
    g_string_free(buf, TRUE);
}
END_TEST

START_TEST(test_unpack_string)
{
    const guchar pack_data_1[] = {
        0x06, 's', 't', 'r', 'i', 'n', 'g'
    };
    const guchar pack_data_2[] = {
        0x1a,
        'a', 'b', 'c', 'd', 'e', 'f', 'g',
        'h', 'i', 'j', 'k', 'l', 'm', 'n',
        'o', 'p', 'q', 'r', 's', 't', 'u',
        'v', 'w', 'x', 'y', 'z'
    };
    const guchar pack_data_null[] = {
        0x00
    };
    const guchar pack_data_error1[] = {
        0x1a, 's', 'h', 'o', 'r', 't'
    };
    const guchar pack_data_error2[] = {
        0x0b, 'T', 'h', 'e', ' ', 's', 't', 'r', 'i', 'n', 'g'
    };
    GString *str = g_string_new(NULL);
    gint read_len;
    GError *error;

    str = g_string_assign(str, "");
    error = gpass_unpack_string(pack_data_1, G_N_ELEMENTS(pack_data_1),
                                &str, &read_len);
    ASSERT_NULL(error);
    ASSERT_EQUAL_STRING("string", str->str);
    ASSERT_EQUAL_INT(G_N_ELEMENTS(pack_data_1), read_len);

    str = g_string_assign(str, "");
    error = gpass_unpack_string(pack_data_2, G_N_ELEMENTS(pack_data_2),
                                &str, &read_len);
    ASSERT_NULL(error);
    ASSERT_EQUAL_STRING("abcdefghijklmnopqrstuvwxyz", str->str);
    ASSERT_EQUAL_INT(G_N_ELEMENTS(pack_data_2), read_len);
    
    str = g_string_assign(str, "");
    error = gpass_unpack_string(pack_data_null, G_N_ELEMENTS(pack_data_null),
                                &str, &read_len);
    ASSERT_NULL(error);
    ASSERT_EQUAL_STRING("", str->str);
    ASSERT_EQUAL_INT(G_N_ELEMENTS(pack_data_null), read_len);

    str = g_string_assign(str, "");
    error = gpass_unpack_string(pack_data_error1,
                                G_N_ELEMENTS(pack_data_error1),
                                &str, &read_len);
    ASSERT_NOT_NULL(error);
    g_error_free(error);

    str = g_string_assign(str, "");
    error = gpass_unpack_string(pack_data_error2,
                                G_N_ELEMENTS(pack_data_error2),
                                &str, &read_len);
    ASSERT_NOT_NULL(error);
    g_error_free(error);

    g_string_free(str, TRUE);
}
END_TEST

/***********************************************************
 *
 * suite / main
 *
 ***********************************************************/
static Suite *
test_suite(void)
{
    Suite *s = suite_create("GPassPack");
    TCase *tc;
    
    tc = tcase_create("functions");
    suite_add_tcase(s, tc);
    tcase_add_checked_fixture(tc, setup, teardown);
    
    tcase_add_test(tc, test_pack_number);
    tcase_add_test(tc, test_unpack_number);
    tcase_add_test(tc, test_pack_string);
    tcase_add_test(tc, test_unpack_string);
    return s;
}

int
main(int argc, char *argv[])
{
    Suite *s;
    SRunner *sr;
    int nf;

    initialize();
    
    s = test_suite();
    sr = srunner_create(s);
    srunner_run_all(sr, CK_ENV);
    nf = srunner_ntests_failed(sr);
    srunner_free(sr);
    
    terminate();
    return (nf == 0) ? EXIT_SUCCESS : EXIT_FAILURE;
}
