#include <gtk/gtk.h>
#include <stdio.h>
#include <string.h>
#include <libintl.h>

#include "config.h"
#include "gsshfs.h"
#include "connections.h"


void define_connections_model () {
	GtkTreeView *view;
	GtkListStore *model;
	
	view = (GtkTreeView *) gtk_builder_get_object (widgetstree, "connections");
	model = gtk_list_store_new (COLUMNS_COUNT,
		G_TYPE_STRING, /* host */
		G_TYPE_STRING, /* port */
                G_TYPE_STRING, /* username */
		G_TYPE_STRING, /* directory */
		G_TYPE_STRING, /* mountpoint */
		G_TYPE_BOOLEAN); /* connected */
	gtk_tree_view_set_model (view, (GtkTreeModel *) model);
}


gchar **load_mounts () {
       gchar **mounts;
       gchar *output;
       gchar *delimiter = "\n";
     
       g_spawn_command_line_sync ("mount", &output, NULL, NULL, NULL);
        mounts = g_strsplit (output, delimiter, 0);
        g_free (output);
       
        return mounts;
}


gboolean is_mounted (gchar *host, gchar *username, gchar *directory, gchar **mounts) {
       gchar *mount;
       gboolean mounted;
       gint i;
     
       mount = g_strdup_printf ("%s@%s:%s", username, host, directory);
       #ifdef DEBUG
               printf ("DEBUG: <is_mounted> MOUNT=%s\n", mount);
       #endif
       mounted = FALSE;
       i=0;
       while (i < g_strv_length (mounts) && ! mounted) {
               #ifdef DEBUG
                       printf ("DEBUG: <is_mounted> MOUNTS[%d]=%s\n", i, mounts[i]);
               #endif
               if (g_ascii_strncasecmp (mount, mounts[i], strlen (mount)) == 0) {
                       mounted = TRUE;
                       #ifdef DEBUG
                               printf ("DEBUG: <is_mounted> MOUNTS[%d]=%s is mounted\n", i, mounts[i]);
                       #endif
               }
               i++;
       }
     
       g_free (mount);
     
       return mounted;
}


void load_connections () {
	gchar *configfilepath;
	FILE *configfile;
	gchar line[201], *host, *port, *username, *directory, *mountpoint;
	gchar **mounts;
	gint i;
	GtkTreeIter iter;
	GtkListStore *model;
	GtkTreeView *view;
	
	view = (GtkTreeView *) gtk_builder_get_object (widgetstree, "connections");
	model = (GtkListStore *) gtk_tree_view_get_model (view);
	
	configfilepath = g_strdup_printf ("%s/%s", g_getenv ("HOME"), CONFIGFILE);
	#ifdef DEBUG
		g_printf ("DEBUG: <load_connections> CONFIG FILE=%s\n", configfilepath);
	#endif
	
	configfile = fopen (configfilepath, "r");
	g_free (configfilepath);
	
	if (configfile != NULL) {
		mounts = load_mounts ();
		
		fgets (line, sizeof(line), configfile);
		while (! feof (configfile)) {
			#ifdef DEBUG
				g_printf ("DEBUG: <load_connections> LINE= %s", line);
			#endif
			
			i = 0; host = line;
			while (line [i] != ':') i++; line[i] = '\0'; port = line + i + 1;
			while (line [i] != ':') i++; line[i] = '\0'; username = line + i + 1;
			while (line [i] != ':') i++; line[i] = '\0'; directory = line + i + 1;
			while (line [i] != ':') i++; line[i] = '\0'; mountpoint = line + i + 1;
			while (line [i] != '\n') i++; line[i] = '\0';
			
			#ifdef DEBUG
				g_printf ("DEBUG: <load_connections> host=%s; port=%s; username=%s; directory=%s; mountpoint=%s.\n", host, port, username, directory, mountpoint);
			#endif
			
			gtk_list_store_append (model, &iter);
			gtk_list_store_set (model, &iter,
				HOST, host,
				PORT, port,
				USERNAME, username,
				DIRECTORY, directory,
				MOUNTPOINT, mountpoint,
				MOUNTED, is_mounted (host, username, directory, mounts), -1);
			
			fgets (line, sizeof(line), configfile);
		}
		fclose (configfile);
		
		g_strfreev (mounts);
	} else {
		#ifdef DEBUG
			g_printf ("DEBUG: <load_connections> Config file not found.\n");
		#endif
	}
}


void save_connections () {
	gchar *configfilepath;
	FILE *configfile;
	gchar *host, *port, *username, *directory, *mountpoint;
	GtkTreeIter iter;
	GtkListStore *model;
	GtkTreeView *view;
	gboolean available_iter; 
	
	configfilepath = g_strdup_printf ("%s/%s", g_getenv ("HOME"), CONFIGFILE);
	configfile = fopen (configfilepath, "w");
	g_free (configfilepath);
	
	if (configfile != NULL) {
		view = (GtkTreeView *) gtk_builder_get_object (widgetstree, "connections");
		model = (GtkListStore *) gtk_tree_view_get_model (view);
		available_iter = gtk_tree_model_get_iter_first ((GtkTreeModel *) model, &iter);
		while (TRUE == available_iter) {
			gtk_tree_model_get ((GtkTreeModel *) model, &iter, HOST, &host, PORT, &port, USERNAME, &username, 
				DIRECTORY, &directory, MOUNTPOINT, &mountpoint, -1);
			fprintf (configfile, "%s:%s:%s:%s:%s\n", host, port, username, directory, mountpoint);
			g_free (host);
			g_free (port);
			g_free (username);
			g_free (directory);
			g_free (mountpoint);
			available_iter = gtk_tree_model_iter_next ((GtkTreeModel *) model, &iter);
		}
		fclose (configfile);
	}
	edited = FALSE;
}


gboolean on_connection_change (
	GtkTreeSelection	*selection,
	GtkTreeModel		*model,
        GtkTreePath		*path,
	gboolean			path_currently_selected,
	gpointer			userdata)
{
	GtkTreeIter iter;
	gboolean mounted;
	GtkButton *remove, *mount, *umount;
	
	if (! path_currently_selected) {
		gtk_tree_model_get_iter (model, &iter, path);
		gtk_tree_model_get (model, &iter, MOUNTED, &mounted, -1);
		remove = (GtkButton *) gtk_builder_get_object (widgetstree, "remove");
		mount = (GtkButton *) gtk_builder_get_object (widgetstree, "mount");
		umount = (GtkButton *) gtk_builder_get_object (widgetstree, "umount");
		if (mounted) {
			gtk_widget_set_sensitive ((GtkWidget *) remove, FALSE);
			gtk_widget_set_sensitive ((GtkWidget *) mount, FALSE);
			gtk_widget_set_sensitive ((GtkWidget *) umount, TRUE);
		} else {
			gtk_widget_set_sensitive ((GtkWidget *) remove, TRUE);
			gtk_widget_set_sensitive ((GtkWidget *) mount, TRUE);
			gtk_widget_set_sensitive ((GtkWidget *) umount, FALSE);
		}
			
	}
	return TRUE;
}


void add_connection (GtkWidget *widget, gpointer user_data) {
	GtkTreeIter iter;
	GtkTreeView *view;
	GtkListStore *model;
	
	view = (GtkTreeView *) gtk_builder_get_object (widgetstree, "connections");
	model = (GtkListStore *) gtk_tree_view_get_model (view);
	gtk_list_store_append (model, &iter);
	gtk_list_store_set (model, &iter, HOST, "", PORT, "22", USERNAME, "", DIRECTORY, "", MOUNTPOINT, "", MOUNTED, FALSE, -1);
	edited = TRUE;
}


void remove_connection (GtkWidget *widget, gpointer user_data) {
	GtkTreeIter iter;
	GtkTreeSelection *selection;
	GtkTreeView *view;
	GtkListStore *model;
	
	view = (GtkTreeView *) gtk_builder_get_object (widgetstree, "connections");
	selection = gtk_tree_view_get_selection (view);
	gtk_tree_selection_get_selected (selection, (GtkTreeModel **) &model, &iter);
	gtk_list_store_remove (model, &iter);
	
	gtk_widget_set_sensitive ((GtkWidget *) gtk_builder_get_object (widgetstree, "remove"), FALSE);
	gtk_widget_set_sensitive ((GtkWidget *) gtk_builder_get_object (widgetstree, "mount"), FALSE);
	gtk_widget_set_sensitive ((GtkWidget *) gtk_builder_get_object (widgetstree, "umount"), FALSE);
	
	edited = TRUE;
}


void on_cell_edited (
	GtkCellRendererText	*cell,
	gchar				*path_string,
	gchar				*new_text,
	gpointer				user_data)
{
	GtkTreeIter iter;
	GtkListStore *model;
	GtkTreeView *view;
	gint column;
	gchar *host, *username, *directory, *mountpoint;
	gint i;
	gboolean mounted;
	
	column = GPOINTER_TO_INT (user_data);
	#ifdef DEBUG
		g_printf ("DEBUG <on_cell_edited> COLUMN=%d, TEXT=%s\n", column, new_text);
	#endif
	
	view = (GtkTreeView *) gtk_builder_get_object (widgetstree, "connections");
	model = (GtkListStore *) gtk_tree_view_get_model (view);
	gtk_tree_model_get_iter_from_string ((GtkTreeModel *) model, &iter, path_string);
	gtk_tree_model_get ((GtkTreeModel *) model, &iter, MOUNTED, &mounted, -1);
	
	if (mounted) {
		umount_connection(NULL, NULL);
		gtk_tree_model_get ((GtkTreeModel *) model, &iter, MOUNTED, &mounted, -1);
	}
	
	if (! mounted) {
		gtk_list_store_set (model, &iter, column, new_text, -1);

		if (HOST ==  column || USERNAME == column || DIRECTORY == column) {
			gtk_tree_model_get ((GtkTreeModel *) model, &iter, HOST, &host, USERNAME, &username, DIRECTORY, &directory, -1);
			for (i = 0; i < strlen(directory); i++) {
				if ('/' == directory[i]) directory[i] = ':';
			}
			#ifdef DEBUG
				g_printf ("DEBUG <on_cell_edited> HOST=%s, USERNAME=%s, DIRECTORY=%s\n", host, username, directory);
			#endif
			mountpoint = g_strdup_printf ("%s@%s:%s", username, host, directory);
			gtk_list_store_set (model, &iter, MOUNTPOINT, mountpoint, -1);
			g_free (host);
			g_free (username);
			g_free (directory);
			g_free (mountpoint);
		}
		
		edited = TRUE;
	}
}


void mount_connection (GtkWidget *widget, gpointer user_data) {
	GtkTreeIter iter;
	GtkTreeSelection *selection;
	GtkTreeView *view;
	GtkListStore *model;
	gchar *host, *port, *username, *directory, *mountpoint;
	gchar *mountpointabsolutepath, *command, **sshfscommand, *sshfscommandline;
	gboolean success;
	GPid sshfspid;
	GtkTreePath * path;
	
	view = (GtkTreeView *) gtk_builder_get_object (widgetstree, "connections");
	selection = gtk_tree_view_get_selection (view);
	gtk_tree_selection_get_selected (selection, (GtkTreeModel **) &model, &iter);
	gtk_tree_model_get ((GtkTreeModel *) model, &iter, HOST, &host, PORT, &port, USERNAME, &username, 
		DIRECTORY, &directory, MOUNTPOINT, &mountpoint, -1);
	mountpointabsolutepath = g_strdup_printf ("%s/%s", g_getenv ("HOME"), mountpoint);
	
	if (0 == strlen (host) || 0 == strlen (port) || 0 == strlen (username) || 0 == strlen (mountpoint))
		display_error (gettext ("Error: please fill host, port, username and mount point fields."));
	else {
		if (check_server_key (host)) {
			command = g_strdup_printf ("mkdir -p '%s'", mountpointabsolutepath);
			success = g_spawn_command_line_sync (command, NULL, NULL, NULL, NULL);
			g_free (command);
			if (! success)
				display_error (gettext ("Error: mount point can't be created."));
			else {
				path = gtk_tree_model_get_path ((GtkTreeModel *) model, &iter);
				if (g_getenv ("SSH_ASKPASS") == NULL)
					g_setenv ("SSH_ASKPASS", GSSHFS_ASKPASS, TRUE);
				sshfscommandline = g_strdup_printf ("setsid sshfs -o idmap=user -o umask=022 -p %s \"%s@%s:%s\" \"%s\"", 
					port, username, host, directory, mountpointabsolutepath);
				g_shell_parse_argv (sshfscommandline, NULL, &sshfscommand, NULL);
				g_free (sshfscommandline);
				g_spawn_async (NULL, sshfscommand, NULL, 
					G_SPAWN_SEARCH_PATH|G_SPAWN_DO_NOT_REAP_CHILD, 
					NULL, NULL, &sshfspid, NULL);
				g_child_watch_add (sshfspid, (GChildWatchFunc) sshfscommandfinished, (gpointer) path);
				g_strfreev (sshfscommand);
			}
		}
	}
	
	g_free (host);
	g_free (port);
	g_free (username);
	g_free (directory);
	g_free (mountpoint);
	g_free (mountpointabsolutepath);
}


void umount_connection (GtkWidget *widget, gpointer user_data) {
	GtkTreeIter iter;
	GtkTreeSelection *selection;
	GtkTreeView *view;
	GtkListStore *model;
	gchar *host, *username, *directory, *mountpoint;
	gchar *mountpointabsolutepath, *command;
	gchar **mounts;
	
	view = (GtkTreeView *) gtk_builder_get_object (widgetstree, "connections");
	selection = gtk_tree_view_get_selection (view);
	gtk_tree_selection_get_selected (selection, (GtkTreeModel **) &model, &iter);
	gtk_tree_model_get ((GtkTreeModel *) model, &iter, HOST, &host, USERNAME, &username, DIRECTORY, &directory, MOUNTPOINT, &mountpoint, -1);
	mountpointabsolutepath = g_strdup_printf ("%s/%s", g_getenv ("HOME"), mountpoint);
	
	command = g_strdup_printf ("fusermount -u '%s'", mountpointabsolutepath);
	g_spawn_command_line_sync (command, NULL, NULL, NULL, NULL);
	g_free (command);
	
	mounts = load_mounts();
	if (! is_mounted (host, username, directory, mounts)) {
		gtk_list_store_set (model, &iter, MOUNTED, FALSE, -1);
		gtk_widget_set_sensitive ((GtkWidget *) gtk_builder_get_object (widgetstree, "remove"), TRUE);
		gtk_widget_set_sensitive ((GtkWidget *) gtk_builder_get_object (widgetstree, "mount"), TRUE);
		gtk_widget_set_sensitive ((GtkWidget *) gtk_builder_get_object (widgetstree, "umount"), FALSE);
		
		command = g_strdup_printf ("rmdir '%s'", mountpointabsolutepath);
		g_spawn_command_line_sync (command, NULL, NULL, NULL, NULL);
		g_free (command);
	} else {
		display_error (gettext ("Error: mount point is occupied."));
	}
	g_strfreev (mounts);
	
	g_free (host);
	g_free (username);
	g_free (directory);
	g_free (mountpoint);
	g_free (mountpointabsolutepath);
}


gboolean check_server_key (gchar *host) {
	gchar *command, *sshkeyscanoutput, *grepknownhostsfileoutput, *sshkeygenoutput;
	gchar *hostkey, *knownhostkey;
	gchar *hostkeyfingerprint;
	gchar *message;
	gboolean success;
	GtkMessageDialog *confirmhostkeydialog;
	gchar *knowhostsfilepath;
	FILE *knowhostsfile;
	gint i;
	
	success = FALSE;
	command = g_strdup_printf ("ssh-keyscan %s", host);
	g_spawn_command_line_sync (command, &sshkeyscanoutput, NULL, NULL, NULL);
	g_free (command);
	
	if (0 == strlen (sshkeyscanoutput)) {
		display_error (gettext ("Error: could not retrieve host key."));
	} else {
		for (hostkey = sshkeyscanoutput; *hostkey != ' '; hostkey++); hostkey++;
		#ifdef DEBUG
			g_printf ("DEBUG <check_server_key> HOST KEY=%s\n", hostkey);
		#endif
		
		command = g_strdup_printf ("mkdir -p %s/.ssh", g_getenv ("HOME"));
		g_spawn_command_line_sync (command, NULL, NULL, NULL, NULL);
		g_free (command);
		
		command = g_strdup_printf ("touch %s/.ssh/known_hosts", g_getenv ("HOME"));
		g_spawn_command_line_sync (command, NULL, NULL, NULL, NULL);
		g_free (command);
		
		command = g_strdup_printf ("grep %s %s/.ssh/known_hosts", host, g_getenv ("HOME")); //retrieving host key from know host file (if present)
		g_spawn_command_line_sync (command, &grepknownhostsfileoutput, NULL, NULL, NULL);
		g_free (command);
		
		command = g_strdup_printf ("sed -i '/%s/d' %s/.ssh/known_hosts", host, g_getenv ("HOME")); //removing host from know hosts file (if present)
		g_spawn_command_line_sync (command, NULL, NULL, NULL, NULL);
		g_free (command);
		
		knowhostsfilepath = g_strdup_printf ("%s/.ssh/known_hosts", g_getenv ("HOME")); //adding host with new key to know hosts file (needed for ssh-keygen)
		knowhostsfile = fopen (knowhostsfilepath, "a");
		g_free (knowhostsfilepath);
		fprintf (knowhostsfile, "%s %s", host, hostkey);
		fclose (knowhostsfile);
		
		command = g_strdup_printf ("ssh-keygen -l -F %s", host); //retrieving host key fingerprint
		g_spawn_command_line_sync (command, &sshkeygenoutput, NULL, NULL, NULL);
		g_free (command);
		for (hostkeyfingerprint = sshkeygenoutput; *hostkeyfingerprint != '\n'; hostkeyfingerprint++); hostkeyfingerprint++;
		for (; *hostkeyfingerprint != ' '; hostkeyfingerprint++); hostkeyfingerprint++;
		for (i=0;  hostkeyfingerprint[i] != ' '; i++); hostkeyfingerprint[i] = '\0';
		if (0 == strlen (grepknownhostsfileoutput)) {
			message = g_strdup_printf ("%s\n%s", gettext ("Please verify carefully host key fingerprint:"), hostkeyfingerprint);
		} else {
			for (knownhostkey = grepknownhostsfileoutput; *knownhostkey != ' '; knownhostkey++); knownhostkey++;
			if (0 == g_ascii_strcasecmp (knownhostkey, hostkey)) {
				success = TRUE;
			} else {
				message = g_strdup_printf ("%s\n%s", gettext ("Warning: host key has changed ! New host key fingerprint is:"), hostkeyfingerprint);
			}
		}
		
		if (! success) { //host key not found in known hosts file or changed
			confirmhostkeydialog = (GtkMessageDialog *) gtk_builder_get_object (widgetstree, "confirmhostkeydialog");
			gtk_message_dialog_set_markup (confirmhostkeydialog, message);
			if (GTK_RESPONSE_YES == gtk_dialog_run ((GtkDialog *) confirmhostkeydialog)) {
				success = TRUE;
			} else {
				command = g_strdup_printf ("sed -i '/%s/d' %s/.ssh/known_hosts", host, g_getenv ("HOME")); //removing host from know hosts file
				g_spawn_command_line_sync (command, NULL, NULL, NULL, NULL);
				g_free (command);
			}
			g_free (message);
		}
		
		g_free (grepknownhostsfileoutput);
		g_free (sshkeygenoutput);
	}
	g_free (sshkeyscanoutput);
	
	return success;
}


void sshfscommandfinished (GPid pid, gint status, gpointer data) {
	GtkTreePath *path = (GtkTreePath *) data;
	gchar **mounts;
	GtkTreeIter iter;
	GtkTreeView *view;
	GtkListStore *model;
	gchar *host, *port, *username, *directory, *mountpoint;
	gchar *mountpointabsolutepath;
	gchar *command;
	
	mounts = load_mounts();
	view = (GtkTreeView *) gtk_builder_get_object (widgetstree, "connections");
	model = (GtkListStore *) gtk_tree_view_get_model (view);
	gtk_tree_model_get_iter ((GtkTreeModel *) model, &iter, path);
	gtk_tree_model_get ((GtkTreeModel *) model, &iter, HOST, &host, USERNAME, &username, 
		DIRECTORY, &directory, MOUNTPOINT, &mountpoint, -1);
	mountpointabsolutepath = g_strdup_printf ("%s/%s", g_getenv ("HOME"), mountpoint);
	
	if (is_mounted (host, username, directory, mounts)) {
		gtk_list_store_set (model, &iter, MOUNTED, TRUE, -1);
		gtk_widget_set_sensitive ((GtkWidget *) gtk_builder_get_object (widgetstree, "remove"), FALSE);
		gtk_widget_set_sensitive ((GtkWidget *) gtk_builder_get_object (widgetstree, "mount"), FALSE);
		gtk_widget_set_sensitive ((GtkWidget *) gtk_builder_get_object (widgetstree, "umount"), TRUE);
		
		command = g_strdup_printf ("xdg-open '%s'", mountpointabsolutepath);
		g_spawn_command_line_sync (command, NULL, NULL, NULL, NULL);
		g_free (command);
		
		save_connections();
	} else {
		command = g_strdup_printf ("rmdir '%s'", mountpointabsolutepath);
		g_spawn_command_line_sync (command, NULL, NULL, NULL, NULL);
		g_free (command);
		display_error (gettext ("Error: cancelled, or invalid username / password."));
	}
	
	g_strfreev (mounts);
	g_free (host);
	g_free (username);
	g_free (directory);
	g_free (mountpoint);
	g_free (mountpointabsolutepath);
	gtk_tree_path_free (path);
}