/*
 * gst-bin-filter.c - Source for GstBinFilter
 *
 * Copyright (C) 2010 Collabora Ltd.
 *  @author: Youness Alaoui <youness.alaoui@collabora.co.uk>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/**
 * SECTION:gst-bin-filter
 * @short_description: A customizable bin filter
 *
 * This filter will let you insert any kind of custom bin inside your pipeline
 * or in the middle of your #GstFilterManager.
 */


#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif


#include <gst/filters/gst-bin-filter.h>
#include <gst/filters/gst-filter-helper.h>

G_DEFINE_TYPE (GstBinFilter, gst_bin_filter, GST_TYPE_FILTER);

static void gst_bin_filter_get_property (GObject * object,
    guint property_id, GValue * value, GParamSpec * pspec);
static void gst_bin_filter_set_property (GObject * object,
    guint property_id, const GValue * value, GParamSpec * pspec);
static void gst_bin_filter_dispose (GObject * object);
static GstPad *gst_bin_filter_apply (GstFilter * filter,
    GstBin * bin, GstPad * pad);
static GstPad *gst_bin_filter_revert (GstFilter * filter,
    GstBin * bin, GstPad * pad);


/* properties */
enum
{
  PROP_DESCRIPTION = 1,
  LAST_PROPERTY
};


struct _GstBinFilterPrivate
{
  gchar *description;
};

static void
gst_bin_filter_class_init (GstBinFilterClass * klass)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS (klass);
  GstFilterClass *gstfilter_class = GST_FILTER_CLASS (klass);

  g_type_class_add_private (klass, sizeof (GstBinFilterPrivate));

  gobject_class->get_property = gst_bin_filter_get_property;
  gobject_class->set_property = gst_bin_filter_set_property;
  gobject_class->dispose = gst_bin_filter_dispose;

  gstfilter_class->apply = gst_bin_filter_apply;
  gstfilter_class->revert = gst_bin_filter_revert;
  gstfilter_class->name = "bin";

  /**
   * GstBinFilter:description:
   *
   * The bin description to be added to the pipeline when the filter is applied
   */
  g_object_class_install_property (gobject_class, PROP_DESCRIPTION,
      g_param_spec_string ("description", "bin description",
          "The bin description to add to the pipeline",
          NULL, G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS));

}

static void
gst_bin_filter_init (GstBinFilter * self)
{
  GstBinFilterPrivate *priv =
      G_TYPE_INSTANCE_GET_PRIVATE (self, GST_TYPE_BIN_FILTER,
      GstBinFilterPrivate);

  self->priv = priv;
  priv->description = NULL;
}


static void
gst_bin_filter_get_property (GObject * object,
    guint property_id, GValue * value, GParamSpec * pspec)
{
  GstBinFilter *self = GST_BIN_FILTER (object);
  GstBinFilterPrivate *priv = self->priv;

  GST_FILTER_LOCK (self);
  switch (property_id) {
    case PROP_DESCRIPTION:
      g_value_set_string (value, priv->description);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
      break;
  }
  GST_FILTER_UNLOCK (self);
}

static void
gst_bin_filter_set_property (GObject * object,
    guint property_id, const GValue * value, GParamSpec * pspec)
{
  GstBinFilter *self = GST_BIN_FILTER (object);
  GstBinFilterPrivate *priv = self->priv;

  GST_FILTER_LOCK (self);
  switch (property_id) {
    case PROP_DESCRIPTION:
      g_free (priv->description);
      priv->description = g_value_dup_string (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
      break;
  }
  GST_FILTER_UNLOCK (self);
}

static void
gst_bin_filter_dispose (GObject * object)
{
  GstBinFilter *self = GST_BIN_FILTER (object);
  GstBinFilterPrivate *priv = self->priv;

  if (priv->description)
    g_free (priv->description);
  priv->description = NULL;

  G_OBJECT_CLASS (gst_bin_filter_parent_class)->dispose (object);
}

/**
 * gst_bin_filter_new:
 * @description: The bin description to apply
 *
 * Creates a new bin filter.
 * The bin filter will insert a custom pipeline as defined by the bin
 * @description in the pipeline whenever the filter gets applied
 *
 * Returns: A new #GstBinFilter
 */
GstBinFilter *
gst_bin_filter_new (const gchar * description)
{

  g_return_val_if_fail (description, NULL);

  return g_object_new (GST_TYPE_BIN_FILTER, "description", description, NULL);
}


static GstPad *
gst_bin_filter_apply (GstFilter * filter, GstBin * bin, GstPad * pad)
{

  GstBinFilter *self = GST_BIN_FILTER (filter);
  GstBinFilterPrivate *priv = self->priv;

  return gst_filter_apply_element_by_description (bin, pad,
      priv->description, NULL, NULL);
}

static GstPad *
gst_bin_filter_revert (GstFilter * filter, GstBin * bin, GstPad * pad)
{
  return gst_filter_revert_element_default (bin, pad, NULL);
}
