#include "limooviewer.h"
#include "thumbnailbar.h"

#include <QVBoxLayout>
#include <QDir>
#include <QUrl>
#include <QAction>
#include <QMainWindow>
#include <QTimer>
#include <QDebug>

#include <SMasterIcons>
#include <Silicon>

class LimooViewerPrivate
{
public:
    ThumbnailBar *thumbnailbar;
    QVBoxLayout *layout;

    QAction *fullscreen_act;
    bool started_on_initialize;
};

LimooViewer::LimooViewer( SApplication *parent ) :
    SPage( tr("Limoo Viewer") , parent )
{
    p =new LimooViewerPrivate;
        p->started_on_initialize = Silicon::onInitialize();

    p->thumbnailbar = new ThumbnailBar();
        p->thumbnailbar->setAcceptDrops( false );

    p->layout = new QVBoxLayout( this );
        p->layout->addWidget( p->thumbnailbar );
        p->layout->setContentsMargins( 0 , 0 , 0 , 0 );

    setAcceptDrops( true );
    init_actions();

    connect( p->thumbnailbar , SIGNAL(closed()) , SLOT(toggleFullscreen()) );

    if( Silicon::onInitialize() )
        Silicon::initializeFinished( this , SLOT(toggleFullscreen()) , Qt::QueuedConnection );
}

void LimooViewer::init_actions()
{
    p->fullscreen_act = new QAction( this );
        p->fullscreen_act->setText( tr("Full Screen") );
        p->fullscreen_act->setIcon( SMasterIcons::icon(QSize(48,48),"view-fullscreen.png") );
        p->fullscreen_act->setCheckable( true );
        p->fullscreen_act->setChecked( false );
        p->fullscreen_act->setShortcut( QKeySequence( Qt::CTRL + Qt::Key_F ) );

    menuPanel()->addAction( p->fullscreen_act );

    connect( p->fullscreen_act  , SIGNAL(triggered()) , this , SLOT(fullScreen()) );

    p->thumbnailbar->addAction( p->fullscreen_act );
}

void LimooViewer::add( const QString & path )
{
    QFileInfo file( path );
    if( !file.isDir() )
    {
        p->thumbnailbar->add( path , file.fileName() );
    }
    else
    {
        QStringList files = QDir(path).entryList(QDir::NoDotAndDotDot|QDir::Files);
        for( int i=0 ; i<files.count() ; i++ )
            add( path + "/" + files.at(i) );
    }
}

ThumbnailBarItem *LimooViewer::at( int index )
{
    return p->thumbnailbar->at(index);
}

int LimooViewer::count() const
{
    return p->thumbnailbar->count();
}

void LimooViewer::fullScreen()
{
    if( !p->fullscreen_act->isChecked() )
    {
        Silicon::mainWindow()->show();
        p->layout->addWidget( p->thumbnailbar );
        p->thumbnailbar->setWindowFlags( Qt::Widget );
        p->thumbnailbar->showNormal();
        p->thumbnailbar->setBackgroundOpacity( 1 );
        p->thumbnailbar->show();
    }
    else
    {
        Silicon::mainWindow()->hide();
        p->layout->removeWidget( p->thumbnailbar );
        p->thumbnailbar->setAutoFillBackground( true );
        p->thumbnailbar->setWindowFlags( Qt::Window );
        p->thumbnailbar->showFullScreen();
        p->thumbnailbar->setBackgroundOpacity( 0.7 );
        p->thumbnailbar->show();
    }
}

void LimooViewer::toggleFullscreen()
{
    if( p->started_on_initialize && p->fullscreen_act->isChecked() )
    {
        parentApp()->quit();
        return;
    }

    p->fullscreen_act->setChecked( !p->fullscreen_act->isChecked() );
    fullScreen();
}

void LimooViewer::dragEnterEvent( QDragEnterEvent *event )
{
    QList<QUrl> urls_list = event->mimeData()->urls();
    if( urls_list.isEmpty() )
        return ;

    event->acceptProposedAction();
}

void LimooViewer::dropEvent( QDropEvent *event )
{
    QList<QUrl> urls_list = event->mimeData()->urls();
    if( urls_list.isEmpty() )
        return ;

    for( int i=0 ; i<urls_list.count() ; i++ )
    {
        QString path = urls_list.at(i).path();

#ifdef Q_OS_WIN32
        if( path[0] == '/' )
            path.remove( 0 , 1 );
#endif

        add( path );
    }

    event->acceptProposedAction();
}

LimooViewer::~LimooViewer()
{
    delete p;
}
