#include "thumbnailbar.h"
#include "imageprovider.h"
#include "thumbnailbaritem.h"
#include "limoocolors.h"

#include <QList>
#include <QVariant>
#include <QMetaObject>
#include <QDeclarativeEngine>
#include <QDeclarativeContext>
#include <QGraphicsObject>
#include <QCloseEvent>
#include <QWheelEvent>
#include <QDebug>

class ThumbnailBarPrivate
{
public:
    QList<ThumbnailBarItem*> list;
    QString id;

    QDeclarativeEngine *engine;
    QDeclarativeContext *root;
    QObject *thumbnailbar;
};

ThumbnailBar::ThumbnailBar( QWidget *parent ) :
    QDeclarativeView( parent )
{
    p = new ThumbnailBarPrivate;
        p->id = ImageProvider::getId();

    p->engine = engine();
        p->engine->addImageProvider( QLatin1String("pixmaps"), new ImageProvider() );

    p->root = rootContext();
        p->root->setContextProperty("Colors",new LimooColors(this));
        p->root->setContextProperty("View",this);

    setSource(QUrl("qrc:/limoo/qml/main.qml"));
    setAttribute(Qt::WA_TranslucentBackground);
    setStyleSheet(QString("background: transparent"));
    setResizeMode( QDeclarativeView::SizeRootObjectToView );

    p->thumbnailbar = rootObject()->findChild<QObject*>("thumbnailbar");
}

ThumbnailBarItem *ThumbnailBar::add( const QString & path , const QString & name )
{
    return insert( count() , path , name );
}

ThumbnailBarItem *ThumbnailBar::insert( int index , const QString & path , const QString & name )
{
    ThumbnailBarItem *item = insert( index , name );
    if( !item->setPixmap( path ) )
    {
        delete item;
        item = 0;
    }

    return item;
}

ThumbnailBarItem *ThumbnailBar::insert( int index , const QString & name )
{
    QVariant returnedValue;
    QMetaObject::invokeMethod( p->thumbnailbar , "add" , Q_RETURN_ARG(QVariant, returnedValue) );

    QObject     *obj  = qvariant_cast<QObject*>(returnedValue);
    ThumbnailBarItem *item = new ThumbnailBarItem( obj );
        obj->setProperty( "index" , index );
        item->setText( name );

    p->list.insert( index , item );
    p->thumbnailbar->setProperty("count",count());

    connect( item , SIGNAL(destroyed(QObject*)) , SLOT(itemDestroyed(QObject*)) );
    reindex();
    return item;
}

void ThumbnailBar::remove( ThumbnailBarItem *item )
{
    delete item;
}

void ThumbnailBar::remove( int index )
{
    delete p->list.at(index);
}

ThumbnailBarItem *ThumbnailBar::at( int index ) const
{
    return p->list.at(index);
}

int ThumbnailBar::indexOf( ThumbnailBarItem *item ) const
{
    return p->list.indexOf(item);
}

int ThumbnailBar::count() const
{
    return p->list.count();
}

void ThumbnailBar::setBackgroundOpacity( qreal opacity )
{
    rootObject()->setProperty( "back_opacity" , opacity );
}

qreal ThumbnailBar::backgroundOpacity() const
{
    return rootObject()->property("back_opacity").toReal();
}

void ThumbnailBar::itemDestroyed( QObject *obj )
{
    p->list.removeOne( static_cast<ThumbnailBarItem*>(obj) );
    p->thumbnailbar->setProperty("count",count());
    reindex();
}

void ThumbnailBar::reindex()
{
    for( int i=0 ; i<p->list.count() ; i++ )
        p->list.at(i)->obj()->setProperty("index",i);
}

void ThumbnailBar::closeEvent( QCloseEvent *event )
{
    emit closed();
    event->ignore();
}

void ThumbnailBar::wheelEvent( QWheelEvent *event )
{
    QDeclarativeView::wheelEvent( event );
    emit wheel( event->delta() );
}

QObject *ThumbnailBar::obj() const
{
    return p->thumbnailbar;
}

ThumbnailBar::~ThumbnailBar()
{
    delete p;
}
