#include "photoviewer.h"

#include <QSize>
#include <QUrl>
#include <QPalette>
#include <QColor>
#include <QVBoxLayout>
#include <QAction>
#include <QList>
#include <QToolBar>
#include <QSlider>
#include <QMargins>
#include <QFileInfo>
#include <QDir>

#include <SPictureWidget>
#include <SScrollWidget>
#include <SDrawer>
#include <SMasterIcons>
#include <SThumbnailBar>

class PhotoViewerPrivate
{
public:
    QAction *next_photo_action;
    QAction *prev_photo_action;

    QAction *fullscreen_act;
    QAction *thumbnail_bar_act;
    QAction *fit_act;
    QAction *original_act;
    QSlider *zoom_slider;
    QToolBar *panel_toolbar;

    QVBoxLayout *base_layout;

    SDrawer *draw_widget;

    SScrollWidget *scroll_widget;
    SThumbnailBar *thumbnail_bar;

    bool full_screen_stt;
};

PhotoViewer::PhotoViewer( const QString & name , SApplication *parent )
    : SPage( name , parent )
{
    p = new PhotoViewerPrivate;

    p->full_screen_stt = false;

    QPalette palette = QWidget::palette();
        palette.setColor( QPalette::Window , QColor(0,0,0) );
        palette.setColor( QPalette::WindowText , QColor(255,255,255) );

    QPalette palette1 = palette;
        palette1.setColor( QPalette::Window , QColor(13,13,13) );

    setPalette( palette );
    setAcceptDrops( true );

    p->draw_widget = new SDrawer();

    p->thumbnail_bar = new SThumbnailBar();
        p->thumbnail_bar->setPalette( palette1 );

    p->scroll_widget = new SScrollWidget();
        p->scroll_widget->setFixedHeight( 73 );
        p->scroll_widget->setEnableVerticalArrow( false );
        p->scroll_widget->setWidget( p->thumbnail_bar );

    p->base_layout = new QVBoxLayout( this );
        p->base_layout->addWidget( p->draw_widget );
        p->base_layout->addWidget( p->scroll_widget );
        p->base_layout->setContentsMargins( 0 , 0 , 0 , 0 );


    QObject::connect( p->thumbnail_bar , SIGNAL(currentItemChanged(QString)) , this , SLOT(showPicture(QString)) );

    setupActions();

    setAutoFillBackground( true );
    p->thumbnail_bar->setAutoFillBackground( true );

    setMinimumSize( QSize(720,350) );
}

void PhotoViewer::setupActions()
{
    QPalette palette = QWidget::palette();
        palette.setColor( QPalette::Window , QColor(0,0,0,73) );
        palette.setColor( QPalette::WindowText , QColor(255,255,255) );

    p->fullscreen_act = new QAction( this );
        p->fullscreen_act->setText( tr("Full Screen") );
        p->fullscreen_act->setIcon( SMasterIcons::icon(QSize(48,48),"view-fullscreen.png") );
        p->fullscreen_act->setCheckable( false );
        p->fullscreen_act->setChecked( false );

    p->thumbnail_bar_act = new QAction( this );
        p->thumbnail_bar_act->setText( tr("Thumbnail Bar") );
        p->thumbnail_bar_act->setIcon( SMasterIcons::icon(QSize(48,48),"view-preview.png") );
        p->thumbnail_bar_act->setCheckable( true );
        p->thumbnail_bar_act->setChecked( true );

    p->fit_act = new QAction( this );
        p->fit_act->setText( tr("Fit") );
        p->fit_act->setIcon( SMasterIcons::icon(QSize(48,48),"page-zoom.png") );
        p->fit_act->setCheckable( true );
        p->fit_act->setChecked( true );

    p->original_act = new QAction( this );
        p->original_act->setText( tr("Original") );
        p->original_act->setIcon( SMasterIcons::icon(QSize(48,48),"zoom-original.png") );
        p->original_act->setCheckable( true );
        p->original_act->setChecked( false );

    p->zoom_slider = new QSlider( Qt::Horizontal );
        p->zoom_slider->setMaximum( 500 );
        p->zoom_slider->setValue( 0 );

    p->panel_toolbar = new QToolBar( this );
        p->panel_toolbar->addAction( p->thumbnail_bar_act );
        p->panel_toolbar->addAction( p->fit_act );
        p->panel_toolbar->addAction( p->original_act );
        p->panel_toolbar->addWidget( p->zoom_slider );
        p->panel_toolbar->setFixedWidth( 400 );
        p->panel_toolbar->setPalette( palette );
        p->panel_toolbar->setToolButtonStyle( Qt::ToolButtonTextBesideIcon );



    p->next_photo_action = new QAction( SMasterIcons::icon( QSize(48,48) , "go-next.png" ) , "Next" , this );
    p->prev_photo_action = new QAction( SMasterIcons::icon( QSize(48,48) , "go-previous.png" ) , "Previous" , this );

    //p->fullscreen_act->setShortcut( QKeySequence( Qt::Key_Control + Qt::Key_F ) );
    p->next_photo_action->setShortcut( QKeySequence( Qt::Key_Right ) );
    p->prev_photo_action->setShortcut( QKeySequence( Qt::Key_Left ) );

    addAction( p->next_photo_action );
    addAction( p->prev_photo_action );

    menuPanel()->addAction( p->fullscreen_act );
    menuPanel()->addAction( p->thumbnail_bar_act );
    menuPanel()->addSeparator();
    menuPanel()->addAction( p->fit_act );
    menuPanel()->addAction( p->original_act );
    menuPanel()->addSeparator();
    menuPanel()->addAction( p->next_photo_action );
    menuPanel()->addAction( p->prev_photo_action );

    QObject::connect( p->thumbnail_bar_act, SIGNAL(triggered(bool))    , p->scroll_widget, SLOT(setShown(bool)) );
    QObject::connect( p->fullscreen_act   , SIGNAL(triggered())        , this            , SLOT(fullScreen())   );
    QObject::connect( p->draw_widget      , SIGNAL(fullscreenRequest()), this            , SLOT(fullScreen())   );

    QObject::connect( p->zoom_slider  , SIGNAL(valueChanged(int)) , p->draw_widget  , SLOT(zoom(int))         );
    QObject::connect( p->fit_act      , SIGNAL(triggered())       , p->draw_widget  , SLOT(setFitSize())      );
    QObject::connect( p->original_act , SIGNAL(triggered())       , p->draw_widget  , SLOT(setOriginalSize()) );

    QObject::connect( p->draw_widget , SIGNAL(zoomed(int))        , p->zoom_slider  , SLOT(setValue(int))     );
    QObject::connect( p->draw_widget , SIGNAL(fitSize(bool))      , p->fit_act      , SLOT(setChecked(bool))  );
    QObject::connect( p->draw_widget , SIGNAL(originalSize(bool)) , p->original_act , SLOT(setChecked(bool))  );

    QObject::connect( p->next_photo_action , SIGNAL(triggered()) , p->thumbnail_bar , SLOT(next()) );
    QObject::connect( p->prev_photo_action , SIGNAL(triggered()) , p->thumbnail_bar , SLOT(previous()));
}

void PhotoViewer::showPicture( const QString & path )
{
    p->draw_widget->draw( path );


// Scrolling to places of current item ======================================//
    QRect item_rect = p->thumbnail_bar->currentItemRect();
    double item_ratio = (double)item_rect.x() / p->thumbnail_bar->width();
    int    max_value  = p->scroll_widget->maximumHorizontal();

    p->scroll_widget->setHorizontalValue( item_ratio*max_value + item_rect.width()*item_ratio );
// END =============//

}

void PhotoViewer::fullScreen()
{
    if( p->full_screen_stt )
    {/*
        setAutoFillBackground( false );
        p->thumbnail_bar->setAutoFillBackground( false );
*/
        setWindowFlags( Qt::Widget );
        showNormal();
        p->full_screen_stt = false;
    }
    else
    {
        setAutoFillBackground( true );
        p->thumbnail_bar->setAutoFillBackground( true );

        setWindowFlags( Qt::Window );
        showFullScreen();
        p->full_screen_stt = true;
    }
}

void PhotoViewer::dropEvent( QDropEvent *event )
{
    QList<QUrl> urls_list = event->mimeData()->urls();
    if( urls_list.isEmpty() )
        return ;

    p->thumbnail_bar->clear();

    QStringList list;
    for( int i=0 ; i<urls_list.count() ; i++ )
    {
        QString path = urls_list.at(i).path();

#ifdef Q_OS_WIN32
        if( path[0] == '/' )
            path.remove( 0 , 1 );
#endif

        if( QFileInfo( path ).isDir() )
            list << path;
        else
            p->thumbnail_bar->addFile( path );
    }

    p->thumbnail_bar->addPathList( list );

    if( p->thumbnail_bar->count() > 0 )
        p->thumbnail_bar->setCurrentIndex( 0 );

    event->acceptProposedAction();
}

void PhotoViewer::dragEnterEvent( QDragEnterEvent *event )
{
    QList<QUrl> urls_list = event->mimeData()->urls();
    if( urls_list.isEmpty() )
        return ;

    event->acceptProposedAction();
}

void PhotoViewer::addRuntimeArgs( const QVariantList & args )
{
    QString single_image_exist;

    for( int i=0 ; i<args.count() ; i++ )
    {
        QVariant var = args.at(i);
        if( var.type() != QVariant::String )
            return;

        QString str = var.toString();

        if( QFileInfo( str ).isDir() )
            p->thumbnail_bar->addPath( str );
        else
        {
            p->thumbnail_bar->addPath( QFileInfo(str).dir().path() );
            single_image_exist = str;
        }
    }

    if( !single_image_exist.isEmpty() )
        p->thumbnail_bar->setCurrentIndex( single_image_exist );
    else if( p->thumbnail_bar->count() > 0 )
        p->thumbnail_bar->setCurrentIndex( 0 );
}

PhotoViewer::~PhotoViewer()
{
    delete p;
}
