#define PROGRESSBAR_MARGIN 5
#define PROGRESSBAR_HEIGHT 20

#include "lyrics.h"

#include <QProgressBar>
#include <QResizeEvent>

#include <tag.h>
#include <fileref.h>

class LyricsPrivate
{
public:
    LyricDownloader *downloader;
    QProgressBar *progressbar;
};

Lyrics::Lyrics(QWidget *parent) :
    QTextBrowser(parent)
{
    p = new LyricsPrivate;
    p->progressbar = new QProgressBar( this );
        p->progressbar->hide();

    p->downloader = new LyricDownloader( this );

    qRegisterMetaType<LyricDownloader::Error>( "LyricDownloader::Error" );

    connect( p->downloader , SIGNAL(Downloaded(QString))                  , SLOT(setText(QString))                      );
    connect( p->downloader , SIGNAL(StateChanged(LyricDownloader::State)) , SLOT(statusChanged(LyricDownloader::State)) );
    connect( p->downloader , SIGNAL(Failed(LyricDownloader::Error))       , SLOT(error(LyricDownloader::Error))         , Qt::QueuedConnection );
}

void Lyrics::init( const Phonon::AudioOutput *audio , const Phonon::MediaObject *media , const Phonon::Path *path , const QStringList *playlist )
{
    Q_UNUSED( audio )
    Q_UNUSED( path )
    Q_UNUSED( playlist )

    playing( media->currentSource().fileName() );
}

void Lyrics::playing( const QString & file )
{
    if( file.isEmpty() )
        return;

    TagLib::FileRef file_ref( file.toUtf8().constData() );
    TagLib::Tag    *tag = file_ref.tag();
    if( !tag )
        return ;

    p->downloader->Download( QString::fromUtf8(tag->artist().to8Bit(true).c_str()) , QString::fromUtf8(tag->title().to8Bit(true).c_str()) );
    clear();
}

void Lyrics::statusChanged( LyricDownloader::State state )
{
    switch( static_cast<int>(state) )
    {
    case LyricDownloader::Idle:
        p->progressbar->setValue( 100 );
        p->progressbar->hide();
        p->progressbar->setFormat( tr("Idle") );
        break;

    case LyricDownloader::ResolvingArtistName:
        p->progressbar->show();
        p->progressbar->setValue( 15 );
        p->progressbar->setFormat( tr("State: Resolving Artist Name") );
        break;

    case LyricDownloader::ResolvingTrackName:
        p->progressbar->show();
        p->progressbar->setValue( 30 );
        p->progressbar->setFormat( tr("State: Resolving Track Name") );
        break;

    case LyricDownloader::DownloadingLyrics:
        p->progressbar->show();
        p->progressbar->setValue( 60 );
        p->progressbar->setFormat( tr("State: Downloading Lyrics") );
        break;
    }
}

void Lyrics::error( LyricDownloader::Error error )
{
    switch( static_cast<int>(error) )
    {
    case LyricDownloader::ArtistNotFound:
        p->progressbar->show();
        p->progressbar->setFormat( tr("Error: Artist Not Found") );
        break;

    case LyricDownloader::TrackNotFound:
        p->progressbar->show();
        p->progressbar->setFormat( tr("Error: Track Not Found") );
        break;

    case LyricDownloader::NetworkError:
        p->progressbar->show();
        p->progressbar->setFormat( tr("Error: Network Error") );
        break;
    }

    setText( p->progressbar->format() );
}

void Lyrics::resizeEvent( QResizeEvent *event )
{
    QTextBrowser::resizeEvent( event );
    p->progressbar->move( PROGRESSBAR_MARGIN , event->size().height() - PROGRESSBAR_MARGIN - PROGRESSBAR_HEIGHT );
    p->progressbar->resize( event->size().width() - PROGRESSBAR_MARGIN*2 , PROGRESSBAR_HEIGHT );
}

Lyrics::~Lyrics()
{
    delete p;
}
