#include "nowplaying.h"

#include <QPainter>
#include <QPaintEvent>
#include <QPainterPath>
#include <QLineEdit>
#include <QFormLayout>
#include <QVBoxLayout>

#include <SPictureWidget>
#include <SColor>

#include <tag.h>
#include <fileref.h>

class NowPlayingPrivate
{
public:
    QHBoxLayout *layout;
    QFormLayout *details_layout;

    QLineEdit *artist_line;
    QLineEdit *album_line;
    QLineEdit *track_line;

    SPictureWidget *picture;
};

NowPlaying::NowPlaying(QWidget *parent) :
    QWidget(parent)
{
    p = new NowPlayingPrivate;

    p->artist_line = new QLineEdit();
        p->artist_line->setReadOnly( true );
        p->artist_line->setStyleSheet( "QLineEdit{border-style: solid;background-color: transparent}" );

    p->album_line = new QLineEdit();
        p->album_line->setReadOnly( true );
        p->album_line->setStyleSheet( "QLineEdit{border-style: solid;background-color: transparent}" );

    p->track_line = new QLineEdit();
        p->track_line->setReadOnly( true );
        p->track_line->setStyleSheet( "QLineEdit{border-style: solid;background-color: transparent}" );

    p->picture = new SPictureWidget();
        p->picture->drawFile( ":/TagargPlayer/Pics/pic.png" );

    p->details_layout = new QFormLayout();
        p->details_layout->addRow( "<b>" + tr("Artist") + "</b>" , p->artist_line );
        p->details_layout->addRow( "<b>" + tr("Album") + "</b>" , p->album_line );
        p->details_layout->addRow( "<b>" + tr("Track") + "</b>" , p->track_line );
        p->details_layout->setSpacing( 23 );
        p->details_layout->setContentsMargins( 4 , 23 , 4 , 4 );

    p->layout = new QHBoxLayout( this );
        p->layout->addWidget( p->picture );
        p->layout->addLayout( p->details_layout );
}

void NowPlaying::init( const Phonon::AudioOutput *audio , const Phonon::MediaObject *media , const Phonon::Path *path , const QStringList *playlist )
{
    Q_UNUSED( audio )
    Q_UNUSED( path )
    Q_UNUSED( playlist )

    playing( media->currentSource().fileName() );
}

void NowPlaying::playing( const QString & file )
{
    if( file.isEmpty() )
        return;

    TagLib::FileRef file_ref( file.toUtf8().constData() );
    TagLib::Tag    *tag = file_ref.tag();
    if( !tag )
        return ;

    p->artist_line->setText( QString::fromUtf8(tag->artist().to8Bit(true).c_str()) );
    p->album_line->setText(  QString::fromUtf8(tag->album().to8Bit(true).c_str()) );
    p->track_line->setText(  QString::fromUtf8(tag->title().to8Bit(true).c_str()) );
}

void NowPlaying::paintEvent( QPaintEvent * )
{
    QRect rct;
        rct.setX( 2 );
        rct.setY( 2 );
        rct.setWidth( rect().width()-4 );
        rct.setHeight( rect().height()-4 );

    SColor back_color( palette().highlight().color() );
        back_color.setAlpha( 37 );

    SColor border_color( palette().highlight().color() );
        border_color.setAlpha( 137 );

    QPainterPath path;
        path.addRoundRect( rct , 5 );

    QPainter painter( this );
        painter.setPen( border_color );
        painter.setRenderHint( QPainter::Antialiasing );
        painter.fillPath( path , back_color );
        painter.drawPath( path );
}

void NowPlaying::resizeEvent( QResizeEvent *event )
{
    p->picture->setFixedSize( event->size().height()-20 , event->size().height()-20 );
}

NowPlaying::~NowPlaying()
{
    delete p;
}
