/*
 *      EasyPMP [CUI] option parser.
 *
 *      Copyright (c) 2005-2007 Naoaki Okazaki
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA, or visit
 * http://www.gnu.org/copyleft/gpl.html .
 *
 */

/* $Id: option.c 336 2007-02-11 16:59:22Z nyaochi $ */

#ifdef	HAVE_CONFIG_H
#include <config.h>
#endif/*HAVE_CONFIG_H*/

#include <os.h>
#include <stdio.h>
#include <stdlib.h>
#ifdef	HAVE_STRING_H
#include <string.h>
#endif/*HAVE_STRING_H*/
#ifdef	_MSC_VER
#include <direct.h>		/* getcwd() */
#endif/*_MSC_VER*/

#include <pmplib/ucs2char.h>
#include <pmplib/filepath.h>
#include <gmi.h>
#include <pmplib/pmp.h>

#ifdef	HAVE_GETOPT_H
#include <getopt.h>
#else
#include "getopt.h"
#endif

#include <easypmp.h>
#include "option.h"

void option_usage(FILE *fp, const char *argv0)
{
	fprintf(fp, "USAGE: %s [OPTIONS] [LOCATION]\n", argv0);
#ifdef	_WIN32
	fprintf(fp, "  LOCATION  Absolute path to the root directory of a player (e.g., D:\\).\n");
#else
	fprintf(fp, "  LOCATION  Mount location of a player (e.g., /mnt/usbdisk/).\n");
#endif
	fprintf(fp, "            If LOCATION is omitted, assume the current directory.\n");
#ifdef	_WIN32
	fprintf(fp, "            If \"-\" is specified, assume the root directory of this executable.\n");
#endif
	fprintf(fp, "\n");

	fprintf(fp, "Media database options:\n");
	fprintf(fp, "  -c, --create          Construct a media database\n");
	fprintf(fp, "  -u, --update          Update the existing media database incrementally\n");
	/*fprintf(fp, "  -z, --source=SOURCE   Specify a source for obtaining media information\n");*/
	fprintf(fp, "  -R, --repr            Output a database structure in plain text\n");
	fprintf(fp, "  -L, --repr-level=VAL  Specify representation level for -R option\n");
	fprintf(fp, "  -t, --strip-words=VAL Specify a list (CSV) of words to strip from artist names\n");
	fprintf(fp, "\n");
	fprintf(fp, "Playlist options:\n");
	fprintf(fp, "  -p, --playlist        Convert playlist files\n");
	fprintf(fp, "  -P, --playlist-source=VALUE   Change the path to the source playlists\n");
	fprintf(fp, "  -r, --reconvert       Discard the existing playlist files and reconvert\n");
	fprintf(fp, "  -f, --find-missing    Correct playlists with corrupt references to media files\n");
	fprintf(fp, "  -i, --ignore-missing  Continue a conversion even if a media file is missing\n");
	fprintf(fp, "\n");
	fprintf(fp, "Player options:\n");
	fprintf(fp, "  -l, --list-device     Show the list of supported devices and exit\n");
	fprintf(fp, "  -d, --device          Specify a device identifier for the player\n");
	fprintf(fp, "  -s, --set=NAME:VALUE  Set a variable NAME to VALUE\n");
	fprintf(fp, "\n");
#ifndef	_WIN32
	fprintf(fp, "Language-specific options:\n");
	fprintf(fp, "  -e, --encoding=SE     Specify character encoding used by the operating system\n");
	fprintf(fp, "  -w, --tagencoding=TE  Specify character encoding for parsing non-Unicode tags\n");
	fprintf(fp, "\n");
#endif/*_WIN32*/
	fprintf(fp, "Miscellaneous options:\n");
	fprintf(fp, "  -v, --version         Show version number and exit\n");
	fprintf(fp, "  -h, --help            Show this help message and exit\n");
	fprintf(fp, "  -x, --help-variable   Show the list of variables and exit\n");
}

void option_usage_variable(FILE* fp)
{
	fprintf(fp, "VARIABLE USAGE:\n");
	fprintf(fp, "  Specify the VALUE of a variable NAME with -s option, i.e., -s NAME:VALUE\n");
	fprintf(fp, "  Note that a pathname ending with \"%c%c\" implies inclusion of its subdirs\n", PATHCHAR, PATHCHAR);
	fprintf(fp, "\n");
	fprintf(fp, "Pathname variables:\n");
	fprintf(fp, "  pmp_music     Relative path to music directory\n");
	fprintf(fp, "  pmp_playlist  Relative path to playlist directory\n");
}

void option_init(option_t* opt)
{
	memset(opt, 0, sizeof(*opt));

	// Set default values here.
	opt->media_info_source = GMIF_TAG;
	opt->system_encoding = strdup(ucs2getenc());
	opt->music_encoding = strdup("ISO-8859-1");
}

void option_finish(option_t* opt)
{
	int i;

	for (i = 0;i < opt->num_strip_words;++i) {
		ucs2free(opt->strip_words[i]);
	}
	ucs2free(opt->strip_words);
	ucs2free(opt->path_to_pmp_music);
	ucs2free(opt->path_to_pmp_playlist);
	ucs2free(opt->path_to_playlist_source);
	free(opt->system_encoding);
	free(opt->music_encoding);
	memset(opt, 0, sizeof(*opt));
}

int option_parse(option_t* opt, int argc, char *argv[], FILE *fpe)
{
	int used_args = 0;

	/*
	 * Parse the command-line arguments.
	 *  We must not use the charaset conversion routine from this function
	 *  since the character encoding has not been decided at this moment.
	 */
	for (;;) {
		int this_option_optind = optind ? optind : 1;
		int option_index = 0;
		static const struct option long_options[] = {
			{"create",		no_argument,		0, 'c'},
			{"update",		no_argument,		0, 'u'},
			{"source",		required_argument,	0, 'z'},
			{"repr",		no_argument,		0, 'R'},
			{"repr-level",		required_argument,	0, 'L'},
			{"strip-words",		required_argument,	0, 't'},
			{"playlist",		no_argument,		0, 'p'},
			{"playlist-source",	no_argument,		0, 'P'},
			{"reconvert",		no_argument,		0, 'r'},
			{"find-missing",	no_argument,		0, 'f'},
			{"ignore-missing",	no_argument,		0, 'i'},
			{"list-device",		no_argument,		0, 'l'},
			{"device",		required_argument,	0, 'd'},
			{"set",			required_argument,	0, 's'},
		#ifndef	_WIN32
			{"encoding",		required_argument,	0, 'e'},
			{"tagencoding",		required_argument,	0, 'w'},
		#endif/*_WIN32*/
			{"version",		no_argument,		0, 'v'},
			{"help",		no_argument,		0, 'h'},
			{"help-variable",	no_argument,		0, 'x'},
			{NULL, 0, 0, 0}
		};
#ifndef	_WIN32
		int c = getopt_long(argc, argv, "cuz:RL:t:pP:rfild:xs:e:w:vh", long_options, &option_index);
#else
		int c = getopt_long(argc, argv, "cuz:RL:t:pP:rfild:xs:vh", long_options, &option_index);
#endif/*_WIN32*/
		if (c == -1) {
			break;
		}

		switch (c) {
		case 'c':
			opt->verb |= MODE_DATABASE;
			break;
		case 'u':
			opt->verb |= (MODE_DATABASE | MODE_DATABASE_UPDATE);
			break;
		/*
		case 'z':
			if		(strcmp(optarg, "m") == 0)		opt->media_info_source = GMIF_TAG;
			else if	(strcmp(optarg, "p") == 0)		opt->media_info_source = GMIF_TAG | GMIF_ARTIST;
			else if	(strcmp(optarg, "pa") == 0)		opt->media_info_source = GMIF_TAG | GMIF_ARTIST | GMIF_ALBUM;
			else if	(strcmp(optarg, "pat") == 0)	opt->media_info_source = GMIF_TAG | GMIF_TITLE | GMIF_ARTIST | GMIF_ALBUM;
			else if	(strcmp(optarg, "pant") == 0)	opt->media_info_source = GMIF_TAG | GMIF_TITLE | GMIF_ARTIST | GMIF_ALBUM | GMIF_NUMBER;
			else if	(strcmp(optarg, "gp") == 0)		opt->media_info_source = GMIF_GENRE | GMIF_TAG | GMIF_ARTIST;
			else if	(strcmp(optarg, "gpa") == 0)	opt->media_info_source = GMIF_GENRE | GMIF_TAG | GMIF_ARTIST | GMIF_ALBUM;
			else if	(strcmp(optarg, "gpat") == 0)	opt->media_info_source = GMIF_GENRE | GMIF_TAG | GMIF_TITLE | GMIF_ARTIST | GMIF_ALBUM;
			else if	(strcmp(optarg, "gpant") == 0)	opt->media_info_source = GMIF_GENRE | GMIF_TAG | GMIF_TITLE | GMIF_ARTIST | GMIF_ALBUM | GMIF_NUMBER;
			else {
				fprintf(fpe, "ERROR: Unrecognized method for obtaining media information!\n");
				return -1;
			}
			break;
		*/
		case 'R':
			opt->verb |= MODE_DATABASE_REPR;
			break;
		case 'L':
			opt->repr_level = atoi(optarg);
			break;
		case 't':
			free(opt->mbs_strip_words);
			opt->mbs_strip_words = strdup(optarg);
			break;

		case 'p':
			opt->verb |= MODE_PLAYLIST;
			break;
		case 'P':
			ucs2free(opt->path_to_playlist_source);
			opt->path_to_playlist_source = mbsdupucs2(optarg);
			break;
		case 'r':
			opt->verb |= MODE_PLAYLIST_RECONVERT;
			break;
		case 'f':
			opt->verb |= MODE_PLAYLIST_FIND;
			break;
		case 'i':
			opt->verb |= MODE_PLAYLIST_SKIP;
			break;

		case 'l':
			opt->verb |= MODE_LIST_DEVICES;
			break;
		case 'd':
			strcpy(opt->model, optarg);
			break;
		case 's':
			if (strncmp("pmp_music:", optarg, 10) == 0) {
				ucs2free(opt->path_to_pmp_music);
				opt->path_to_pmp_music = mbsdupucs2(optarg+10);
			} else if (strncmp("pmp_playlist:", optarg, 13) == 0) {
				ucs2free(opt->path_to_pmp_playlist);
				opt->path_to_pmp_playlist = mbsdupucs2(optarg+13);
			} else {
				fprintf(fpe, "Unrecognized variable: %s\n", optarg);
				return -1;
				break;
			}
			break;
#ifndef	_WIN32
		case 'e':
			free(opt->system_encoding);
			opt->system_encoding = strdup(optarg);
			break;
		case 'w':
			free(opt->music_encoding);
			opt->music_encoding = strdup(optarg);
			break;
#endif/*_WIN32*/
		case 'v':
			opt->verb |= MODE_VERSION;
			break;
		case 'h':
			opt->verb |= MODE_HELP;
			break;
		case 'x':
			opt->verb |= MODE_HELP_VARIABLE;
			break;
		case '?':
			return -1;
			break;
		}
	}

	return optind;
}
