/*
 *      PMP library implementation for iriver H100/300.
 *
 *      Copyright (c) 2005-2007 Naoaki Okazaki
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 *
 */

/* $Id: pmp_irivnavi.c 328 2007-02-10 17:50:11Z nyaochi $ */

#ifdef	HAVE_CONFIG_H
#include <config.h>
#endif/*HAVE_CONFIG_H*/

#include <os.h>
#ifdef	HAVE_STRING_H
#include <string.h>
#endif/*HAVE_STRING_H*/
#ifdef	HAVE_STDLIB_H
#include <stdlib.h>
#endif/*HAVE_STDLIB_H*/
#include <stddef.h>
#include <pmplib/ucs2char.h>
#include <pmplib/filepath.h>
#include <pmplib/pmp.h>

#include "irivnavi.h"

#ifdef	PMP_IRIVNAVI_EXPORTS
#define	PMPIRIVNAVIAPI	__declspec(dllexport)
#else
#define	PMPIRIVNAVIAPI
#endif

enum {
	MODEL_NONE = 0,
	MODEL_IRIVER_H100,
	MODEL_IRIVER_H300,
};

static const ucs2char_t ucs2cs_music_path[] =
	{'M','u','s','i','c',0};
static const ucs2char_t ucs2cs_playlist_path[] =
	{'P','l','a','y','l','i','s','t','s',0};
static const ucs2char_t ucs2cs_playlist_ext[] =
	{'.','m','3','u',0};
static const ucs2char_t ucs2cs_irivnavi_idb[] =
	{'i','R','i','v','N','a','v','i','.','i','D','B',0};
static const ucs2char_t ucs2cs_h100_sys[] =
	{'H','1','0','0','.','s','y','s',0};
static const ucs2char_t ucs2cs_h300_sys[] =
	{'H','3','0','0','.','s','y','s',0};
static const ucs2char_t ucs2cs_system_h100_sys[] =
	{'S','y','s','t','e','m',PATHCHAR,'H','1','0','0','.','s','y','s',0};
static const ucs2char_t ucs2cs_system_h300_sys[] =
	{'S','y','s','t','e','m',PATHCHAR,'H','3','0','0','.','s','y','s',0};
static const uint32_t g_codecs[] = 
	{PMPCODEC_MPEGLAYER3, PMPCODEC_VORBIS, PMPCODEC_WMA, PMPCODEC_WAV};
static const char *g_music_exts[] = 
	{".mp3", ".ogg", ".wma", ".wav"};

#define	MODELID_H100	"iriver_h100"
#define	MODELID_H300	"iriver_h300"

typedef struct {
	ucs2char_t	path_to_root[MAX_PATH];
	ucs2char_t	path_to_music[MAX_PATH];
	ucs2char_t	path_to_playlist[MAX_PATH];
	ucs2char_t	db_filename[MAX_PATH];
	ucs2char_t	playlist_ext[MAX_PATH];
} irivnavi_environment_t;

typedef struct {
	irivnavi_environment_t	env;
} pmp_internal_t;

typedef struct {
	irivnavi_t db;
	pmp_music_record_t* records;
	int num_records;
	pmp_playlist_t* playlists;
	int num_playlists;
} pmp_music_internal_t;

static uint32_t pmp_add_ref(pmp_t* pmp);
static uint32_t pmp_release(pmp_t* pmp);
static result_t pmp_open(pmp_t* pmp, uint32_t flag);
static result_t pmp_close(pmp_t* pmp);
static result_t pmp_create_instance_music(pmp_t* pmp, pmp_music_t** ptr_pmpdb);

static uint32_t pmpmusic_release(pmp_music_t* music);
static uint32_t pmpmusic_open(pmp_music_t* music);
static result_t pmpmusic_close(pmp_music_t* music);
static result_t pmpmusic_set_records(pmp_music_t* music, const pmp_music_record_t* records, uint32_t num_records);
static result_t pmpmusic_get_records(pmp_music_t* music, pmp_music_record_t* records, uint32_t* num_records);
static result_t pmpmusic_dump(pmp_music_t* music, FILE *fp, int level);
static result_t pmpmusic_set_playlists(pmp_music_t* music, const pmp_playlist_t* playlists, uint32_t num_playlists);

static int exists_sysfile(const ucs2char_t* path_to_device, const ucs2char_t* sysfilename)
{
	ucs2char_t filename[MAX_PATH];

	ucs2cpy(filename, path_to_device);
	filepath_addslash(filename);
	ucs2cat(filename, sysfilename);
	return filepath_file_exists(filename);
}

static void free_device_info(pmp_device_information_t* info)
{
	uint32_t i;
	for (i = 0;i < info->num_audio_extensions;++i) {
		ucs2free(info->audio_extensions[i]);
	}
	ucs2free(info->audio_codecs);
	ucs2free(info->audio_extensions);
	memset(info, 0, sizeof(*info));
}

PMPIRIVNAVIAPI result_t pmp_enumerate_devid(pmplib_enumerate_devid_callback_t callback, void *instance)
{
	callback(instance, MODELID_H100);
	callback(instance, MODELID_H300);
	return 0;
}

PMPIRIVNAVIAPI result_t pmp_create(pmp_t** ptr_pmp, const ucs2char_t* path_to_device, const char *id)
{
	int model = MODEL_NONE;
	result_t ret = 0;
	size_t i = 0, n = 0;
	pmp_t* pmp = NULL;
	pmp_internal_t* pmpi = NULL;
	pmp_device_information_t* info = NULL;
	pmp_device_description_t* decl = NULL;

	*ptr_pmp = 0;

	// Check the model of the player.
	if (id && *id) {
		// A device identifier is specified.
		if (strcmp(id, MODELID_H100) == 0) {
			model = MODEL_IRIVER_H100;
		}
		if (strcmp(id, MODELID_H300) == 0) {
			model = MODEL_IRIVER_H300;
		}
	} else {
		// Try automatic device recognition.
		if (exists_sysfile(path_to_device, ucs2cs_h100_sys)) {
			model = MODEL_IRIVER_H100;
		}
		if (exists_sysfile(path_to_device, ucs2cs_system_h100_sys)) {
			model = MODEL_IRIVER_H100;
		}
		if (exists_sysfile(path_to_device, ucs2cs_h300_sys)) {
			model = MODEL_IRIVER_H300;
		}
		if (exists_sysfile(path_to_device, ucs2cs_system_h300_sys)) {
			model = MODEL_IRIVER_H300;
		}
	}

	// Fail if this library does not support the player.
	if (model == MODEL_NONE) {
		return PMPERR_DEVICENOTFOUND;
	}

	// Create a PMP instance.
	pmp = (pmp_t*)calloc(1, sizeof(pmp_t));
	if (!pmp) {
		return PMPERR_INSUFFICIENTMEMORY;
	}

	// Set member methods.
	pmp->add_ref = pmp_add_ref;
	pmp->release = pmp_release;
	pmp->open = pmp_open;
	pmp->close = pmp_close;
	pmp->add_ref(pmp);

	// Allocate the internal variables.
	pmpi = (pmp_internal_t*)calloc(1, sizeof(pmp_internal_t));
	if (!pmpi) {
		pmp_release(pmp);
		return PMPERR_INSUFFICIENTMEMORY;
	}
	pmp->instance = pmpi;

	// Initialize the internal variables.
	info = &pmp->info;
	decl = (pmp_device_description_t*)&info->decl;

	ucs2cpy(info->path_to_root, path_to_device);
	ucs2cpy(info->path_to_music, ucs2cs_music_path);	// Force "\Music" directory for now.
	ucs2cpy(info->path_to_playlist, ucs2cs_playlist_path);	// Force "\Playlist" directory for now.

	// Set model information.
	switch (model) {
	case MODEL_IRIVER_H100:
		strcpy(decl->id, MODELID_H100);
		strcpy(decl->manufacturer, "iriver");
		strcpy(decl->name, "H100 series");
		break;
	case MODEL_IRIVER_H300:
		strcpy(decl->id, MODELID_H300);
		strcpy(decl->manufacturer, "iriver");
		strcpy(decl->name, "H300 series");
		break;
	}
	strcpy(decl->mode, "UM");
	strcpy(decl->language, PMP_DECLUNAVAIL);
	strcpy(decl->version, PMP_DECLUNAVAIL);
	strcpy(decl->max_version, PMP_DECLUNAVAIL);
	strcpy(decl->min_version, PMP_DECLUNAVAIL);

	info->music_flag = PMPMF_SUPPORT | PMPMF_RECURSIVE;
	info->playlist_flag = PMPPF_SUPPORT;

	// Audio codecs.
	info->num_audio_codecs = sizeof(g_codecs) / sizeof(g_codecs[0]);
	info->audio_codecs = (uint32_t*)ucs2malloc(sizeof(uint32_t) * info->num_audio_codecs);
	memcpy(info->audio_codecs, g_codecs, sizeof(g_codecs));

	// Obtain the number of extensions separated by '\0' characters.
	info->num_audio_extensions = sizeof(g_music_exts) / sizeof(g_music_exts[0]);
	info->audio_extensions = (ucs2char_t**)ucs2malloc(sizeof(ucs2char_t*) * info->num_audio_extensions);
	for (i = 0;i < info->num_audio_extensions;++i) {
		info->audio_extensions[i] = mbsdupucs2(g_music_exts[i]);
	}

	// Create music instance.
	ret = pmp_create_instance_music(pmp, &pmp->music);
	if (ret != 0) {
		pmp_release(pmp);
		return ret;
	}

	// Return this instance to the caller.
	*ptr_pmp = pmp;
	return 0;
}

static uint32_t pmp_add_ref(pmp_t* pmp)
{
	return pmplib_interlocked_increment(&pmp->ref_count);
}

static uint32_t pmp_release(pmp_t* pmp)
{
	uint32_t count = pmplib_interlocked_decrement(&pmp->ref_count);
	if (count == 0) {
		pmpmusic_release(pmp->music);
		free_device_info(&pmp->info);
		free(pmp->instance);
		free(pmp);
	}
	return count;
}

static result_t pmp_open(pmp_t* pmp, uint32_t flag)
{
	result_t ret = 0;

	// Set the open flag.
	pmp->flag = flag;
	
	// Open the music database.
	ret = pmpmusic_open(pmp->music);
	if (ret) {
		return ret;
	}

	return 0;
}

static result_t pmp_close(pmp_t* pmp)
{
	result_t ret = 0;

	// Close the music database.
	ret = pmpmusic_close(pmp->music);
	if (ret) {
		return ret;
	}

	return 0;
}

static result_t pmp_create_instance_music(pmp_t* pmp, pmp_music_t** ptr_music)
{
	pmp_music_t* music = NULL;
	pmp_internal_t* pmpi = (pmp_internal_t*)pmp->instance;
	pmp_music_internal_t* pmpmi = NULL;

	*ptr_music = 0;

	music = calloc(1, sizeof(pmp_music_t));
	if (!music) {
		return PMPERR_INSUFFICIENTMEMORY;
	}

	pmpmi = calloc(1, sizeof(pmp_music_internal_t));
	if (!pmpmi) {
		free(music);
		return PMPERR_INSUFFICIENTMEMORY;
	}

	music->set_records = pmpmusic_set_records;
	music->get_records = pmpmusic_get_records;
	music->dump = pmpmusic_dump;
	music->set_playlists = pmpmusic_set_playlists;
	music->pmp = pmp;
	music->instance = pmpmi;

	*ptr_music = music;
	return 0;
}

static uint32_t pmpmusic_release(pmp_music_t* music)
{
	pmp_music_internal_t* pmpmi = (pmp_music_internal_t*)music->instance;
	pmplib_records_finish(pmpmi->records, pmpmi->num_records);
	free(pmpmi->playlists);
	free(pmpmi);
	free(music);
	return 0;
}

uint32_t pmpmusic_open(pmp_music_t* music)
{
	pmp_music_internal_t* pmpmi = (pmp_music_internal_t*)music->instance;
	result_t ret = 0;
	FILE *fp = NULL;
	ucs2char_t filename[MAX_PATH];
	long size = 0;
	int i, j;
	uint8_t *buffer = NULL;
	static const ucs2char_t ucs2cs_mp3[] = {'.','m','p','3',0};
	static const ucs2char_t ucs2cs_ogg[] = {'.','o','g','g',0};
	static const ucs2char_t ucs2cs_wma[] = {'.','w','m','a',0};
	static const ucs2char_t ucs2cs_wav[] = {'.','w','a','v',0};

	irivnavi_init(&pmpmi->db);

	if (music->pmp->flag & PMPOF_MUSIC_DB_READ) {
		// Open a database file in the root directory.
		ucs2cpy(filename, music->pmp->info.path_to_root);
		filepath_addslash(filename);
		ucs2cat(filename, ucs2cs_irivnavi_idb);
		fp = ucs2fopen(filename, "rb");
		if (!fp) {
			return PMPERR_OPENFORREAD;
		}

		// Obtain the stream size.
		if (fseek(fp, 0, SEEK_END) != 0) {
			return PMPERR_OPENFORREAD;
		}
		if ((size = ftell(fp)) == -1) {
			return PMPERR_OPENFORREAD;
		}
		if (fseek(fp, 0, SEEK_SET) != 0) {
			return PMPERR_OPENFORREAD;
		}

		// Allocate a buffer that stores the whole data.
		buffer = (uint8_t*)malloc(size);
		if (!buffer) {
			ret = PMPERR_INSUFFICIENTMEMORY;
			goto error_exit;
		}

		// Read at one time.
		fread(buffer, 1, size, fp);

		// Read the data from the buffer.
		if (irivnavi_serialize(&pmpmi->db, buffer, 0) == 0) {
			ret = PMPERR_INCONSISTENCY;
			goto error_exit;
		}

		// Free the buffer and file.
		free(buffer);
		fclose(fp);

		// 
		pmpmi->num_records = pmpmi->db.num_records;
		pmpmi->records = (pmp_music_record_t*)ucs2malloc(sizeof(pmp_music_record_t) * pmpmi->num_records);

		for (i = 0, j = 0;i < pmpmi->num_records;++i) {
			size_t length = 0;
			ucs2char_t* tmp = NULL;
			const record_t* src = &pmpmi->db.records[i];
			pmp_music_record_t* dst = &pmpmi->records[j];

			pmplib_record_init(dst);

			tmp = mbsdupucs2(src->filename);
			length  = ucs2len(music->pmp->info.path_to_root);
			length += ucs2len(tmp);
			length += 3;

			dst->filename = ucs2malloc(sizeof(ucs2char_t) * length);
			filepath_combinepath(dst->filename, length, music->pmp->info.path_to_root, tmp);
			dst->title = mbsdupucs2(src->title);
			dst->artist = mbsdupucs2(src->artist);
			dst->album = mbsdupucs2(src->album);
			dst->genre = mbsdupucs2(src->genre);

			// Set codec information according to the file extensions.
			if (filepath_hasext(dst->filename, ucs2cs_mp3)) {
				dst->codec = PMPCODEC_MPEGLAYER3;
			} else if (filepath_hasext(dst->filename, ucs2cs_ogg)) {
				dst->codec = PMPCODEC_VORBIS;
			} else if (filepath_hasext(dst->filename, ucs2cs_wma)) {
				dst->codec = PMPCODEC_WMA;
			} else if (filepath_hasext(dst->filename, ucs2cs_wav)) {
				dst->codec = PMPCODEC_WAV;
			}

			dst->ts_update = src->timestamp;

			ucs2free(tmp);
			++j;
		}
	}
	irivnavi_finish(&pmpmi->db);
	return 0;

error_exit:
	free(buffer);
	if (fp)	fclose(fp);
	irivnavi_finish(&pmpmi->db);
	return ret;
}

result_t pmpmusic_close(pmp_music_t* music)
{
	int i, j;
	pmp_music_internal_t* pmpmi = (pmp_music_internal_t*)music->instance;
	result_t ret = 0;
	FILE *fp = NULL;
	ucs2char_t filename[MAX_PATH];
	uint8_t *buffer = NULL;
	static const ucs2char_t ucs2cs_unknown[] = {'u','n','k','n','o','w','n',0};
	static const ucs2char_t m3u_ext[] = {'.','m','3','u',0};

	irivnavi_init(&pmpmi->db);

	if (music->pmp->flag & PMPOF_MUSIC_DB_WRITE) {
		// Clear the database.
		irivnavi_finish(&pmpmi->db);
		irivnavi_init(&pmpmi->db);

		// Initialize the record array.
		if (irivnavi_init_records(&pmpmi->db, pmpmi->num_records) != 0) {
			ret = PMPERR_INSUFFICIENTMEMORY;
			goto error_exit;
		}
		
		// Convert records from pmp_music_record_t to ip2db_record_t.
		for (i = 0, j = 0;i < pmpmi->num_records;++i) {
			const pmp_music_record_t* src = &pmpmi->records[i];
			record_t* dst = &pmpmi->db.records[j];

			// Set record fields.
			record_init(dst);
			dst->filename = ucs2dupmbs(filepath_changeroot(src->filename, music->pmp->info.path_to_root));
			dst->title = src->title ? ucs2dupmbs(src->title) : ucs2dupmbs(filepath_skippath(src->filename));;
			dst->artist = ucs2dupmbs(src->artist ? src->artist : ucs2cs_unknown);
			dst->album = ucs2dupmbs(src->album ? src->album : ucs2cs_unknown);
			dst->genre = ucs2dupmbs(src->genre ? src->genre : ucs2cs_unknown);
			dst->timestamp = src->ts_update;

			++j;
		}

		pmpmi->db.num_records = pmpmi->num_records;

		// Open a database file in the root directory.
		ucs2cpy(filename, music->pmp->info.path_to_root);
		filepath_addslash(filename);
		ucs2cat(filename, ucs2cs_irivnavi_idb);
		filepath_removefile(filename);
		fp = ucs2fopen(filename, "wb");
		if (!fp) {
			ret = PMPERR_OPENFORREAD;
			goto error_exit;
		}

		irivnavi_update(&pmpmi->db);

		// Allocate a buffer that stores the whole data.
		buffer = (uint8_t*)malloc(pmpmi->db.size);
		if (!buffer) {
			ret = PMPERR_INSUFFICIENTMEMORY;
			goto error_exit;
		}

		// Write the data to the buffer.
		if (irivnavi_serialize(&pmpmi->db, buffer, 1) == 0) {
			ret = PMPERR_INCONSISTENCY;
			goto error_exit;
		}

		// Write at one time.
		fwrite(buffer, 1, pmpmi->db.size, fp);

		free(buffer);
		fclose(fp);
	}

	if (music->pmp->flag & PMPOF_MUSIC_PL_WRITE) {
		int i;

		for (i = 0;i < pmpmi->num_playlists;++i) {
			const ucs2char_t* path_to_root = music->pmp->info.path_to_root;
			const ucs2char_t* path_to_playlist = music->pmp->info.path_to_playlist;
			const pmp_playlist_t* pl = &pmpmi->playlists[i];
			ucs2char_t dst[MAX_PATH];

			filepath_combinepath(dst, MAX_PATH, path_to_root, path_to_playlist);
			filepath_addslash(dst);
			ucs2cat(dst, pl->name);
			ucs2cat(dst, m3u_ext);


			if (playlist_write(
				dst,
				pl->entries,
				pl->num_entries,
				path_to_root
				) != 0) {
				return PMPERR_WRITE;
			}
		}
	}

	irivnavi_finish(&pmpmi->db);
	return 0;

error_exit:
	free(buffer);
	if (fp)	fclose(fp);
	irivnavi_finish(&pmpmi->db);
	return ret;
}

static result_t pmpmusic_set_records(pmp_music_t* music, const pmp_music_record_t* records, uint32_t num_records)
{
	pmp_t* pmp = music->pmp;
	pmp_music_internal_t* pmpmi = (pmp_music_internal_t*)music->instance;

	/* Free records attached to pmpmi. */
	pmplib_records_finish(pmpmi->records, pmpmi->num_records);

	/* Allocate new records. */
	pmpmi->records = (pmp_music_record_t*)ucs2malloc(sizeof(pmp_music_record_t) * num_records);
	pmpmi->num_records = num_records;
	pmplib_records_clone(pmpmi->records, records, num_records);

	return 0;
 }

static result_t pmpmusic_get_records(pmp_music_t* music, pmp_music_record_t* records, uint32_t* num_records)
{
	pmp_t* pmp = music->pmp;
	pmp_music_internal_t* pmpmi = (pmp_music_internal_t*)music->instance;

	if (!records) {
		*num_records = pmpmi->num_records;
		return 0;
	} else if (*num_records == pmpmi->num_records) {
		pmplib_records_clone(records, pmpmi->records, pmpmi->num_records);
		return 0;
	} else {
		return PMPERR_INSUFFICIENTMEMORY;
	}
}

static result_t pmpmusic_dump(pmp_music_t* music, FILE *fp, int level)
{
	pmp_music_internal_t* pmpmi = (pmp_music_internal_t*)music->instance;
	irivnavi_repr(&pmpmi->db, fp, level);
	return 0;
}

static result_t pmpmusic_set_playlists(pmp_music_t* music, const pmp_playlist_t* playlists, uint32_t num_playlists)
{
	pmp_music_internal_t* pmpmi = (pmp_music_internal_t*)music->instance;

	// Free playlists attached to pmpmi.
	pmplib_playlists_finish(pmpmi->playlists, pmpmi->num_playlists);

	// Allocate a new playlists.
	pmpmi->playlists = (pmp_playlist_t*)calloc(num_playlists, sizeof(pmp_playlist_t));
	pmpmi->num_playlists = num_playlists;

	// Copy the playlist array.
	pmplib_playlists_clone(pmpmi->playlists, playlists, num_playlists);

	return 0;
}
