/*
 *      iRiver H10 specific routines and header templates.
 *
 *      Copyright (c) 2005-2007 Naoaki Okazaki
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 *
 */

/* $Id: model_iriver_h10.c 342 2007-02-11 18:11:43Z nyaochi $ */

#ifdef	HAVE_CONFIG_H
#include <config.h>
#endif/*HAVE_CONFIG_H*/
#ifdef	HAVE_STRING_H
#include <string.h>
#endif/*HAVE_STRING_H*/

#include <os.h>
#include <stdio.h>
#include <stdlib.h>
#include <pmplib/ucs2char.h>
#include <pmplib/filepath.h>

#include "serialize.h"
#include "util.h"
#include "pp1db.h"
#include "hdr_template.h"

enum {
	PP1DB_DATFIELD_UNKNOWN2 = 0,// @0:	(INT)	
	PP1DB_DATFIELD_PATHNAME,	// @1:	(STR)	
	PP1DB_DATFIELD_FILENAME,	// @2:	(STR)	
	PP1DB_DATFIELD_FORMAT,		// @3:	(INT)
	PP1DB_DATFIELD_TITLE,		// @4:	(STR)
	PP1DB_DATFIELD_ARTIST,		// @5:	(STR)
	PP1DB_DATFIELD_ALBUM,		// @6:	(STR)
	PP1DB_DATFIELD_GENRE,		// @7:	(STR)
	PP1DB_DATFIELD_RATING,		// @8:	(INT)
	PP1DB_DATFIELD_PLAYCOUNT,	// @9:	(INT)
	PP1DB_DATFIELD_RECENTPLAY,	// @10:	(INT)
	PP1DB_DATFIELD_UNKNOWN3,	// @11:	(INT)
	PP1DB_DATFIELD_TRACKNUMBER,	// @12:	(INT)
	PP1DB_DATFIELD_YEAR,		// @13:	(INT)
	PP1DB_DATFIELD_FILESIZE,	// @14:	(INT)
	PP1DB_DATFIELD_DURATION,	// @15:	(INT)
	PP1DB_DATFIELD_SAMPLERATE,	// @16:	(INT)
	PP1DB_DATFIELD_BITRATE,		// @17:	(INT)
	PP1DB_DATFIELD_UNKNOWN4,	// @18:	(INT)	
	PP1DB_DATFIELD_UNKNOWN5,	// @19:	(STR)
	PP1DB_DATFIELD_UNKNOWN6,	// @20:	(INT)
	PP1DB_DATFIELD_UNKNOWN7,	// @21:	(STR)
};

void iriver_h10_dat_repr(const dat_t* record, FILE *fp)
{
	fprintf(fp, "  inactive: %d\n", record->status);
	fprintf(fp, "  unknown1: %d\n", record->unknown1);
	fprintf(fp, "  unknown2: %d\n", record->fields[PP1DB_DATFIELD_UNKNOWN2].value.dword);
	fprints(fp, "  file_path: %s\n", record->fields[PP1DB_DATFIELD_PATHNAME].value.str);
	fprints(fp, "  file_name: %s\n", record->fields[PP1DB_DATFIELD_FILENAME].value.str);
	fprintf(fp, "  media_type: %d\n", record->fields[PP1DB_DATFIELD_FORMAT].value.dword);
	fprints(fp, "  title: %s\n", record->fields[PP1DB_DATFIELD_TITLE].value.str);
	fprints(fp, "  artist: %s\n", record->fields[PP1DB_DATFIELD_ARTIST].value.str);
	fprints(fp, "  album: %s\n", record->fields[PP1DB_DATFIELD_ALBUM].value.str);
	fprints(fp, "  genre: %s\n", record->fields[PP1DB_DATFIELD_GENRE].value.str);
	fprintf(fp, "  rating: %d\n", record->fields[PP1DB_DATFIELD_RATING].value.dword);
	fprintf(fp, "  play_count: %d\n", record->fields[PP1DB_DATFIELD_PLAYCOUNT].value.dword);
	fprintt(fp, "  recent_play: %s", record->fields[PP1DB_DATFIELD_RECENTPLAY].value.dword);
	fprintf(fp, "  unknown3: %d\n", record->fields[PP1DB_DATFIELD_UNKNOWN3].value.dword);
	fprintf(fp, "  number: %d\n", record->fields[PP1DB_DATFIELD_TRACKNUMBER].value.dword);
	fprintf(fp, "  year: %d\n", record->fields[PP1DB_DATFIELD_YEAR].value.dword);
	fprintf(fp, "  filesize: %d\n", record->fields[PP1DB_DATFIELD_FILESIZE].value.dword);
	fprintf(fp, "  duration: %d\n", record->fields[PP1DB_DATFIELD_DURATION].value.dword);
	fprintf(fp, "  samplerate: %d\n", record->fields[PP1DB_DATFIELD_SAMPLERATE].value.dword);
	fprintf(fp, "  bitrate: %d\n", record->fields[PP1DB_DATFIELD_BITRATE].value.dword);
	fprintf(fp, "  unknown4: %d\n", record->fields[PP1DB_DATFIELD_UNKNOWN4].value.dword);
	fprints(fp, "  unknown5: %s\n", record->fields[PP1DB_DATFIELD_UNKNOWN5].value.str);
	fprintf(fp, "  unknown6: %d\n", record->fields[PP1DB_DATFIELD_UNKNOWN6].value.dword);
	fprints(fp, "  unknown7: %s\n", record->fields[PP1DB_DATFIELD_UNKNOWN7].value.str);
}

int iriver_h10_dat_set(dat_t* dst, const pmp_music_record_t* src, const ucs2char_t* path_to_root)
{
	static const ucs2char_t ucs2cs_unknown[] = {0};
	static const ucs2char_t ucs2cs_empty[] = {0};

	// Set fields.
	dst->status = 0;
	dst->unknown1 = 0;
	dst->fields[PP1DB_DATFIELD_PATHNAME].value.str = ucs2dup(filepath_skiproot(src->filename, path_to_root));
	filepath_remove_filespec(dst->fields[PP1DB_DATFIELD_PATHNAME].value.str);
	filepath_addslash(dst->fields[PP1DB_DATFIELD_PATHNAME].value.str);
	filepath_encode(dst->fields[PP1DB_DATFIELD_PATHNAME].value.str);
	dst->fields[PP1DB_DATFIELD_FILENAME].value.str = ucs2dup(filepath_skippath(src->filename));
	dst->fields[PP1DB_DATFIELD_FORMAT].value.dword = 0;
	dst->fields[PP1DB_DATFIELD_TITLE].value.str = ucs2dup(src->title ? src->title : dst->fields[PP1DB_DATFIELD_FILENAME].value.str);
	dst->fields[PP1DB_DATFIELD_ARTIST].value.str = ucs2dup(src->artist ? src->artist : ucs2cs_unknown);
	dst->fields[PP1DB_DATFIELD_ALBUM].value.str = ucs2dup(src->album ? src->album : ucs2cs_unknown);
	dst->fields[PP1DB_DATFIELD_GENRE].value.str = ucs2dup(src->genre ? src->genre : ucs2cs_unknown);
	dst->fields[PP1DB_DATFIELD_RATING].value.dword = src->rating;
	dst->fields[PP1DB_DATFIELD_PLAYCOUNT].value.dword = src->play_count;
	dst->fields[PP1DB_DATFIELD_RECENTPLAY].value.dword = src->ts_playback;
	dst->fields[PP1DB_DATFIELD_UNKNOWN3].value.dword = 0;
	dst->fields[PP1DB_DATFIELD_TRACKNUMBER].value.dword = src->track_number;
	if (src->date) {
		dst->fields[PP1DB_DATFIELD_YEAR].value.dword = ucs2toi(src->date);
	}
	dst->fields[PP1DB_DATFIELD_FILESIZE].value.dword = src->filesize;
	dst->fields[PP1DB_DATFIELD_DURATION].value.dword = src->duration;
	dst->fields[PP1DB_DATFIELD_SAMPLERATE].value.dword = src->sample_rate;
	dst->fields[PP1DB_DATFIELD_BITRATE].value.dword = src->bitrate;
	dst->fields[PP1DB_DATFIELD_UNKNOWN4].value.dword = 0;
	dst->fields[PP1DB_DATFIELD_UNKNOWN5].value.str = ucs2dup(ucs2cs_empty);
	dst->fields[PP1DB_DATFIELD_UNKNOWN6].value.dword = src->ts_update;
	dst->fields[PP1DB_DATFIELD_UNKNOWN7].value.str = ucs2dup(ucs2cs_empty);
	return 0;
}

int iriver_h10_dat_get(pmp_music_record_t* dst, const dat_t* src, const ucs2char_t* path_to_root)
{
	static const ucs2char_t ucs2cs_mp3[] = {'.','m','p','3',0};
	static const ucs2char_t ucs2cs_wma[] = {'.','w','m','a',0};
	static const ucs2char_t ucs2cs_wav[] = {'.','w','a','v',0};
	ucs2char_t tmp[128];
	size_t length = 0;

	length  = ucs2len(path_to_root);
	length += ucs2len(src->fields[PP1DB_DATFIELD_PATHNAME].value.str);
	length += ucs2len(src->fields[PP1DB_DATFIELD_FILENAME].value.str);
	length += 3;

	dst->filename = (ucs2char_t*)ucs2malloc(sizeof(ucs2char_t) * length);
	if (dst->filename) {
		filepath_combinepath(dst->filename, length, path_to_root, src->fields[PP1DB_DATFIELD_PATHNAME].value.str);
		ucs2cat(dst->filename, src->fields[PP1DB_DATFIELD_FILENAME].value.str);
		filepath_decode(dst->filename);
	}

	dst->title = ucs2dup(src->fields[PP1DB_DATFIELD_TITLE].value.str);
	dst->artist = ucs2dup(src->fields[PP1DB_DATFIELD_ARTIST].value.str);
	dst->album = ucs2dup(src->fields[PP1DB_DATFIELD_ALBUM].value.str);
	dst->genre = ucs2dup(src->fields[PP1DB_DATFIELD_GENRE].value.str);
	itoucs2(src->fields[PP1DB_DATFIELD_YEAR].value.dword, tmp, 10);
	dst->date = ucs2dup(tmp);

	// Set codec information according to the file extensions.
	if (filepath_hasext(dst->filename, ucs2cs_mp3)) {
		dst->codec = PMPCODEC_MPEGLAYER3;
	} else if (filepath_hasext(dst->filename, ucs2cs_wma)) {
		dst->codec = PMPCODEC_WMA;
	} else if (filepath_hasext(dst->filename, ucs2cs_wav)) {
		dst->codec = PMPCODEC_WAV;
	}

	dst->track_number = src->fields[PP1DB_DATFIELD_TRACKNUMBER].value.dword;
	dst->sample_rate = src->fields[PP1DB_DATFIELD_SAMPLERATE].value.dword;
	dst->bitrate = src->fields[PP1DB_DATFIELD_BITRATE].value.dword;
	dst->duration = src->fields[PP1DB_DATFIELD_DURATION].value.dword;
	dst->filesize = src->fields[PP1DB_DATFIELD_FILESIZE].value.dword;
	dst->ts_update = src->fields[PP1DB_DATFIELD_UNKNOWN6].value.dword;
	dst->rating = src->fields[PP1DB_DATFIELD_RATING].value.dword;
	dst->play_count = src->fields[PP1DB_DATFIELD_PLAYCOUNT].value.dword;
	dst->ts_playback = src->fields[PP1DB_DATFIELD_RECENTPLAY].value.dword;

	return 0;
}

int iriver_h10_parse_model(const ucs2char_t* firmware, pp1model_t* model)
{
	/*
	iriver H10 5GB (UMS) 2.53

	 ADDRESS   00 01 02 03 04 05 06 07 08 09 0A 0B 0C 0D 0E 0F   0123456789ABCDEF 
	------------------------------------------------------------------------------
	 00000200  DD 03 00 EA CA 03 00 EA 0E F0 B0 E1 CE 03 00 EA   ................ 
	 00000210  D3 03 00 EA FE FF FF EA BB 03 00 EA B1 03 00 EA   ................ 
	 00000220  70 6F 72 74 61 6C 70 6C 61 79 65 72 00 30 2E 30   portalplayer.0.0 
	 00000230  C0 00 3D 03 14 00 94 CC 00 00 00 00 00 00 00 00   ..=............. 
	 00000240  50 50 35 30 32 30 41 46 2D 30 32 2E 35 33 2D 4B   PP5020AF-02.53-K 
	 00000250  4F 52 2D 55 4D 2D 44 54 00 00 00 00 32 30 30 35   OR-UM-DT....2005 
	 00000260  2E 30 35 2E 32 36 00 00 28 42 75 69 6C 64 20 31   .05.26..(Build 1 
	 00000270  32 39 2E 35 29 00 00 00 44 69 67 69 74 61 6C 20   29.5)...Digital  
	 00000280  4D 65 64 69 61 20 50 6C 61 74 66 6F 72 6D 00 00   Media Platform.. 
	 00000290  43 6F 70 79 72 69 67 68 74 28 63 29 20 31 39 39   Copyright(c) 199 
	 000002A0  39 20 2D 20 32 30 30 33 20 50 6F 72 74 61 6C 50   9 - 2003 PortalP 
	 000002B0  6C 61 79 65 72 2C 20 49 6E 63 2E 20 20 41 6C 6C   layer, Inc.  All 
	 000002C0  20 72 69 67 68 74 73 20 72 65 73 65 72 76 65 64    rights reserved 
	 000002D0  2E 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00   ................ 
	 000002E0  00 01 00 00 EC 00 00 00 14 03 3D 00 00 10 A0 E1   ..........=..... 
	 000002F0  40 25 A0 E3 00 20 81 E5 01 00 A0 E3 0E F0 A0 E1   @%... .......... 
	*/
	FILE *fp = ucs2fopen(firmware, "rb");
	if (fp) {
		char line[29], *p = NULL, *q = NULL;
		memset(line, 0, sizeof(line));

		// Seek to the firmware information.
		if (fseek(fp, 0x0240, SEEK_SET) != 0) {
			fclose(fp);
			return 0;
		}

		// Read the firmware information.
		if (fread(line, sizeof(char), 28, fp) != 28) {
			fclose(fp);
			return 0;
		}

		fclose(fp);

		// Parse product identifier.
		p = strchr(line, '-');
		if (!p) {
			return 0;
		}
		*p++ = 0;
		//strcpy(model->name, line);
		strcpy(model->name, "H10");

		// Parse and check version number.
		q = strchr(p, '-');
		if (!q) {
			return 0;
		}
		*q++ = 0;
		strcpy(model->version, p);
		p = q;

		// Parse language.
		q = strchr(p, '-');
		if (!q) {
			return 0;
		}
		*q++ = 0;
		strcpy(model->language, p);

		// Parse and check firmware mode.
		p = strchr(q, '-');
		if (!p) {
			return 0;
		}
		*p++ = 0;
		strcpy(model->mode, q);
		
		return 1;
	}
	return 0;
}




/********************************************************************
 * Database template for iriver H10 UMS players.                    *
 ********************************************************************/
static fd_template_t hdrtmpl_fd_h10_ums[] = {
	{0x0000F001, 2,   4, 0, 0, 1, 0, 0, "System\\DATA\\H10DB_@DEV.idx"},
	{0x0000F002, 1, 128, 0, 0, 1, 0, 0, "System\\DATA\\H10DB_FPTH.idx"},
	{0x0000F003, 1, 128, 0, 0, 1, 0, 0, "System\\DATA\\H10DB_FNAM.idx"},
	{0x0000F00A, 2,   4, 0, 0, 1, 0, 0, "System\\DATA\\H10DB_FRMT.idx"},
	{0x0000002E, 1,  40, 0, 0, 1, 0, 0, "System\\DATA\\H10DB_TIT2.idx"},
	{0x0000003C, 1,  40, 0, 0, 1, 0, 0, "System\\DATA\\H10DB_TPE1.idx"},
	{0x0000001C, 1,  40, 0, 0, 1, 0, 0, "System\\DATA\\H10DB_TALB.idx"},
	{0x0000001F, 1,  20, 0, 0, 1, 0, 0, "System\\DATA\\H10DB_TCON.idx"},
	{0x0000E002, 2,   4, 0, 0, 1, 0, 0, "System\\DATA\\H10DB_@DU3.idx"},
	{0x0000E003, 2,   4, 0, 0, 1, 0, 0, "System\\DATA\\H10DB_@DU4.idx"},
	{0x0000E004, 2,   4, 0, 0, 1, 0, 0, "System\\DATA\\H10DB_@DU5.idx"},
	{0x0000E005, 2,   4, 0, 0, 0, 0, 0, ""},
	{0x00000043, 2,   4, 0, 0, 1, 0, 0, "System\\DATA\\H10DB_TRCK.idx"},
	{0x0000004E, 2,   4, 0, 0, 0, 0, 0, ""},
	{0x0000F009, 2,   4, 0, 0, 0, 0, 0, ""},
	{0x0000F007, 2,   4, 0, 0, 0, 0, 0, ""},
	{0x0000F006, 2,   4, 0, 0, 0, 0, 0, ""},
	{0x0000F005, 2,   4, 0, 0, 0, 0, 0, ""},
	{0x0000E000, 2,   4, 0, 0, 1, 0, 0, "System\\DATA\\H10DB_@DU1.idx"},
	{0x0000E001, 1,  40, 0, 0, 1, 0, 0, "System\\DATA\\H10DB_@DU2.idx"},
	{0x00000083, 2,   4, 0, 0, 0, 0, 0, ""},
	{0x00000084, 1,  64, 0, 0, 0, 0, 0, ""},
};

static uint32_t hdrtmpl_max_dat_field_size_h10_ums[] =
	{8, 12, 268, 524, 528, 608, 688, 768, 808, 812, 816, 820, 824, 828, 832, 836, 840, 844, 848, 852, 932, 936};


/********** H10 5GB/6GB UMS firmware 2.05-2.53 **********/
static hdr_template_t hdrtmpl_h10_5gb_ums_0205_0253 = {
	0, 0, "System\\DATA\\H10DB.dat", 1, "System\\DATA\\H10DB.hdr", 0x00000428, 0, 0, 22,
	hdrtmpl_fd_h10_ums,
	hdrtmpl_max_dat_field_size_h10_ums,
	0, 0,
	{206148, 22, 4000, 1032, 0, iriver_h10_dat_repr, iriver_h10_dat_set, iriver_h10_dat_get},
};
int hdr_init_h10_5gb_ums_0205_0253(hdr_t* hdr)
{
	return apply_template(hdr, &hdrtmpl_h10_5gb_ums_0205_0253);
}





/********************************************************************
 * Database template for iriver H10 MTP players.                    *
 ********************************************************************/
static fd_template_t hdrtmpl_fd_h10_mtp[] = {
	{0x0000F001, 2,   4, 0, 0, 1, 0, 0, "SYSTEM\\DATA\\H10DB_@DEV.IDX"},
	{0x0000F002, 1, 256, 0, 0, 1, 0, 0, "SYSTEM\\DATA\\H10DB_FPTH.IDX"},
	{0x0000F003, 1, 256, 0, 0, 1, 0, 0, "SYSTEM\\DATA\\H10DB_FNAM.IDX"},
	{0x0000F00A, 2,   4, 0, 0, 1, 0, 0, "SYSTEM\\DATA\\H10DB_FRMT.IDX"},
	{0x0000002E, 1, 256, 0, 0, 1, 0, 0, "SYSTEM\\DATA\\H10DB_TIT2.IDX"},
	{0x0000003C, 1, 256, 0, 0, 1, 0, 0, "SYSTEM\\DATA\\H10DB_TPE1.IDX"},
	{0x0000001C, 1, 256, 0, 0, 1, 0, 0, "SYSTEM\\DATA\\H10DB_TALB.IDX"},
	{0x0000001F, 1, 256, 0, 0, 1, 0, 0, "SYSTEM\\DATA\\H10DB_TCON.IDX"},
	{0x0000E002, 2,   4, 0, 0, 1, 0, 0, "SYSTEM\\DATA\\H10DB_@DU3.IDX"},
	{0x0000E003, 2,   4, 0, 0, 1, 0, 0, "SYSTEM\\DATA\\H10DB_@DU4.IDX"},
	{0x0000E004, 2,   4, 0, 0, 1, 0, 0, "SYSTEM\\DATA\\H10DB_@DU5.IDX"},
	{0x0000E005, 2,   4, 0, 0, 0, 0, 0, ""},
	{0x00000043, 2,   4, 0, 0, 1, 0, 0, "SYSTEM\\DATA\\H10DB_TRCK.IDX"},
	{0x0000004E, 2,   4, 0, 0, 0, 0, 0, ""},
	{0x0000F009, 2,   4, 0, 0, 0, 0, 0, ""},
	{0x0000F007, 2,   4, 0, 0, 0, 0, 0, ""},
	{0x0000F006, 2,   4, 0, 0, 0, 0, 0, ""},
	{0x0000F005, 2,   4, 0, 0, 0, 0, 0, ""},
	{0x0000E000, 2,   4, 0, 0, 1, 0, 0, "SYSTEM\\DATA\\H10DB_@DU1.IDX"},
	{0x0000E001, 1,  40, 0, 0, 1, 0, 0, "SYSTEM\\DATA\\H10DB_@DU2.IDX"},
	{0x00000083, 2,   4, 0, 0, 0, 0, 0, ""},
	{0x00000084, 1,  64, 0, 0, 0, 0, 0, ""},
};

static uint32_t hdrtmpl_max_dat_field_size_h10_mtp[] =
	{8, 12, 524, 1036, 1040, 1552, 2064, 2576, 3088, 3092, 3096, 3100, 3104, 3108, 3112, 3116, 3120, 3124, 3128, 3132, 3212, 3216};


/********** H10 5GB/6GB MTP firmware 2.03-2.10 **********/
static hdr_template_t hdrtmpl_h10_5gb_mtp_0203_0210 = {
	0, 0, "SYSTEM\\DATA\\H10DB.DAT", 1, "SYSTEM\\DATA\\H10DB.HDR", 0x00000D10, 0, 0, 22,
	hdrtmpl_fd_h10_mtp,
	hdrtmpl_max_dat_field_size_h10_mtp,
	0, 0,
	{205116, 22, 4000, 0, 0, iriver_h10_dat_repr, iriver_h10_dat_set, iriver_h10_dat_get},
};
int hdr_init_h10_5gb_mtp_0203_0210(hdr_t* hdr)
{
	return apply_template(hdr, &hdrtmpl_h10_5gb_mtp_0203_0210);
}



/********** H10 5GB/6GB MTP firmware 2.51 **********/
static hdr_template_t hdrtmpl_h10_5gb_mtp_0251 = {
	0, 0, "SYSTEM\\DATA\\H10DB.DAT", 1, "SYSTEM\\DATA\\H10DB.HDR", 0x00000D10, 0, 0, 22,
	hdrtmpl_fd_h10_mtp,
	hdrtmpl_max_dat_field_size_h10_mtp,
	0, 1,
	{205120, 22, 4000, 0, 1, iriver_h10_dat_repr, iriver_h10_dat_set, iriver_h10_dat_get},
};
int hdr_init_h10_5gb_mtp_0251(hdr_t* hdr)
{
	return apply_template(hdr, &hdrtmpl_h10_5gb_mtp_0251);
}



/********** H10 20GB MTP firmware 1.00-1.02 **********/
static hdr_template_t hdrtmpl_h10_20gb_mtp_0100_0102 = {
	0, 0, "SYSTEM\\DATA\\H10DB.DAT", 1, "SYSTEM\\DATA\\H10DB.HDR", 0x00000D10, 0, 0, 22,
	hdrtmpl_fd_h10_mtp,
	hdrtmpl_max_dat_field_size_h10_mtp,
	0, 1,
	{397116, 22, 8000, 0, 0, iriver_h10_dat_repr, iriver_h10_dat_set, iriver_h10_dat_get},
};
int hdr_init_h10_20gb_mtp_0100_0102(hdr_t* hdr)
{
	return apply_template(hdr, &hdrtmpl_h10_20gb_mtp_0100_0102);
}



/********** H10 20GB MTP firmware 2.51 **********/
static hdr_template_t hdrtmpl_h10_20gb_mtp_0251 = {
	0, 0, "SYSTEM\\DATA\\H10DB.DAT", 1, "SYSTEM\\DATA\\H10DB.HDR", 0x00000D10, 0, 0, 22,
	hdrtmpl_fd_h10_mtp,
	hdrtmpl_max_dat_field_size_h10_mtp,
	0, 1,
	{397120, 22, 8000, 0, 1, iriver_h10_dat_repr, iriver_h10_dat_set, iriver_h10_dat_get},
};
int hdr_init_h10_20gb_mtp_0251(hdr_t* hdr)
{
	return apply_template(hdr, &hdrtmpl_h10_20gb_mtp_0251);
}

