/*
 *      MSI MEGA PLAYER specific routines and header templates.
 *
 *      Copyright (c) 2005-2007 Naoaki Okazaki
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 *
 */

/* $Id: model_msi_megaplayer.c 342 2007-02-11 18:11:43Z nyaochi $ */

#ifdef	HAVE_CONFIG_H
#include <config.h>
#endif/*HAVE_CONFIG_H*/
#ifdef	HAVE_STRING_H
#include <string.h>
#endif/*HAVE_STRING_H*/

#include <os.h>
#include <stdio.h>
#include <stdlib.h>
#include <pmplib/ucs2char.h>
#include <pmplib/filepath.h>

#include "serialize.h"
#include "util.h"
#include "pp1db.h"
#include "hdr_template.h"

enum {
	PP1DB_DATFIELD_UNKNOWN2 = 0,// @0:	(INT)	
	PP1DB_DATFIELD_PATHNAME,	// @1:	(STR)	
	PP1DB_DATFIELD_FILENAME,	// @2:	(STR)	
	PP1DB_DATFIELD_FORMAT,		// @3:	(INT)
	PP1DB_DATFIELD_BITRATE,		// @4:	(INT)
	PP1DB_DATFIELD_SAMPLERATE,	// @5:	(INT)
	PP1DB_DATFIELD_DURATION,	// @6:	(STR)
	PP1DB_DATFIELD_ARTIST,		// @7:	(STR)
	PP1DB_DATFIELD_ALBUM,		// @8:	(STR)
	PP1DB_DATFIELD_GENRE,		// @9:	(STR)
	PP1DB_DATFIELD_TITLE,		// @10:	(STR)
	PP1DB_DATFIELD_TRACKNUMBER,	// @11:	(INT)
	PP1DB_DATFIELD_YEAR,		// @12: (INT)
	PP1DB_DATFIELD_FILESIZE,	// @13: (INT)
};

static void dat_repr(const dat_t* record, FILE *fp)
{
	fprintf(fp, "  inactive: %d\n", record->status);
	fprintf(fp, "  unknown1: %d\n", record->unknown1);
	fprintf(fp, "  unknown2: %d\n", record->fields[PP1DB_DATFIELD_UNKNOWN2].value.dword);
	fprints(fp, "  file_path: %s\n", record->fields[PP1DB_DATFIELD_PATHNAME].value.str);
	fprints(fp, "  file_name: %s\n", record->fields[PP1DB_DATFIELD_FILENAME].value.str);
	fprintf(fp, "  media_type: %d\n", record->fields[PP1DB_DATFIELD_FORMAT].value.dword);
	fprintf(fp, "  bitrate: %d\n", record->fields[PP1DB_DATFIELD_BITRATE].value.dword);
	fprintf(fp, "  samplerate: %d\n", record->fields[PP1DB_DATFIELD_SAMPLERATE].value.dword);
	fprints(fp, "  duration: %s\n", record->fields[PP1DB_DATFIELD_DURATION].value.str);
	fprints(fp, "  artist: %s\n", record->fields[PP1DB_DATFIELD_ARTIST].value.str);
	fprints(fp, "  album: %s\n", record->fields[PP1DB_DATFIELD_ALBUM].value.str);
	fprints(fp, "  genre: %s\n", record->fields[PP1DB_DATFIELD_GENRE].value.str);
	fprints(fp, "  title: %s\n", record->fields[PP1DB_DATFIELD_TITLE].value.str);
	fprintf(fp, "  number: %d\n", record->fields[PP1DB_DATFIELD_TRACKNUMBER].value.dword);
	fprintf(fp, "  year: %d\n", record->fields[PP1DB_DATFIELD_YEAR].value.dword);
	fprintf(fp, "  filesize: %d\n", record->fields[PP1DB_DATFIELD_FILESIZE].value.dword);
}

static int dat_set(dat_t* dst, const pmp_music_record_t* src, const ucs2char_t* path_to_root)
{
	static const ucs2char_t ucs2cs_unknown[] = {0};
	static const ucs2char_t ucs2cs_empty[] = {0};
	ucs2char_t tmp[128];

	// Set fields.
	dst->status = 0;
	dst->unknown1 = 0;
	dst->fields[PP1DB_DATFIELD_PATHNAME].value.str = ucs2dup(filepath_skiproot(src->filename, path_to_root));
	filepath_remove_filespec(dst->fields[PP1DB_DATFIELD_PATHNAME].value.str);
	filepath_addslash(dst->fields[PP1DB_DATFIELD_PATHNAME].value.str);
	filepath_encode(dst->fields[PP1DB_DATFIELD_PATHNAME].value.str);
	dst->fields[PP1DB_DATFIELD_FILENAME].value.str = ucs2dup(filepath_skippath(src->filename));
	dst->fields[PP1DB_DATFIELD_FORMAT].value.dword = 0;
	dst->fields[PP1DB_DATFIELD_BITRATE].value.dword = src->bitrate;
	dst->fields[PP1DB_DATFIELD_SAMPLERATE].value.dword = src->sample_rate;
	itoucs2(src->duration, tmp, 10);
	dst->fields[PP1DB_DATFIELD_DURATION].value.str = ucs2dup(tmp);
	dst->fields[PP1DB_DATFIELD_ARTIST].value.str = ucs2dup(src->artist ? src->artist : ucs2cs_unknown);
	dst->fields[PP1DB_DATFIELD_ALBUM].value.str = ucs2dup(src->album ? src->album : ucs2cs_unknown);
	dst->fields[PP1DB_DATFIELD_GENRE].value.str = ucs2dup(src->genre ? src->genre : ucs2cs_unknown);
	dst->fields[PP1DB_DATFIELD_TITLE].value.str = ucs2dup(src->title ? src->title : dst->fields[PP1DB_DATFIELD_FILENAME].value.str);
	dst->fields[PP1DB_DATFIELD_TRACKNUMBER].value.dword = src->track_number;
	if (src->date) {
		dst->fields[PP1DB_DATFIELD_YEAR].value.dword = ucs2toi(src->date);
	}
	dst->fields[PP1DB_DATFIELD_FILESIZE].value.dword = src->filesize;
	return 0;
}

static int dat_get(pmp_music_record_t* dst, const dat_t* src, const ucs2char_t* path_to_root)
{
	static const ucs2char_t ucs2cs_mp3[] = {'.','m','p','3',0};
	static const ucs2char_t ucs2cs_wma[] = {'.','w','m','a',0};
	static const ucs2char_t ucs2cs_wav[] = {'.','w','a','v',0};
	ucs2char_t tmp[128];
	size_t length = 0;

	length  = ucs2len(path_to_root);
	length += ucs2len(src->fields[PP1DB_DATFIELD_PATHNAME].value.str);
	length += ucs2len(src->fields[PP1DB_DATFIELD_FILENAME].value.str);
	length += 3;

	dst->filename = (ucs2char_t*)ucs2malloc(sizeof(ucs2char_t) * length);
	if (dst->filename) {
		filepath_combinepath(dst->filename, length, path_to_root, src->fields[PP1DB_DATFIELD_PATHNAME].value.str);
		ucs2cat(dst->filename, src->fields[PP1DB_DATFIELD_FILENAME].value.str);
		filepath_decode(dst->filename);
	}

	dst->bitrate = src->fields[PP1DB_DATFIELD_BITRATE].value.dword;
	dst->sample_rate = src->fields[PP1DB_DATFIELD_SAMPLERATE].value.dword;
	dst->duration = ucs2toi(src->fields[PP1DB_DATFIELD_DURATION].value.str);
	dst->artist = ucs2dup(src->fields[PP1DB_DATFIELD_ARTIST].value.str);
	dst->album = ucs2dup(src->fields[PP1DB_DATFIELD_ALBUM].value.str);
	dst->genre = ucs2dup(src->fields[PP1DB_DATFIELD_GENRE].value.str);
	dst->title = ucs2dup(src->fields[PP1DB_DATFIELD_TITLE].value.str);

	// Set codec information according to the file extensions.
	if (filepath_hasext(dst->filename, ucs2cs_mp3)) {
		dst->codec = PMPCODEC_MPEGLAYER3;
	} else if (filepath_hasext(dst->filename, ucs2cs_wma)) {
		dst->codec = PMPCODEC_WMA;
	} else if (filepath_hasext(dst->filename, ucs2cs_wav)) {
		dst->codec = PMPCODEC_WAV;
	}

	dst->track_number = src->fields[PP1DB_DATFIELD_TRACKNUMBER].value.dword;
	itoucs2(src->fields[PP1DB_DATFIELD_YEAR].value.dword, tmp, 10);
	dst->date = ucs2dup(tmp);
	dst->filesize = src->fields[PP1DB_DATFIELD_FILESIZE].value.dword;
	//dst->ts_update = src->fields[PP1DB_DATFIELD_UNKNOWN4].value.dword;

	return 0;
}

int megaplayer_parse_model(const ucs2char_t* firmware, pp1model_t* model)
{
	/*
	MSI MEGA PLAYER 540 01.00.16

	 ADDRESS   00 01 02 03 04 05 06 07 08 09 0A 0B 0C 0D 0E 0F   0123456789ABCDEF 
	------------------------------------------------------------------------------
	 00000200  27 04 00 EA 14 04 00 EA 0E F0 B0 E1 18 04 00 EA   '............... 
	 00000210  1D 04 00 EA FE FF FF EA 05 04 00 EA FB 03 00 EA   ................ 
	 00000220  70 6F 72 74 61 6C 70 6C 61 79 65 72 00 30 2E 30   portalplayer.0.0 
	 00000230  C0 00 39 7F 20 00 CA 66 00 00 00 00 00 00 00 00   ..9. ..f........ 
	 00000240  50 50 35 30 32 30 41 46 2D 30 35 2E 32 32 2D 50   PP5020AF-05.22-P 
	 00000250  50 30 37 2D 30 35 2E 32 32 2D 4D 4E 30 35 2D 30   P07-05.22-MN05-0 
	 00000260  31 2E 30 30 2E 31 36 2D 44 54 00 00 32 30 30 35   1.00.16-DT..2005 
	 00000270  2E 30 33 2E 31 36 00 00 28 42 75 69 6C 64 20 31   .03.16..(Build 1 
	 00000280  32 39 29 00 44 69 67 69 74 61 6C 20 4D 65 64 69   29).Digital Medi 
	 00000290  61 20 50 6C 61 74 66 6F 72 6D 00 00 43 6F 70 79   a Platform..Copy 
	 000002A0  72 69 67 68 74 28 63 29 20 31 39 39 39 20 2D 20   right(c) 1999 -  
	 000002B0  32 30 30 33 20 50 6F 72 74 61 6C 50 6C 61 79 65   2003 PortalPlaye 
	 000002C0  72 2C 20 49 6E 63 2E 20 20 41 6C 6C 20 72 69 67   r, Inc.  All rig 
	 000002D0  68 74 73 20 72 65 73 65 72 76 65 64 2E 00 00 00   hts reserved.... 
	 000002E0  00 01 00 00 EC 00 00 00 20 7F 39 00 00 10 A0 E1   ........ .9..... 
	 000002F0  40 25 A0 E3 00 20 81 E5 01 00 A0 E3 0E F0 A0 E1   @%... .......... 
	*/
	FILE *fp = ucs2fopen(firmware, "rb");
	if (fp) {
		char buff[129], *p = NULL, *q = NULL;
		memset(buff, 0, sizeof(buff));

		// Seek to the firmware information.
		if (fseek(fp, 0x0240, SEEK_SET) != 0) {
			fclose(fp);
			return 0;
		}

		// Read the firmware information.
		if (fread(buff, sizeof(char), 128, fp) != 128) {
			fclose(fp);
			return 0;
		}

		// Close the firmware.
		fclose(fp);

		// Match the identifier string.
		if (strncmp(buff, "PP5020AF-05.22-PP07-05.22-MN05-", 31) != 0) {
			return 0;
		}

		// Obtain the model name.
		strcpy(model->name, "MEGA PLAYER 540");
		memset(model->version, 0, sizeof(model->version));
		strncpy(model->version, buff+0x1F, 8);
		strcpy(model->language, "");
		strcpy(model->mode, "UM");

		return 1;
	} else {
		return 0;
	}
}




static fd_template_t hdrtmpl_fd_megaplayer540[] = {
	{0x0000F001, 2,   4, 0, 0, 1, 0, 0, "System\\DATA\\PP5000_@DEV.idx"},
	{0x0000F002, 1, 128, 0, 0, 1, 0, 0, "System\\DATA\\PP5000_FPTH.idx"},
	{0x0000F003, 1, 128, 0, 0, 1, 0, 0, "System\\DATA\\PP5000_FNAM.idx"},
	{0x0000F00A, 2,   4, 0, 0, 1, 0, 0, "System\\DATA\\PP5000_FRMT.idx"},
	{0x0000F005, 2,   4, 0, 0, 0, 0, 0, ""},
	{0x0000F006, 2,   4, 0, 0, 0, 0, 0, ""},
	{0x0000F007, 1,   4, 0, 0, 0, 0, 0, ""},
	{0x0000003C, 1, 256, 0, 0, 1, 0, 0, "System\\DATA\\PP5000_TPE1.idx"},
	{0x0000001C, 1, 256, 0, 0, 1, 0, 0, "System\\DATA\\PP5000_TALB.idx"},
	{0x0000001F, 1, 256, 0, 0, 1, 0, 0, "System\\DATA\\PP5000_TCON.idx"},
	{0x0000002E, 1, 256, 0, 0, 1, 0, 0, "System\\DATA\\PP5000_TIT2.idx"},
	{0x00000043, 2,   4, 0, 0, 0, 0, 0, ""},
	{0x0000004E, 2,   4, 0, 0, 0, 0, 0, ""},
	{0x0000F009, 2,   4, 0, 0, 0, 0, 0, ""},
	{0x00000000, 0,   0, 0, 0, 0, 0, 0, ""},
	{0x00000000, 0,   0, 0, 0, 0, 0, 0, ""},
	{0x00000000, 0,   0, 0, 0, 0, 0, 0, ""},
	{0x00000000, 0,   0, 0, 0, 0, 0, 0, ""},
	{0x00000000, 0,   0, 0, 0, 0, 0, 0, ""},
	{0x00000000, 0,   0, 0, 0, 0, 0, 0, ""},
};

static uint32_t hdrtmpl_max_dat_field_size_megaplayer540[] =
	{8, 12, 268, 524, 528, 532, 536, 544, 1056, 1568, 2080, 2592, 2596, 2600, 0, 0, 0, 0, 0, 0};


/********** MSI MEGA PLAYER 540 **********/
static hdr_template_t hdrtmpl_megaplayer540 = {
	0, 0, "System\\DATA\\PP5000.DAT", 0, "System\\DATA\\PP5000.HDR", 0x00000A2C, 0, 0, 14,
	hdrtmpl_fd_megaplayer540,
	hdrtmpl_max_dat_field_size_megaplayer540,
	0, 0,
	{100020, 20, 2000, 0, 0, dat_repr, dat_set, dat_get},
};

int hdr_init_msi_megaplayer540(hdr_t* hdr)
{
	return apply_template(hdr, &hdrtmpl_megaplayer540);
}
