/*
 *      Playlist reader/writer for PortalPlayer (e.g., *.plp and *.pla).
 *
 *      Copyright (c) 2005-2007 Naoaki Okazaki
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 *
 */

/* $Id: playlist.c 328 2007-02-10 17:50:11Z nyaochi $ */

#ifdef	HAVE_CONFIG_H
#include <config.h>
#endif/*HAVE_CONFIG_H*/

#include <os.h>
#include <stdio.h>
#include <stdlib.h>
#include <pmplib/ucs2char.h>
#include <pmplib/filepath.h>

#include "serialize.h"
#include "pp1db.h"

static int write_ucs2le(FILE *fp, const ucs2char_t* str)
{
	int count = 0;

	for (;*str;str++) {
		uint8_t v[2];
		v[0] = *str & 0x00FF;
		v[1] = *str >> 8;
		if (fwrite(v, sizeof(uint8_t), 2, fp) != 2) {
			break;
		}
		count++;
	}
	return count;
}

int pp1db_playlist_write(
	pp1db_t* db,
	const ucs2char_t *name,
	ucs2char_t* const mediafiles[],
	int num_mediafiles,
	const ucs2char_t *path_to_root,
	const ucs2char_t *path_to_playlist,
	const ucs2char_t *ext
	)
{
	FILE *fp = NULL;
	static const ucs2char_t header1_str[] = {'P','L','P',' ','P','L','A','Y','L','I','S','T','\r','\n', 0};
	static const ucs2char_t header2_str[] = {'V','E','R','S','I','O','N',' ','1','.','2','0','\r','\n', 0};
	static const ucs2char_t entry_str[] = {'H','D','D',',',' ', 0};
	static const ucs2char_t crlf_str[] = {'\r','\n', 0};
	ucs2char_t dst[MAX_PATH];

	filepath_combinepath(dst, MAX_PATH, path_to_root, path_to_playlist);
	filepath_addslash(dst);
	ucs2cat(dst, name);
	ucs2cat(dst, ext);
	
	fp = ucs2fopen(dst, "wb");
	if (!fp) {
		return -1;
	} else {
		int i;

		write_ucs2le(fp, header1_str);
		write_ucs2le(fp, header2_str);
		write_ucs2le(fp, crlf_str);

		for (i = 0;i < num_mediafiles;i++) {
			if (mediafiles[i][0]) {
				ucs2char_t filename[MAX_PATH];
				ucs2cpy(filename, filepath_skiproot(mediafiles[i], path_to_root));

				filepath_encode(filename);

				write_ucs2le(fp, entry_str);
				write_ucs2le(fp, filename);
				write_ucs2le(fp, crlf_str);
			}
		}

		write_ucs2le(fp, crlf_str);

		fclose(fp);
		return 0;
	}
}
