/*
 *      UCS-2 implementation (common routines).
 *
 *      Copyright (c) 2005-2007 Naoaki Okazaki
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 *
 */

/* $Id: ucs2char.c 328 2007-02-10 17:50:11Z nyaochi $ */

#ifdef	HAVE_CONFIG_H
#include <config.h>
#endif/*HAVE_CONFIG_H*/

#include <os.h>
#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include <pmplib/ucs2char.h>

/** 
 * \addtogroup ucs2 
 * @{
 */

#define	COMP(a, b)	((a)>(b))-((a)<(b))


void *ucs2malloc(size_t size)
{
	return malloc(size);
}

void *ucs2calloc(size_t size)
{
	return calloc(size, 1);
}

void *ucs2realloc(void *ptr, size_t size)
{
	return realloc(ptr, size);
}

void ucs2free(void* str)
{
	free(str);
}

int ucs2issurrogate(ucs2char_t c)
{
	return (0xD800 <= c && c <= 0xDFFF);
}

int ucs2isspace(ucs2char_t c)
{
	return ((0x0009 <= c && c <= 0x000D) || c == 0x0020);
}

int ucs2isdigit(ucs2char_t c)
{
	return (0x0030 <= c && c <= 0x0039);
}

size_t ucs2len(const ucs2char_t* string)
{
	size_t length = 0;
	while (*string++)	length++;
	return length;
}

ucs2char_t* ucs2cpy(ucs2char_t* dst, const ucs2char_t* src)
{
	ucs2char_t *p = dst;
	for (;;) {
		*p = *src;
		if (*src == 0) {
			break;
		}
		p++;
		src++;
	}
	return dst;
}

ucs2char_t* ucs2cat(ucs2char_t* dst, const ucs2char_t* src)
{
	ucs2char_t *p = dst;
	while (*p)	p++;
	ucs2cpy(p, src);
	return dst;
}

size_t ucs2cspn(const ucs2char_t *str, const ucs2char_t *charset)
{
	const ucs2char_t *p, *q;

	for (p = str;*p;++p) {
		for (q = charset;*q;++q) {
			if (*p == *q) {
				return (p - str);
			}
		}
	}
	return (p - str);
}

size_t ucs2spn(const ucs2char_t *str, const ucs2char_t *charset)
{
	const ucs2char_t *p, *q;

	for (p = str;*p;++p) {
		for (q = charset;*q;++q) {
			if (*p != *q) {
				return (p - str);
			}
		}
	}
	return (p - str);
}

ucs2char_t* ucs2pbrk(const ucs2char_t *str, const ucs2char_t *search)
{
	const ucs2char_t *p, *q;

	for (p = str;*p;++p) {
		for (q = search;*q;++q) {
			if (*p == *q) {
				return (ucs2char_t*)p;
			}
		}
	}
	return NULL;
}

ucs2char_t* ucs2ncat(ucs2char_t *dst, const ucs2char_t *src, size_t count)
{
	size_t i;
	ucs2char_t *p = dst + ucs2len(dst);
	for (i = 0;i < count;i++) {
		*p = *src;
		if (!*src) {
			break;
		}
		p++;
		src++;
	}
	return dst;
}

ucs2char_t* ucs2ncpy(ucs2char_t* dst, const ucs2char_t* src, size_t count)
{
	size_t i;
	ucs2char_t *p = dst;
	for (i = 0;i < count;i++) {
		*p = *src;
		if (!*src) {
			break;
		}
		p++;
		src++;
	}
	return dst;
}


ucs2char_t* ucs2chr(const ucs2char_t* string, ucs2char_t c)
{
	ucs2char_t* ret = 0;
	while (*string) {
		if (*string == c)	{
			ret = (ucs2char_t*)string;
			break;
		}
		string++;
	}
	return ret;
}

ucs2char_t* ucs2rchr(const ucs2char_t* string, ucs2char_t c)
{
	ucs2char_t* ret = 0;
	while (*string) {
		if (*string == c) {
			ret = (ucs2char_t*)string;
		}
		string++;
	}
	return ret;
}

ucs2char_t* ucs2str(const ucs2char_t* str, const ucs2char_t* search)
{
	size_t length = ucs2len(search);
	while (*str) {
		if (memcmp(str, search, length*sizeof(ucs2char_t)) == 0) {
			return (ucs2char_t*)str;
		}
		str++;
	}
	return 0;
}

int ucs2cmp(const ucs2char_t* x, const ucs2char_t* y)
{
	while (*x && *y && *x == *y) {
		x++;
		y++;
	}
	return COMP(*x, *y);
}

int ucs2ncmp(const ucs2char_t* x, const ucs2char_t* y, size_t n)
{
	size_t i;

	for (i = 0;i < n-1;i++) {
		if (!*x || !*y || *x != *y) {
			break;
		}
		x++;
		y++;
	}
	return COMP(*x, *y);
}

int ucs2icmp(const ucs2char_t* x, const ucs2char_t* y)
{
	ucs2char_t a, b;

	do {
		a = ucs2upper(*x);
		b = ucs2upper(*y);
		if (!*x || !*y) {
			break;
		}
		x++;
		y++;
	} while (a == b);
	return COMP(a, b);
}

int ucs2incmp(const ucs2char_t* x, const ucs2char_t* y, size_t n)
{
	size_t i;
	ucs2char_t a = 0, b = 0;

	for (i = 0;i < n-1;i++) {
		a = ucs2upper(*x);
		b = ucs2upper(*y);
		if (!*x || !*y || a != b) {
			break;
		}
		x++;
		y++;
	}
	return COMP(ucs2upper(*x), ucs2upper(*y));
}

ucs2char_t* ucs2dup(const ucs2char_t* src)
{
	ucs2char_t* dst = (ucs2char_t*)malloc((ucs2len(src)+1) * sizeof(ucs2char_t));
	ucs2cpy(dst, src);
	return dst;
}

ucs2char_t* ucs2ndup(const ucs2char_t* src, size_t length)
{
	ucs2char_t* dst = NULL;
	size_t src_length = ucs2len(src);
	if (length < src_length) {
		src_length = length;
	}
	dst = (ucs2char_t*)malloc((src_length+1) * sizeof(ucs2char_t));
	if (dst) {
		size_t i;
		for (i = 0;i < src_length;i++) {
			dst[i] = src[i];
		}
		dst[i] = 0;
	}
	return dst;
}

ucs2char_t* ucs2strip(ucs2char_t* str)
{
	if (str && *str) {
		ucs2char_t* p = NULL;
		for (p = str + ucs2len(str) - 1;str <= p;p--) {
			if (!ucs2isspace(*p)) {
				*(p + 1) = 0;
				break;
			}
		}

		for (p = str;p < str + ucs2len(str);p++) {
			if (!ucs2isspace(*p)) {
				break;
			}
		}

		if (p != str) {
			ucs2char_t *q = NULL;
			for (q = str;*p;p++,q++) {
				*q = *p;
			}
			*q = 0;
		}
	}
	return str;
}

ucs2char_t ucs2lower(ucs2char_t ch)
{
	/* iriver only converts what we call one-bytes characters. */
	/* We must limit the range of case conversion. */
	if (!(ch & 0xFF80)) {
		ch = tolower(ch);
	}
	return ch;
}

ucs2char_t* ucs2lwr(ucs2char_t* str)
{
	ucs2char_t* p = str;
	while (*p) {
		*p = ucs2lower(*p);
		p++;
	}
	return str;
}

ucs2char_t ucs2upper(ucs2char_t ch)
{
	/* iriver only converts what we call one-bytes characters. */
	/* We must limit the range of case conversion. */
	if (!(ch & 0xFF80)) {
		ch = toupper(ch);
	}
	return ch;
}

ucs2char_t* ucs2upr(ucs2char_t* str)
{
	ucs2char_t* p = str;
	while (*p) {
		*p = ucs2upper(*p);
		p++;
	}
	return str;
}

static void print_ucs2(const ucs2char_t* str)
{
  while (*str) {
    fprintf(stderr, "0x%X %c ", *str);
    str++;
  }
  fprintf(stderr, "\n");
}

ucs2char_t* mbsdupucs2(const char *mbstr)
{
	size_t ucs2_size = 	mbstoucs2(NULL, 0, mbstr, strlen(mbstr)) + 1;
	ucs2char_t* dst = (ucs2char_t*)malloc(ucs2_size * sizeof(ucs2char_t));
	if (dst) {
		mbstoucs2(dst, ucs2_size * sizeof(ucs2char_t), mbstr, strlen(mbstr)+1);
	}
	return dst;
}

ucs2char_t* mbsdupucs2_charset(const char *mbstr, const char *charset)
{
	size_t ucs2_size = 	mbstoucs2_charset(NULL, 0, mbstr, strlen(mbstr), charset) + 1;
	ucs2char_t* dst = (ucs2char_t*)malloc(ucs2_size * sizeof(ucs2char_t));
	if (dst) {
		mbstoucs2_charset(dst, ucs2_size * sizeof(ucs2char_t), mbstr, strlen(mbstr)+1, charset);
	}
	return dst;
}

ucs2char_t* utf8dupucs2(const char *mbstr)
{
	size_t ucs2_size = 	utf8toucs2(NULL, 0, mbstr, strlen(mbstr)) + 1;
	ucs2char_t* dst = (ucs2char_t*)malloc(ucs2_size * sizeof(ucs2char_t));
	if (dst) {
		utf8toucs2(dst, ucs2_size * sizeof(ucs2char_t), mbstr, strlen(mbstr)+1);
	}
	return dst;
}

char *ucs2dupmbs(const ucs2char_t *ucs2str)
{
	size_t mbs_size = ucs2tombs(NULL, 0, ucs2str, ucs2len(ucs2str)) + 1;
	char* dst = (char *)malloc(mbs_size * sizeof(char));
	if (dst) {
		ucs2tombs(dst, mbs_size, ucs2str, ucs2len(ucs2str)+1);
	}
	return dst;
}
