/*
 * gst-single-filter-manager.c - Source for GstSingleFilterManager
 *
 * Copyright (C) 2010 Collabora Ltd.
 *  @author: Youness Alaoui <youness.alaoui@collabora.co.uk>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * SECTION:gst-single-filter-manager
 * @short_description: A filter manager that can only be applied once
 *
 * This class, implementing the #GstFilterManager interface can only be applied
 * once on a single pad.
 *
 * See also #GstFilterManager
 */


#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif


#include <gst/filters/gst-single-filter-manager.h>
#include "gstfilters-marshal.h"


static void gst_single_filter_manager_interface_init (GstFilterManagerInterface
    * iface);

G_DEFINE_TYPE_WITH_CODE (GstSingleFilterManager,
    gst_single_filter_manager, G_TYPE_OBJECT,
    G_IMPLEMENT_INTERFACE (GST_TYPE_FILTER_MANAGER,
        gst_single_filter_manager_interface_init));

static void gst_single_filter_manager_dispose (GObject * object);
static void gst_single_filter_manager_finalize (GObject * object);
static void gst_single_filter_manager_get_property (GObject * object,
    guint property_id, GValue * value, GParamSpec * pspec);
static void free_filter_id (GstFilterId * id);
static void pad_block_do_nothing (GstPad * pad,
    gboolean blocked, gpointer user_data);

static GList *gst_single_filter_manager_list_filters (GstFilterManager * iface);
static GstFilterId
    * gst_single_filter_manager_insert_filter_before (GstFilterManager * iface,
    GstFilter * filter, GstFilterId * before);
static GstFilterId
    * gst_single_filter_manager_insert_filter_after (GstFilterManager * iface,
    GstFilter * filter, GstFilterId * after);
static GstFilterId *gst_single_filter_manager_replace_filter (GstFilterManager *
    iface, GstFilter * filter, GstFilterId * replace);
static GstFilterId *gst_single_filter_manager_insert_filter (GstFilterManager *
    iface, GstFilter * filter, gint position);
static GstFilterId
    * gst_single_filter_manager_insert_filter_unlock (GstSingleFilterManager *
    self, GstFilter * filter, gint position);
static gboolean gst_single_filter_manager_remove_filter (GstFilterManager *
    iface, GstFilterId * id);
static GstFilter *gst_single_filter_manager_get_filter_by_id (GstFilterManager *
    iface, GstFilterId * id);
static GstPad *gst_single_filter_manager_apply (GstFilterManager * iface,
    GstBin * bin, GstPad * pad);
static GstPad *gst_single_filter_manager_revert (GstFilterManager * iface,
    GstBin * bin, GstPad * pad);
static gboolean gst_single_filter_manager_handle_message (GstFilterManager *
    iface, GstMessage * message);

typedef enum
{ INSERT, REMOVE, REPLACE } ModificationAction;

/* signals  */
enum
{
  SIGNAL_FILTER_APPLIED,
  SIGNAL_FILTER_FAILED,
  SIGNAL_FILTER_REVERTED,
  LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = { 0 };

enum
{
  PROP_APPLIED = 1,
  PROP_APPLIED_BIN,
  PROP_APPLIED_PAD,
  PROP_OUT_PAD,
  LAST_PROPERTY
};

typedef struct
{
  ModificationAction action;
  GstFilterId *id;
  gint insert_position;
  GstFilterId *replace_id;
} FilterModification;

struct _GstSingleFilterManagerPrivate
{
  GList *applied_filters;
  int list_id;
  GList *filters;
  GstBin *applied_bin;
  GstPad *applied_pad;
  GstPad *out_pad;
  GstPad *blocked_pad;
  GQueue *modifications;
  GMutex *mutex;
  GMutex *modifs_mutex;
  gboolean applying;
};

struct _GstFilterId
{
  GstFilter *filter;
  GstPad *in_pad;
  GstPad *out_pad;
};


static void
gst_single_filter_manager_class_init (GstSingleFilterManagerClass * klass)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS (klass);

  g_type_class_add_private (klass, sizeof (GstSingleFilterManagerPrivate));

  gobject_class->get_property = gst_single_filter_manager_get_property;
  gobject_class->dispose = gst_single_filter_manager_dispose;
  gobject_class->finalize = gst_single_filter_manager_finalize;

  /**
   * GstSingleFilterManager:applied:
   *
   * Whether or not this filter manager has been applied on a pad or not
   */
  g_object_class_install_property (gobject_class, PROP_APPLIED,
      g_param_spec_boolean ("applied", "Applied status",
          "Whether the filter manager has been applied",
          FALSE, G_PARAM_READABLE | G_PARAM_STATIC_STRINGS));

  /**
   * GstSingleFilterManager:applied-bin:
   *
   * The #GstBin this filter manager was applied on or #NULL if not applied yet
   */
  g_object_class_install_property (gobject_class, PROP_APPLIED_BIN,
      g_param_spec_object ("applied-bin", "Applied bin",
          "If applied, the bin it was applied on",
          GST_TYPE_BIN, G_PARAM_READABLE | G_PARAM_STATIC_STRINGS));

  /**
   * GstSingleFilterManager:applied-pad:
   *
   * The #GstPad this filter manager was applied on or #NULL if not applied yet
   */
  g_object_class_install_property (gobject_class, PROP_APPLIED_PAD,
      g_param_spec_object ("applied-pad", "Applied pad",
          "If applied, the pad it was applied on",
          GST_TYPE_PAD, G_PARAM_READABLE | G_PARAM_STATIC_STRINGS));

  /**
   * GstSingleFilterManager:out-pad:
   *
   * The output #GstPad at the end of this filter manager or #NULL if not
   * applied yet.
   *
   * See also: gst_filter_manager_revert()
   */
  g_object_class_install_property (gobject_class, PROP_OUT_PAD,
      g_param_spec_object ("out-pad", "Output pad",
          "If applied, the output pad it applied",
          GST_TYPE_PAD, G_PARAM_READABLE | G_PARAM_STATIC_STRINGS));

  /**
   * GstSingleFilterManager::filter-applied:
   * @filter_manager: The #GstSingleFilterManager
   * @filter_id: The #GstFilterId of the filter that was applied
   *
   * This signal is sent when the filter manager applies a filter
   * and the application was successful.
   */
  signals[SIGNAL_FILTER_APPLIED] = g_signal_new ("filter-applied",
      G_OBJECT_CLASS_TYPE (klass),
      G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
      0,
      NULL, NULL, _gst_filters_marshal_VOID__POINTER,
      G_TYPE_NONE, 1, G_TYPE_POINTER);

  /**
   * GstSingleFilterManager::filter-failed:
   * @filter_manager: The #GstSingleFilterManager
   * @filter_id: The #GstFilterId of the filter that failed to apply
   *
   * This signal is sent when a filter fails to apply in the filter manager
   */
  signals[SIGNAL_FILTER_FAILED] = g_signal_new ("filter-failed",
      G_OBJECT_CLASS_TYPE (klass),
      G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
      0,
      NULL, NULL, _gst_filters_marshal_VOID__POINTER,
      G_TYPE_NONE, 1, G_TYPE_POINTER);

  /**
   * GstSingleFilterManager::filter-reverted:
   * @filter_manager: The #GstSingleFilterManager
   * @filter_id: The #GstFilterId of the filter that was reverted
   *
   * This signal is sent when a filter gets reverted from the filter manager.
   */
  signals[SIGNAL_FILTER_REVERTED] = g_signal_new ("filter-reverted",
      G_OBJECT_CLASS_TYPE (klass),
      G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
      0,
      NULL, NULL, _gst_filters_marshal_VOID__POINTER,
      G_TYPE_NONE, 1, G_TYPE_POINTER);

}

static void
gst_single_filter_manager_interface_init (GstFilterManagerInterface * iface)
{
  iface->list_filters = gst_single_filter_manager_list_filters;
  iface->insert_filter_before = gst_single_filter_manager_insert_filter_before;
  iface->insert_filter_after = gst_single_filter_manager_insert_filter_after;
  iface->replace_filter = gst_single_filter_manager_replace_filter;
  iface->insert_filter = gst_single_filter_manager_insert_filter;
  iface->remove_filter = gst_single_filter_manager_remove_filter;
  iface->get_filter_by_id = gst_single_filter_manager_get_filter_by_id;
  iface->apply = gst_single_filter_manager_apply;
  iface->revert = gst_single_filter_manager_revert;
  iface->handle_message = gst_single_filter_manager_handle_message;

}

static void
gst_single_filter_manager_init (GstSingleFilterManager * self)
{
  GstSingleFilterManagerPrivate *priv =
      G_TYPE_INSTANCE_GET_PRIVATE (self, GST_TYPE_SINGLE_FILTER_MANAGER,
      GstSingleFilterManagerPrivate);

  self->priv = priv;
  priv->modifications = g_queue_new ();
  priv->mutex = g_mutex_new ();
  priv->modifs_mutex = g_mutex_new ();
}


static void
gst_single_filter_manager_dispose (GObject * object)
{
  GstSingleFilterManager *self = GST_SINGLE_FILTER_MANAGER (object);
  GstSingleFilterManagerPrivate *priv = self->priv;

  g_mutex_lock (priv->mutex);

  if (priv->applied_bin) {
    g_critical ("Disposing of GstSingleFilterManager while applied.\n"
        "Make sure to properly revert any filter manager before\n"
        "disposing it. Unexpected behavior to be expected!");
    /* Force the revert to make sure the priv->modifications gets emptied and
     * the pad block is unblocked if it was.. */
    g_mutex_unlock (priv->mutex);
    gst_filter_manager_revert (GST_FILTER_MANAGER (self),
        priv->applied_bin, priv->out_pad);
    g_mutex_lock (priv->mutex);
  }

  g_assert (g_queue_is_empty (priv->modifications));
  g_assert (!priv->blocked_pad);

  if (priv->applied_filters)
    g_list_free (priv->applied_filters);
  priv->applied_filters = NULL;
  priv->list_id = -1;

  if (priv->filters) {
    g_list_foreach (priv->filters, (GFunc) free_filter_id, NULL);
    g_list_free (priv->filters);
  }
  priv->filters = NULL;

  if (priv->applied_bin)
    gst_object_unref (priv->applied_bin);
  if (priv->applied_pad)
    gst_object_unref (priv->applied_pad);
  if (priv->out_pad)
    gst_object_unref (priv->out_pad);

  g_mutex_unlock (priv->mutex);

  G_OBJECT_CLASS (gst_single_filter_manager_parent_class)->dispose (object);
}

static void
gst_single_filter_manager_finalize (GObject * object)
{
  GstSingleFilterManager *self = GST_SINGLE_FILTER_MANAGER (object);
  GstSingleFilterManagerPrivate *priv = self->priv;

  g_queue_free (priv->modifications);
  g_mutex_free (priv->mutex);
  g_mutex_free (priv->modifs_mutex);

  G_OBJECT_CLASS (gst_single_filter_manager_parent_class)->finalize (object);
}

static void
gst_single_filter_manager_get_property (GObject * object,
    guint property_id, GValue * value, GParamSpec * pspec)
{
  GstSingleFilterManager *self = GST_SINGLE_FILTER_MANAGER (object);
  GstSingleFilterManagerPrivate *priv = self->priv;

  g_mutex_lock (priv->mutex);
  switch (property_id) {
    case PROP_APPLIED:
      g_value_set_boolean (value, priv->applied_bin ? TRUE : FALSE);
      break;
    case PROP_APPLIED_BIN:
      g_value_set_object (value, priv->applied_bin);
      break;
    case PROP_APPLIED_PAD:
      g_value_set_object (value, priv->applied_pad);
      break;
    case PROP_OUT_PAD:
      g_value_set_object (value, priv->out_pad);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
      break;
  }
  g_mutex_unlock (priv->mutex);
}

/**
 * gst_single_filter_manager_new:
 *
 * Creates a Single filter manager.
 * The Single filter manager is a filter manager that can only be applied once.
 *
 * Returns: (transfer full): A new #GstFilterManager
 */
GstFilterManager *
gst_single_filter_manager_new (void)
{
  return g_object_new (GST_TYPE_SINGLE_FILTER_MANAGER, NULL);
}

static void
pad_block_do_nothing (GstPad * pad, gboolean blocked, gpointer user_data)
{
}

static void
destroy_pad_block_data (gpointer user_data)
{
  GstSingleFilterManager *self = user_data;
  /* Unref the reference that was held by the pad_block */
  g_object_unref (self);
}

static void
free_filter_id (GstFilterId * id)
{
  g_object_unref (id->filter);
  if (id->in_pad)
    gst_object_unref (id->in_pad);
  if (id->out_pad)
    gst_object_unref (id->out_pad);

  g_slice_free (GstFilterId, id);
}


static void
apply_modifs (GstPad * pad, gboolean blocked, gpointer user_data)
{
  GstSingleFilterManager *self = user_data;
  GstSingleFilterManagerPrivate *priv = self->priv;
  GstBin *applied_bin = NULL;


  g_mutex_lock (priv->mutex);

  /* There could be a race where this thread and a _revert get called at the
   * same time. In which case the _revert will unblock the pad and unref our
   * reference on self. Once it releases the lock, we might get here, in which
   * case we need to check if the pad is blocked, and if it's not, then just
   * return and ignore we were ever called.
   */
  if (!gst_pad_is_blocked (pad)) {
    g_mutex_unlock (priv->mutex);
    return;
  }

  g_mutex_lock (priv->modifs_mutex);

  g_assert (priv->applied_bin);
  applied_bin = gst_object_ref (priv->applied_bin);

  while (!g_queue_is_empty (priv->modifications)) {
    FilterModification *modif = g_queue_pop_head (priv->modifications);
    GstPad *current_pad = NULL;
    GstPad *out_pad = NULL;
    GstPad *srcpad = NULL;
    GstPad *sinkpad = NULL;
    GList *current_pos = NULL;
    GstFilterId *current_id = NULL;
    gboolean remove = FALSE;
    GstFilterId *to_remove = NULL;
    gboolean insert = FALSE;
    gint insert_position = -1;

    /* Set initial values representing the current action */
    if (modif->action == INSERT) {
      current_pos = g_list_nth (priv->applied_filters, modif->insert_position);
      insert = TRUE;
    } else if (modif->action == REMOVE) {
      to_remove = modif->id;
      /* Do not set remove to TRUE because we don't know yet if the filter
         to remove had failed or not */
    } else if (modif->action == REPLACE) {
      current_pos = g_list_find (priv->applied_filters, modif->replace_id);
      to_remove = modif->replace_id;
      insert = TRUE;
      /* Do not set remove to TRUE because we don't know yet if the filter
         to remove had failed or not */
    }

    /* Find the last successfully applied filter after the current one if
       the one at our position had failed to apply */
    for (; current_pos; current_pos = current_pos->next) {
      current_id = current_pos->data;
      if (current_id->in_pad)
        break;
    }

    /* If we want to insert or to replace a failed filter, then find the correct
       pads to unlink/link */
    if (modif->action == INSERT ||
        (modif->action == REPLACE && !to_remove->in_pad)) {
      if (!current_pos) {
        if (GST_PAD_IS_SRC (priv->applied_pad)) {
          current_pad = gst_object_ref (priv->out_pad);
          srcpad = gst_object_ref (current_pad);
          sinkpad = gst_pad_get_peer (srcpad);
        } else {
          current_pad = gst_object_ref (priv->applied_pad);
          sinkpad = gst_object_ref (current_pad);
          srcpad = gst_pad_get_peer (sinkpad);
        }
      } else {
        if (GST_PAD_IS_SRC (priv->applied_pad)) {
          sinkpad = gst_object_ref (current_id->in_pad);
          current_pad = gst_pad_get_peer (sinkpad);
          srcpad = gst_object_ref (current_pad);
        } else {
          current_pad = gst_object_ref (current_id->out_pad);
          sinkpad = gst_object_ref (current_pad);
          srcpad = gst_pad_get_peer (sinkpad);
        }
      }
    } else {
      /* If the action is REMOVE or REPLACE and the filter to remove had been
         applied successfully */
      if (to_remove->in_pad) {
        remove = TRUE;
        current_pad = gst_object_ref (to_remove->out_pad);
        if (GST_PAD_IS_SRC (priv->applied_pad)) {
          srcpad = gst_object_ref (current_pad);
          sinkpad = gst_pad_get_peer (srcpad);
        } else {
          sinkpad = gst_object_ref (current_pad);
          srcpad = gst_pad_get_peer (sinkpad);
        }
      }
    }

    /* We may have nothing to do and no need to unlink if we want to REMOVE a
       filter which had failed to apply previously */
    if (insert || remove) {
      out_pad = NULL;

      g_mutex_unlock (priv->mutex);
      // unlink
      if (gst_pad_unlink (srcpad, sinkpad)) {
        if (remove) {
          out_pad = gst_filter_revert (to_remove->filter, applied_bin,
              current_pad);
          g_signal_emit (self, signals[SIGNAL_FILTER_REVERTED], 0, to_remove);
        } else if (insert) {
          out_pad = gst_filter_apply (modif->id->filter, applied_bin,
              current_pad);
          if (out_pad)
            g_signal_emit (self, signals[SIGNAL_FILTER_APPLIED], 0, modif->id);
          else
            g_signal_emit (self, signals[SIGNAL_FILTER_FAILED], 0, modif->id);
        }
      }
      g_mutex_lock (priv->mutex);

      if (out_pad) {
        /* If we want to replace, we need to apply after the revert */
        if (remove && insert) {
          GstPad *out_pad2 = NULL;

          g_mutex_unlock (priv->mutex);
          out_pad2 = gst_filter_apply (modif->id->filter, applied_bin, out_pad);
          if (out_pad2)
            g_signal_emit (self, signals[SIGNAL_FILTER_APPLIED], 0, modif->id);
          else
            g_signal_emit (self, signals[SIGNAL_FILTER_FAILED], 0, modif->id);
          g_mutex_lock (priv->mutex);

          if (out_pad2) {
            modif->id->in_pad = gst_pad_get_peer (out_pad);
            modif->id->out_pad = gst_object_ref (out_pad2);
            gst_object_unref (out_pad);
            out_pad = out_pad2;
          } else {
            if (modif->id->in_pad)
              gst_object_unref (modif->id->in_pad);
            if (modif->id->out_pad)
              gst_object_unref (modif->id->out_pad);
            modif->id->in_pad = modif->id->out_pad = NULL;
          }
        } else if (insert) {
          modif->id->in_pad = gst_pad_get_peer (current_pad);
          modif->id->out_pad = gst_object_ref (out_pad);
        }

        if (GST_PAD_IS_SRC (priv->applied_pad)) {
          gst_object_unref (srcpad);
          srcpad = gst_object_ref (out_pad);
        } else {
          gst_object_unref (sinkpad);
          sinkpad = gst_object_ref (out_pad);
        }

        /* Update the out pad */
        if (current_pad == priv->out_pad) {
          gst_object_unref (priv->out_pad);
          priv->out_pad = gst_object_ref (out_pad);
        }
        gst_object_unref (out_pad);
      } else if (insert) {
        if (modif->id->in_pad)
          gst_object_unref (modif->id->in_pad);
        if (modif->id->out_pad)
          gst_object_unref (modif->id->out_pad);
        modif->id->in_pad = modif->id->out_pad = NULL;
      }

      gst_object_unref (current_pad);

      // Link
      if (GST_PAD_LINK_SUCCESSFUL (gst_pad_link (srcpad, sinkpad))) {
        gst_object_unref (srcpad);
        gst_object_unref (sinkpad);
      } else {
        /* TODO: what should we do here???? */
      }
    }

    /* Synchronize our applied_filters list with the filters list */
    if (modif->action == REPLACE)
      insert_position = g_list_index (priv->applied_filters, to_remove);
    else if (modif->action == INSERT)
      insert_position = modif->insert_position;

    if (insert_position != -1) {
      priv->applied_filters = g_list_insert (priv->applied_filters,
          modif->id, insert_position);
      priv->list_id++;
    }
    if (to_remove) {
      priv->applied_filters = g_list_remove (priv->applied_filters, to_remove);
      free_filter_id (to_remove);
      priv->list_id++;
    }

    g_slice_free (FilterModification, modif);
  }
  gst_object_unref (applied_bin);
  g_mutex_unlock (priv->modifs_mutex);
  g_mutex_unlock (priv->mutex);

  gst_pad_set_blocked_async (pad, FALSE, pad_block_do_nothing, NULL);
}

static void
block_pad (GstSingleFilterManager * self, gboolean block)
{
  GstSingleFilterManagerPrivate *priv = self->priv;
  GstPad *pad = NULL;

  if (block) {
    if (GST_PAD_IS_SRC (priv->applied_pad))
      pad = gst_object_ref (priv->applied_pad);
    else
      pad = gst_pad_get_peer (priv->out_pad);

    if (pad) {
      /* Keep a reference to self for the pad block thread */
      g_object_ref (self);
      gst_pad_set_blocked_async_full (pad, TRUE, apply_modifs, self,
          destroy_pad_block_data);
      /* Do not ref the pad, we reffed it before */
      priv->blocked_pad = pad;
    }
  } else if (priv->blocked_pad) {
    gst_pad_set_blocked_async (priv->blocked_pad, FALSE,
        pad_block_do_nothing, NULL);
    /* Remove the reference held previously during the block */
    gst_object_unref (priv->blocked_pad);
    priv->blocked_pad = NULL;
  }
}

static void
new_modification (GstSingleFilterManager * self,
    ModificationAction action,
    GstFilterId * id, gint insert_position, GstFilterId * replace_id)
{
  GstSingleFilterManagerPrivate *priv = self->priv;
  FilterModification *modif = g_slice_new0 (FilterModification);

  modif->action = action;
  modif->id = id;
  modif->insert_position = insert_position;
  modif->replace_id = replace_id;

  g_queue_push_tail (priv->modifications, modif);
  if (!priv->applying)
    block_pad (self, TRUE);
}

static GstFilterId *
gst_single_filter_manager_insert_filter_before (GstFilterManager * iface,
    GstFilter * filter, GstFilterId * before)
{
  GstSingleFilterManager *self = GST_SINGLE_FILTER_MANAGER (iface);
  GstSingleFilterManagerPrivate *priv = self->priv;
  gint index = -1;

  g_mutex_lock (priv->mutex);

  index = g_list_index (priv->filters, before);

  if (index < 0) {
    g_mutex_unlock (priv->mutex);
    return NULL;
  }

  return gst_single_filter_manager_insert_filter_unlock (self, filter, index);
}

static GstFilterId *
gst_single_filter_manager_insert_filter_after (GstFilterManager * iface,
    GstFilter * filter, GstFilterId * after)
{
  GstSingleFilterManager *self = GST_SINGLE_FILTER_MANAGER (iface);
  GstSingleFilterManagerPrivate *priv = self->priv;
  gint index = -1;

  g_mutex_lock (priv->mutex);
  index = g_list_index (priv->filters, after);

  if (index < 0) {
    g_mutex_unlock (priv->mutex);
    return NULL;
  }

  return gst_single_filter_manager_insert_filter_unlock (self, filter,
      index + 1);
}

static GstFilterId *
gst_single_filter_manager_replace_filter (GstFilterManager * iface,
    GstFilter * filter, GstFilterId * replace)
{
  GstSingleFilterManager *self = GST_SINGLE_FILTER_MANAGER (iface);
  GstSingleFilterManagerPrivate *priv = self->priv;
  gint index = -1;
  GstFilterId *id = NULL;

  g_mutex_lock (priv->mutex);

  index = g_list_index (priv->filters, replace);

  if (index < 0) {
    g_mutex_unlock (priv->mutex);
    return NULL;
  }

  id = g_slice_new0 (GstFilterId);
  id->filter = g_object_ref (filter);

  priv->filters = g_list_remove (priv->filters, replace);
  priv->filters = g_list_insert (priv->filters, id, index);

  if (priv->applied_bin)
    new_modification (self, REPLACE, id, 0, replace);
  else
    free_filter_id (replace);

  g_mutex_unlock (priv->mutex);

  return id;
}

static GstFilterId *
gst_single_filter_manager_insert_filter_unlock (GstSingleFilterManager * self,
    GstFilter * filter, gint position)
{
  GstSingleFilterManagerPrivate *priv = self->priv;
  GstFilterId *id = g_slice_new0 (GstFilterId);

  if (position < 0 || position > g_list_length (priv->filters))
    position = g_list_length (priv->filters);

  id->filter = g_object_ref (filter);

  priv->filters = g_list_insert (priv->filters, id, position);

  if (priv->applied_bin)
    new_modification (self, INSERT, id, position, NULL);

  g_mutex_unlock (priv->mutex);

  return id;
}

static GstFilterId *
gst_single_filter_manager_insert_filter (GstFilterManager * iface,
    GstFilter * filter, gint position)
{
  GstSingleFilterManager *self = GST_SINGLE_FILTER_MANAGER (iface);
  GstSingleFilterManagerPrivate *priv = self->priv;

  g_mutex_lock (priv->mutex);

  return gst_single_filter_manager_insert_filter_unlock (self, filter,
      position);
}

static GList *
gst_single_filter_manager_list_filters (GstFilterManager * iface)
{
  GstSingleFilterManager *self = GST_SINGLE_FILTER_MANAGER (iface);
  GstSingleFilterManagerPrivate *priv = self->priv;
  GList *ret = NULL;

  g_mutex_lock (priv->mutex);
  ret = g_list_copy (priv->filters);
  g_mutex_unlock (priv->mutex);

  return ret;
}

static gboolean
gst_single_filter_manager_remove_filter (GstFilterManager * iface,
    GstFilterId * id)
{
  GstSingleFilterManager *self = GST_SINGLE_FILTER_MANAGER (iface);
  GstSingleFilterManagerPrivate *priv = self->priv;
  GList *find = NULL;

  g_mutex_lock (priv->mutex);
  find = g_list_find (priv->filters, id);

  if (!find) {
    g_mutex_unlock (priv->mutex);
    return FALSE;
  }

  priv->filters = g_list_remove (priv->filters, id);

  if (priv->applied_bin)
    new_modification (self, REMOVE, id, 0, NULL);
  else
    free_filter_id (id);

  g_mutex_unlock (priv->mutex);

  return TRUE;
}

static GstFilter *
gst_single_filter_manager_get_filter_by_id (GstFilterManager * iface,
    GstFilterId * id)
{
  GstSingleFilterManager *self = GST_SINGLE_FILTER_MANAGER (iface);
  GstSingleFilterManagerPrivate *priv = self->priv;
  GstFilter *result = NULL;

  g_mutex_lock (priv->mutex);
  if (g_list_find (priv->filters, id))
    result = g_object_ref (id->filter);
  g_mutex_unlock (priv->mutex);

  return result;
}


static GstPad *
gst_single_filter_manager_apply (GstFilterManager * iface,
    GstBin * bin, GstPad * pad)
{
  GstSingleFilterManager *self = GST_SINGLE_FILTER_MANAGER (iface);
  GstSingleFilterManagerPrivate *priv = self->priv;
  GList *i = NULL;

  if (!bin || !pad)
    return NULL;

  g_mutex_lock (priv->mutex);

  /* Check applied filters to make sure we're not in a revert */
  if (priv->applied_bin || priv->applied_filters) {
    g_mutex_unlock (priv->mutex);
    return NULL;
  }

  /* Set the applied_bin/applied_pad here so a parallel append will work */
  priv->applied_pad = gst_object_ref (pad);
  priv->applied_bin = gst_object_ref (bin);
  priv->applying = TRUE;

  if (GST_PAD_IS_SRC (pad))
    i = priv->filters;
  else
    i = g_list_last (priv->filters);

  pad = gst_object_ref (pad);
  while (i) {
    GstFilterId *id = i->data;
    GstPad *out_pad = NULL;

    g_mutex_unlock (priv->mutex);
    out_pad = gst_filter_apply (id->filter, bin, pad);
    if (out_pad)
      g_signal_emit (self, signals[SIGNAL_FILTER_APPLIED], 0, id);
    else
      g_signal_emit (self, signals[SIGNAL_FILTER_FAILED], 0, id);
    g_mutex_lock (priv->mutex);


    if (out_pad) {
      id->in_pad = gst_pad_get_peer (pad);
      id->out_pad = gst_object_ref (out_pad);
      gst_object_unref (pad);
      pad = out_pad;
    } else {
      if (id->in_pad)
        gst_object_unref (id->in_pad);
      if (id->out_pad)
        gst_object_unref (id->out_pad);
      id->in_pad = id->out_pad = NULL;
    }

    if (GST_PAD_IS_SRC (pad))
      i = i->next;
    else
      i = i->prev;
  }

  priv->out_pad = gst_object_ref (pad);
  priv->applied_filters = g_list_copy (self->priv->filters);
  priv->list_id++;
  priv->applying = FALSE;

  /* If new modifications were added while we were applying, then block the
     pad so they can be executed */
  if (!g_queue_is_empty (priv->modifications))
    block_pad (self, TRUE);

  g_mutex_unlock (priv->mutex);

  return pad;
}

static GstPad *
gst_single_filter_manager_revert (GstFilterManager * iface,
    GstBin * bin, GstPad * pad)
{
  GstSingleFilterManager *self = GST_SINGLE_FILTER_MANAGER (iface);
  GstSingleFilterManagerPrivate *priv = self->priv;
  GList *i = NULL;


  if (!bin || !pad)
    return NULL;

  g_mutex_lock (priv->mutex);
  if (!priv->applied_bin) {
    g_mutex_unlock (priv->mutex);
    return NULL;
  }

  if (priv->applied_bin != bin || priv->out_pad != pad) {
    g_mutex_unlock (priv->mutex);
    return NULL;
  }

  /* Disable the pad block but don't free removed filter ids */
  if (priv->blocked_pad)
    block_pad (self, FALSE);

  /* Remove applied_bin as it's unnecessary and will prevent concurrent
     additions/removals from triggering the pad_block */
  gst_object_unref (priv->applied_bin);
  priv->applied_bin = NULL;

  /* If we're called at the same time as a pad_block, wait until it finishes
   * doing all of its modifications by releasing the lock for it to continue
   * and waiting for it by requesting the modifs_mutex
   */
  if (!g_mutex_trylock (priv->modifs_mutex)) {
    g_mutex_unlock (priv->mutex);
    g_mutex_lock (priv->modifs_mutex);
    g_mutex_lock (priv->mutex);
  }

  if (GST_PAD_IS_SRC (pad))
    i = g_list_last (priv->applied_filters);
  else
    i = priv->applied_filters;


  pad = gst_object_ref (pad);

  while (i) {
    GstFilterId *id = i->data;
    GstPad *out_pad = NULL;

    if (id->in_pad) {
      g_mutex_unlock (priv->mutex);
      out_pad = gst_filter_revert (id->filter, bin, pad);
      g_signal_emit (self, signals[SIGNAL_FILTER_REVERTED], 0, id);
      g_mutex_lock (priv->mutex);

      gst_object_unref (id->in_pad);
      if (id->out_pad)
        gst_object_unref (id->out_pad);
      id->in_pad = id->out_pad = NULL;

      if (out_pad) {
        gst_object_unref (pad);
        pad = out_pad;
      }
    }

    if (GST_PAD_IS_SRC (pad))
      i = i->prev;
    else
      i = i->next;
  }

  if (priv->applied_pad != pad) {
    g_warning ("Reverting failed, result pad different from applied pad");
    //pad = priv->applied_pad;
  }

  gst_object_unref (priv->applied_pad);
  priv->applied_pad = NULL;
  gst_object_unref (priv->out_pad);
  priv->out_pad = NULL;

  /* Now free the removed filter ids */
  while (!g_queue_is_empty (priv->modifications)) {
    FilterModification *modif = g_queue_pop_head (priv->modifications);

    if (modif->action == REMOVE)
      free_filter_id (modif->id);
    else if (modif->action == REPLACE)
      free_filter_id (modif->replace_id);

    g_slice_free (FilterModification, modif);
  }

  if (priv->applied_filters)
    g_list_free (priv->applied_filters);
  priv->applied_filters = NULL;
  priv->list_id++;

  g_mutex_unlock (priv->modifs_mutex);
  g_mutex_unlock (priv->mutex);

  return pad;
}

static gboolean
gst_single_filter_manager_handle_message (GstFilterManager * iface,
    GstMessage * message)
{
  GstSingleFilterManager *self = GST_SINGLE_FILTER_MANAGER (iface);
  GstSingleFilterManagerPrivate *priv = self->priv;
  GList *i = NULL;
  gboolean drop = FALSE;
  gint list_id = 0;

  g_mutex_lock (priv->mutex);
retry:
  for (i = priv->applied_filters, list_id = priv->list_id;
      i && !drop && list_id == priv->list_id; i = i->next) {
    GstFilterId *id = i->data;

    /* Only handle messages to successfully applied filters */
    if (id->in_pad) {
      g_mutex_unlock (priv->mutex);
      drop = gst_filter_handle_message (id->filter, message);
      g_mutex_lock (priv->mutex);
    }
  }
  if (list_id != priv->list_id)
    goto retry;
  g_mutex_unlock (priv->mutex);

  return drop;
}
